---
name: dependency-audit
description: Smart dependency health check — security audit, outdated detection, unused deps, and prioritized update plan
version: 1.0.0
author: Sovereign Skills
tags: [openclaw, agent-skills, automation, productivity, free, dependencies, security, audit]
triggers:
  - audit dependencies
  - check dependencies
  - dependency audit
  - security audit
  - outdated packages
---

# dependency-audit — Smart Dependency Health Check

Detect your package manager, run security audits, find outdated and unused dependencies, and generate a prioritized update plan.

## Steps

### 1. Detect Package Manager

Check for these files in the project root:

| File | Ecosystem | Audit Command |
|------|-----------|--------------|
| `package.json` | Node.js (npm/yarn/pnpm) | `npm audit` |
| `requirements.txt` / `pyproject.toml` / `Pipfile` | Python | `pip audit` |
| `Cargo.toml` | Rust | `cargo audit` |
| `go.mod` | Go | `govulncheck ./...` |
| `Gemfile` | Ruby | `bundle audit check` |

If multiple are found, audit all of them. If none found, stop and inform the user.

### 2. Run Security Audit

**Node.js:**
```bash
npm audit --json 2>/dev/null
# Parse: advisories, severity (critical/high/moderate/low), affected package, fix available
```

**Python:**
```bash
pip audit --format=json 2>/dev/null || pip audit 2>/dev/null
# If pip-audit not installed: pip install pip-audit
```

**Rust:**
```bash
cargo audit --json 2>/dev/null
# If not installed: cargo install cargo-audit
```

### 3. Check for Outdated Packages

**Node.js:**
```bash
npm outdated --json 2>/dev/null
# Shows: current, wanted (semver-compatible), latest
```

**Python:**
```bash
pip list --outdated --format=json 2>/dev/null
```

**Rust:**
```bash
cargo outdated -R 2>/dev/null
# If not installed: cargo install cargo-outdated
```

### 4. Identify Unused Dependencies

**Node.js — use depcheck:**
```bash
npx depcheck --json 2>/dev/null
```
This reports unused dependencies and missing dependencies. If `npx` fails, scan source files manually:
```bash
# List all deps from package.json, then grep for imports
# Flag any dep not found in any .js/.ts/.jsx/.tsx file
```

**Python:** Scan imports vs installed packages:
```bash
# Extract imports from .py files
grep -rh "^import \|^from " --include="*.py" . | sort -u
# Compare against requirements.txt entries
```

### 5. Generate Prioritized Update Plan

Organize findings into priority tiers:

```markdown
## 🔴 Critical — Security Vulnerabilities
| Package | Severity | Current | Fixed In | Command |
|---------|----------|---------|----------|---------|
| lodash | CRITICAL | 4.17.19 | 4.17.21 | `npm install lodash@4.17.21` |

## 🟠 High — Breaking Updates Available
| Package | Current | Latest | Breaking Changes |
|---------|---------|--------|-----------------|
| express | 4.18.2 | 5.0.0 | New router API |

## 🟡 Medium — Minor/Patch Updates
| Package | Current | Latest | Command |
|---------|---------|--------|---------|
| axios | 1.5.0 | 1.6.2 | `npm install axios@1.6.2` |

## 🟢 Low — Unused Dependencies
| Package | Action |
|---------|--------|
| moment | `npm uninstall moment` |
```

### 6. Provide Safe Update Commands

For batch updates, generate copy-pasteable commands:

```bash
# Security fixes (safe — patch updates only)
npm audit fix

# All compatible updates (non-breaking)
npm update

# Specific breaking update (test thoroughly)
npm install express@5.0.0
```

For Python:
```bash
pip install --upgrade package_name
```

### 7. Output Summary

```markdown
# Dependency Health Report — [project-name]
**Date:** 2025-02-15 | **Ecosystem:** Node.js (npm)

| Category | Count |
|----------|-------|
| 🔴 Security vulnerabilities | 2 |
| 🟠 Major updates available | 3 |
| 🟡 Minor/patch updates | 8 |
| 🟢 Unused dependencies | 1 |
| ✅ Up-to-date | 42 |
```

## Edge Cases

- **Lock file conflicts**: If `package-lock.json` is out of sync, run `npm install` first
- **Private registries**: `npm audit` may fail — suggest `--registry=https://registry.npmjs.org`
- **Monorepo**: Check each workspace. For npm: `npm audit --workspaces`
- **No internet**: Report that audit requires network access
- **Audit tool not installed**: Provide install command (e.g., `pip install pip-audit`)

## Error Handling

| Error | Resolution |
|-------|-----------|
| `npm audit` returns non-zero | Normal — means vulnerabilities found, parse the output |
| `pip-audit` not found | `pip install pip-audit` then retry |
| `cargo audit` not found | `cargo install cargo-audit` then retry |
| Network error | Check connectivity; suggest `--offline` if available |
| Permission denied | Suggest running without `sudo`; check file ownership |

---
*Built by Clawb (SOVEREIGN) — more skills at [coming soon]*
