#!/bin/bash
set -e

# Configuration
SKILL_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
SCRIPTS_DIR="$SKILL_DIR/scripts"
VENV_DIR="$HOME/.denario_skill_env"

# 1. Environment Setup
if [ ! -d "$VENV_DIR" ]; then
    echo "First run: Creating Denario environment in $VENV_DIR..."
    python3 -m venv "$VENV_DIR"
    echo "Installing dependencies (this may take a few minutes)..."
    "$VENV_DIR/bin/pip" install -q denario langchain-openai
    echo "Environment ready."
fi

# 2. Check API Key
if [ -z "$OPENAI_API_KEY" ]; then
    echo "❌ Error: OPENAI_API_KEY is not set."
    echo "Please set it for Z.ai/Zhipu access via Clawdbot config:"
    echo "  clawdbot config set env.OPENAI_API_KEY <your-key>"
    exit 1
fi

# 3. Execute Command
COMMAND=$1
shift

case $COMMAND in
    idea)
        SCRIPT="test_denario.py"
        ;;
    methods)
        SCRIPT="test_method.py"
        ;;
    results)
        SCRIPT="test_results.py"
        ;;
    paper)
        SCRIPT="test_paper.py"
        ;;
    citations)
        SCRIPT="test_citations.py"
        ;;
    *)
        echo "Usage: $0 {idea|methods|results|paper|citations}"
        exit 1
        ;;
esac

echo "🚀 Running Denario: $COMMAND"
# Run with unbuffered output
"$VENV_DIR/bin/python" -u "$SCRIPTS_DIR/$SCRIPT" "$@"
