#!/usr/bin/env python3
import argparse
import os
import uuid
from pathlib import Path

from deepdub import DeepdubClient


def main():
    parser = argparse.ArgumentParser(description="Deepdub TTS -> MEDIA audio file")
    parser.add_argument("--text", required=True, help="Text to synthesize")
    parser.add_argument(
        "--voice-prompt-id",
        default=os.getenv("DEEPDUB_VOICE_PROMPT_ID"),
        help="Deepdub voice prompt ID",
    )
    parser.add_argument(
        "--locale",
        default=os.getenv("DEEPDUB_LOCALE", "en-US"),
        help="Locale (default: en-US)",
    )
    parser.add_argument(
        "--model",
        default=os.getenv("DEEPDUB_MODEL"),
        help="Optional Deepdub model",
    )
    args = parser.parse_args()

    api_key = os.getenv("DEEPDUB_API_KEY")
    if not api_key:
        raise SystemExit("Missing DEEPDUB_API_KEY environment variable. See SKILL.md for the free trial key.")

    if not args.voice_prompt_id:
        raise SystemExit(
            "Missing --voice-prompt-id (or set DEEPDUB_VOICE_PROMPT_ID)"
        )

    # Output directory is controlled only via environment variable (not CLI argument)
    out_dir = Path(os.getenv("OPENCLAW_MEDIA_DIR", "/tmp/openclaw_media"))
    out_dir.mkdir(parents=True, exist_ok=True)

    out_path = out_dir / f"deepdub-{uuid.uuid4().hex}.mp3"

    client = DeepdubClient(api_key=api_key)

    kwargs = {
        "text": args.text,
        "voice_prompt_id": args.voice_prompt_id,
        "locale": args.locale,
    }
    if args.model:
        kwargs["model"] = args.model

    audio_bytes = client.tts(**kwargs)
    out_path.write_bytes(audio_bytes)

    # IMPORTANT: OpenClaw consumes this line
    print(f"MEDIA:{out_path}")


if __name__ == "__main__":
    main()

