#!/usr/bin/env python3
"""
Get Daily Stoic entry for a given date.
Usage: python3 get-stoic.py [MM-DD] [--format text|json|html|telegram]
"""

import json
import sys
import argparse
from datetime import datetime
from pathlib import Path

def get_today():
    """Get today's date, using São Paulo timezone if pytz available."""
    try:
        import pytz
        tz = pytz.timezone('America/Sao_Paulo')
        return datetime.now(tz)
    except ImportError:
        return datetime.now()

def load_data():
    """Load the stoic daily JSON data."""
    script_dir = Path(__file__).parent.parent
    data_file = script_dir / 'assets' / 'stoic-daily.json'
    with open(data_file, 'r') as f:
        return json.load(f)

def parse_entry(entry):
    """Parse the content field into structured data."""
    content = entry['content']
    lines = content.split('\n')
    
    # First line is date, second is title
    date_label = entry['date_label']
    title = lines[1] if len(lines) > 1 else ""
    
    # Find quote and source
    # Quote starts with curly quote " (U+201C) or regular "
    # Source line starts with em dash — (U+2014)
    quote_lines = []
    source = ""
    reflection_start = 2
    in_quote = False
    
    for i, line in enumerate(lines[2:], start=2):
        # Check if line starts with em dash (source attribution)
        if line.startswith('\u2014') or line.startswith('—'):
            source = line.lstrip('\u2014—').strip()
            reflection_start = i + 1
            break
        # Check if this is a quote line (starts with opening curly quote or regular quote)
        elif line.startswith('\u201c') or line.startswith('"') or in_quote:
            in_quote = True
            quote_lines.append(line)
            # Check if quote ends on this line
            if '\u201d' in line or ('"' in line and not line.startswith('"')):
                in_quote = False
    
    # Clean up quote
    quote = ' '.join(quote_lines)
    # Remove leading/trailing quotes
    quote = quote.strip('\u201c\u201d"\'')
    
    # Get reflection (everything after source)
    reflection = '\n'.join(lines[reflection_start:]).strip()
    
    return {
        'date_label': date_label,
        'title': title,
        'quote': quote,
        'source': source,
        'reflection': reflection
    }

def format_text(parsed):
    """Plain text format."""
    return f"""📜 **{parsed['date_label']}**
**{parsed['title']}**

_"{parsed['quote']}"_
—{parsed['source']}

{parsed['reflection']}"""

def format_telegram(parsed):
    """Telegram markdown format."""
    return f"""📜 **{parsed['date_label']}**
**{parsed['title']}**

_"{parsed['quote']}"_
—{parsed['source']}

{parsed['reflection']}"""

def format_json(parsed):
    """JSON format."""
    return json.dumps(parsed, indent=2)

def format_html(parsed):
    """Email-ready HTML format."""
    # Convert paragraph breaks
    reflection_html = '</p><p>'.join(parsed['reflection'].split('\n'))
    reflection_html = f"<p>{reflection_html}</p>"
    
    template = '''<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <style>
        body {{ font-family: Georgia, serif; max-width: 600px; margin: 0 auto; padding: 20px; background: #f5f5f5; }}
        .container {{ background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }}
        .date {{ color: #666; font-size: 14px; text-transform: uppercase; letter-spacing: 1px; }}
        h1 {{ color: #333; font-size: 24px; margin: 10px 0 20px; }}
        .quote {{ font-style: italic; color: #444; border-left: 3px solid #8b0000; padding-left: 15px; margin: 20px 0; }}
        .source {{ color: #666; font-size: 14px; margin-bottom: 20px; }}
        .reflection p {{ line-height: 1.6; color: #333; margin-bottom: 1em; }}
        .footer {{ margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee; color: #999; font-size: 12px; text-align: center; }}
    </style>
</head>
<body>
    <div class="container">
        <div class="date">{date_label}</div>
        <h1>{title}</h1>
        <div class="quote">"{quote}"</div>
        <div class="source">—{source}</div>
        <div class="reflection">{reflection_html}</div>
        <div class="footer">Daily Stoic • Powered by Clawdbot</div>
    </div>
</body>
</html>'''
    return template.format(
        date_label=parsed['date_label'],
        title=parsed['title'],
        quote=parsed['quote'],
        source=parsed['source'],
        reflection_html=reflection_html
    )

def main():
    parser = argparse.ArgumentParser(description='Get Daily Stoic entry')
    parser.add_argument('date', nargs='?', help='Date in MM-DD format (default: today)')
    parser.add_argument('--format', '-f', choices=['text', 'json', 'html', 'telegram'], 
                        default='text', help='Output format')
    args = parser.parse_args()

    # Determine date
    if args.date:
        date_key = args.date
    else:
        date_key = get_today().strftime('%m-%d')

    # Load and find entry
    data = load_data()
    entry = data.get(date_key)
    
    if not entry:
        print(f"No entry found for {date_key}", file=sys.stderr)
        sys.exit(1)

    # Parse and format
    parsed = parse_entry(entry)
    
    formatters = {
        'text': format_text,
        'json': format_json,
        'html': format_html,
        'telegram': format_telegram
    }
    
    print(formatters[args.format](parsed))

if __name__ == '__main__':
    main()
