# -*- coding: utf-8 -*-
"""
每日热榜 Skill - 配置管理
"""

import os
from typing import Optional


class Config:
    """配置管理类"""

    def __init__(self):
        # 每日热榜服务地址
        self.api_url: str = os.getenv(
            "DAILY_HOT_API_URL",
            "http://localhost:6688"
        )

        # 缓存时间（秒）
        self.cache_ttl: int = int(os.getenv("DAILY_HOT_CACHE_TTL", "3600"))

        # 返回最大条数
        self.max_items: int = int(os.getenv("DAILY_HOT_MAX_ITEMS", "20"))

        # 请求超时时间（秒）
        self.timeout: int = int(os.getenv("DAILY_HOT_TIMEOUT", "10"))

        # 启用调试模式
        self.debug: bool = os.getenv("DAILY_HOT_DEBUG", "false").lower() == "true"

        # 数据存储路径
        self.data_dir: str = os.getenv(
            "DAILY_HOT_DATA_DIR",
            "/root/.openclaw/workspace/skills/daily-hot-news/data"
        )

        # 是否自动保存每日热榜
        self.auto_save: bool = os.getenv("DAILY_HOT_AUTO_SAVE", "true").lower() == "true"

    def get_api_url(self, endpoint: str = "") -> str:
        """获取完整的 API 地址"""
        # 移除末尾斜杠
        base_url = self.api_url.rstrip("/")
        if endpoint:
            return f"{base_url}/{endpoint}"
        return base_url

    def is_service_available(self) -> bool:
        """检查服务是否配置"""
        return bool(self.api_url)

    def get_data_path(self, source_id: str = "", date_str: str = "") -> str:
        """获取数据文件路径"""
        import os
        from datetime import datetime

        if not date_str:
            date_str = datetime.now().strftime("%Y-%m-%d")

        data_dir = self.data_dir

        if source_id:
            # 按热榜源分目录存储
            source_dir = os.path.join(data_dir, source_id)
            os.makedirs(source_dir, exist_ok=True)
            return os.path.join(source_dir, f"{date_str}.json")
        else:
            # 主目录
            os.makedirs(data_dir, exist_ok=True)
            return os.path.join(data_dir, f"all_{date_str}.json")


# 全局配置实例
config = Config()
