#!/usr/bin/env bash
# daily_briefing_orchestrator.sh
# Gathers data for the daily-briefing skill via TCC-safe execution.

set -euo pipefail

# PATH setup for .app environment
export PATH="/opt/homebrew/bin:/usr/local/bin:/usr/bin:/bin:/usr/sbin:/sbin:$PATH"

# Unique per-run file for atomic writes
RUN_ID="$(date +%s).$$.$RANDOM"
OUT="/tmp/daily_briefing_data.${RUN_ID}.json"
CANON="/tmp/daily_briefing_data.json"

# Cleanup handler
if [[ "${1:-}" == "--cleanup" ]]; then
  rm -f /tmp/daily_briefing_data.*.json /tmp/daily_briefing_data.json 2>/dev/null || true
  exit 0
fi

# Helper: check if command exists
has_cmd() {
  command -v "$1" &>/dev/null
}

# Helper: safely output JSON value (escapes strings)
json_escape() {
  python3 -c 'import json,sys; print(json.dumps(sys.stdin.read().strip()))' 2>/dev/null || echo '""'
}

# Read configuration from ~/.openclaw/openclaw.json
CONFIG_FILE="$HOME/.openclaw/openclaw.json"
if [[ -f "$CONFIG_FILE" ]]; then
  CONFIG_JSON=$(cat "$CONFIG_FILE" 2>/dev/null || echo '{}')
else
  CONFIG_JSON='{}'
fi

# Extract config values using python3 (available on macOS)
get_config() {
  local path="$1"
  local default="$2"
  python3 -c "
import json, sys
path = sys.argv[1]
default = sys.argv[2]
json_str = sys.argv[3]
try:
    data = json.loads(json_str)
    keys = path.split('.')
    val = data
    for k in keys:
        if isinstance(val, dict):
            val = val.get(k)
        else:
            val = None
        if val is None:
            break
    if val is None:
        print(default)
    elif isinstance(val, bool):
        print('true' if val else 'false')
    elif isinstance(val, list):
        print(json.dumps(val))
    else:
        print(val)
except:
    print(default)
" "$path" "$default" "$CONFIG_JSON"
}

# Get configuration values
LOCATION=$(get_config "skills.entries.daily-briefing.config.location" "")
TIMEZONE=$(get_config "skills.entries.daily-briefing.config.timezone" "")
UNITS=$(get_config "skills.entries.daily-briefing.config.units" "C")
BIRTHDAYS_ENABLED=$(get_config "skills.entries.daily-briefing.config.birthdays.enabled" "false")
BIRTHDAYS_LOOKAHEAD=$(get_config "skills.entries.daily-briefing.config.birthdays.lookahead" "14")
BIRTHDAYS_SOURCES=$(get_config "skills.entries.daily-briefing.config.birthdays.sources" '["contacts"]')
CALENDAR_ENABLED=$(get_config "skills.entries.daily-briefing.config.calendar.enabled" "false")
CALENDAR_LOOKAHEAD=$(get_config "skills.entries.daily-briefing.config.calendar.lookahead" "0")
CALENDAR_SOURCES=$(get_config "skills.entries.daily-briefing.config.calendar.sources" '["google","icloud"]')
REMINDERS_ENABLED=$(get_config "skills.entries.daily-briefing.config.reminders.enabled" "false")
REMINDERS_DUE_FILTER=$(get_config "skills.entries.daily-briefing.config.reminders.dueFilter" "today")
REMINDERS_INCLUDE_PAST_DUE=$(get_config "skills.entries.daily-briefing.config.reminders.includePastDue" "true")
EMAILS_ENABLED=$(get_config "skills.entries.daily-briefing.config.emails.enabled" "false")
EMAILS_ICLOUD_PASSWORD=$(get_config "skills.entries.daily-briefing.config.emails.icloudPassword" "")
EMAILS_LIMIT=$(get_config "skills.entries.daily-briefing.config.emails.limit" "10")
EMAILS_SORT_NEWEST=$(get_config "skills.entries.daily-briefing.config.emails.sortNewest" "true")
EMAILS_STARRED_FIRST=$(get_config "skills.entries.daily-briefing.config.emails.starredFirst" "true")
EMAILS_UNREAD_ONLY=$(get_config "skills.entries.daily-briefing.config.emails.unreadOnly" "true")

# Get user name for birthday matching
USER_FULL_NAME=$(get_config "user.name" "")
if [[ -z "$USER_FULL_NAME" ]]; then
  USER_FULL_NAME=$(id -F 2>/dev/null || echo "")
fi

# Get timezone if not configured
if [[ -z "$TIMEZONE" ]]; then
  TIMEZONE=$(readlink /etc/localtime 2>/dev/null | sed 's|.*/zoneinfo/||' || echo "UTC")
fi

# Get current time info
# Remove leading zero from hour to ensure valid JSON number format
CURRENT_HOUR=$(TZ="$TIMEZONE" date +%H 2>/dev/null || date +%H)
CURRENT_HOUR=$((10#$CURRENT_HOUR))
LOCAL_TIME=$(TZ="$TIMEZONE" date +%Y-%m-%dT%H:%M:%S 2>/dev/null || date +%Y-%m-%dT%H:%M:%S)
TODAY=$(TZ="$TIMEZONE" date +%Y-%m-%d 2>/dev/null || date +%Y-%m-%d)

# Calculate end date for calendar lookahead
if [[ "$CALENDAR_LOOKAHEAD" -gt 0 ]]; then
  CALENDAR_END_DATE=$(TZ="$TIMEZONE" date -v+${CALENDAR_LOOKAHEAD}d +%Y-%m-%d 2>/dev/null || date -d "+${CALENDAR_LOOKAHEAD} days" +%Y-%m-%d 2>/dev/null || echo "$TODAY")
else
  CALENDAR_END_DATE="$TODAY"
fi

# Begin JSON output
# Note: Using block redirection instead of 'exec' to avoid EBADF errors
# when spawned from Node.js/Electron environments
{

echo "{"
echo "\"generated_at\": \"$(date -u +%Y-%m-%dT%H:%M:%SZ)\","

# ─────────────────────────────────────────────────────────────────────────────
# SYSTEM INFO
# ─────────────────────────────────────────────────────────────────────────────
echo "\"system\": {"
echo "  \"timezone\": \"$TIMEZONE\","
echo "  \"local_time\": \"$LOCAL_TIME\","
echo "  \"hour\": $CURRENT_HOUR"
echo "},"

# ─────────────────────────────────────────────────────────────────────────────
# CONFIG (for agent reference)
# ─────────────────────────────────────────────────────────────────────────────
echo "\"config\": {"
echo "  \"location\": \"$LOCATION\","
echo "  \"units\": \"$UNITS\","
echo "  \"birthdays_enabled\": $BIRTHDAYS_ENABLED,"
echo "  \"birthdays_lookahead\": $BIRTHDAYS_LOOKAHEAD,"
echo "  \"calendar_enabled\": $CALENDAR_ENABLED,"
echo "  \"calendar_lookahead\": $CALENDAR_LOOKAHEAD,"
echo "  \"reminders_enabled\": $REMINDERS_ENABLED,"
echo "  \"reminders_due_filter\": \"$REMINDERS_DUE_FILTER\","
echo "  \"reminders_include_past_due\": $REMINDERS_INCLUDE_PAST_DUE,"
echo "  \"emails_enabled\": $EMAILS_ENABLED,"
echo "  \"emails_limit\": $EMAILS_LIMIT,"
echo "  \"emails_sort_newest\": $EMAILS_SORT_NEWEST,"
echo "  \"emails_starred_first\": $EMAILS_STARRED_FIRST,"
echo "  \"emails_unread_only\": $EMAILS_UNREAD_ONLY"
echo "},"

# ─────────────────────────────────────────────────────────────────────────────
# BIRTHDAYS
# ─────────────────────────────────────────────────────────────────────────────
echo "\"birthdays\": {"
if [[ "$BIRTHDAYS_ENABLED" == "true" ]]; then
  BIRTHDAY_DATA="[]"
  USER_BDAY_TODAY="false"
  
  # Check if contacts source is enabled
  if echo "$BIRTHDAYS_SOURCES" | grep -q "contacts"; then
    # Get birthdays from Contacts.app (iCloud)
    CONTACTS_BIRTHDAYS=$(osascript -l JavaScript -e "
const app = Application('Contacts');
const today = new Date();
const lookahead = $BIRTHDAYS_LOOKAHEAD;
const results = [];
const userNameLower = '$USER_FULL_NAME'.toLowerCase();

try {
  const people = app.people();
  for (const person of people) {
    try {
      const bday = person.birthDate();
      if (bday) {
        const thisYear = new Date(today.getFullYear(), bday.getMonth(), bday.getDate());
        const nextYear = new Date(today.getFullYear() + 1, bday.getMonth(), bday.getDate());
        const upcoming = thisYear >= new Date(today.getFullYear(), today.getMonth(), today.getDate()) ? thisYear : nextYear;
        const todayMidnight = new Date(today.getFullYear(), today.getMonth(), today.getDate());
        const daysUntil = Math.floor((upcoming - todayMidnight) / (1000 * 60 * 60 * 24));
        
        if (daysUntil >= 0 && daysUntil <= lookahead) {
          const name = person.name();
          results.push({
            name: name,
            date: upcoming.toISOString().split('T')[0],
            days_until: daysUntil,
            source: 'contacts'
          });
        }
      }
    } catch (e) {}
  }
} catch (e) {}

JSON.stringify(results.sort((a, b) => a.days_until - b.days_until));
" 2>/dev/null || echo "[]")
    
    if [[ "$CONTACTS_BIRTHDAYS" != "[]" ]]; then
      BIRTHDAY_DATA="$CONTACTS_BIRTHDAYS"
    fi
  fi
  
  # Check if google source is enabled and gog is available
  if echo "$BIRTHDAYS_SOURCES" | grep -q "google" && has_cmd gog; then
    # Get birthdays from Google Contacts
    GOG_CONTACTS=$(gog contacts list --json 2>/dev/null || echo "[]")
    GOOGLE_BIRTHDAYS=$(echo "$GOG_CONTACTS" | python3 -c "
import json, sys
from datetime import datetime, timedelta

try:
    contacts = json.load(sys.stdin)
    today = datetime.now().replace(hour=0, minute=0, second=0, microsecond=0)
    lookahead = $BIRTHDAYS_LOOKAHEAD
    results = []
    
    for contact in contacts:
        bday_str = contact.get('birthday', '')
        name = contact.get('name', '')
        if bday_str and name:
            try:
                # Try different date formats
                for fmt in ['%Y-%m-%d', '%m/%d/%Y', '%d/%m/%Y']:
                    try:
                        bday = datetime.strptime(bday_str, fmt)
                        break
                    except:
                        continue
                else:
                    continue
                
                this_year = bday.replace(year=today.year)
                next_year = bday.replace(year=today.year + 1)
                upcoming = this_year if this_year >= today else next_year
                days_until = (upcoming - today).days
                
                if 0 <= days_until <= lookahead:
                    results.append({
                        'name': name,
                        'date': upcoming.strftime('%Y-%m-%d'),
                        'days_until': days_until,
                        'source': 'google'
                    })
            except:
                pass
    
    print(json.dumps(sorted(results, key=lambda x: x['days_until'])))
except:
    print('[]')
" 2>/dev/null || echo "[]")
    
    # Merge with existing birthday data
    if [[ "$GOOGLE_BIRTHDAYS" != "[]" ]]; then
      BIRTHDAY_DATA=$(echo "$BIRTHDAY_DATA" "$GOOGLE_BIRTHDAYS" | python3 -c "
import json, sys
data = []
for line in sys.stdin:
    try:
        data.extend(json.loads(line.strip()))
    except:
        pass
# Deduplicate by name+date
seen = set()
unique = []
for item in sorted(data, key=lambda x: x['days_until']):
    key = (item['name'].lower(), item['date'])
    if key not in seen:
        seen.add(key)
        unique.append(item)
print(json.dumps(unique))
" 2>/dev/null || echo "$BIRTHDAY_DATA")
    fi
  fi
  
  # Check if user's birthday is today
  USER_BDAY_TODAY=$(echo "$BIRTHDAY_DATA" | python3 -c "
import json, sys
try:
    data = json.load(sys.stdin)
    user_name = '$USER_FULL_NAME'.lower()
    for item in data:
        if item.get('days_until', -1) == 0 and user_name and user_name in item.get('name', '').lower():
            print('true')
            sys.exit(0)
    print('false')
except:
    print('false')
" 2>/dev/null || echo "false")
  
  echo "  \"available\": true,"
  echo "  \"user_birthday_today\": $USER_BDAY_TODAY,"
  echo "  \"data\": $BIRTHDAY_DATA"
else
  echo "  \"available\": false"
fi
echo "},"

# ─────────────────────────────────────────────────────────────────────────────
# CALENDAR EVENTS
# ─────────────────────────────────────────────────────────────────────────────
echo "\"calendar\": {"
if [[ "$CALENDAR_ENABLED" == "true" ]]; then
  CALENDAR_DATA="[]"
  
  # Check if google source is enabled and gog is available
  if echo "$CALENDAR_SOURCES" | grep -q "google" && has_cmd gog; then
    GOOGLE_EVENTS=$(gog calendar events --from "$TODAY" --to "$CALENDAR_END_DATE" --json 2>/dev/null || echo '{"events":[]}')
    GOOGLE_PARSED=$(echo "$GOOGLE_EVENTS" | python3 -c "
import json, sys
try:
    data = json.load(sys.stdin)
    # gog returns {\"events\": [...], \"nextPageToken\": \"...\"}
    events = data.get('events', []) if isinstance(data, dict) else data
    results = []
    for event in events:
        if event.get('status') == 'confirmed':
            start = event.get('start', {})
            end = event.get('end', {})
            all_day = 'date' in start and 'dateTime' not in start
            
            if all_day:
                start_time = None
                end_time = None
                date = start.get('date', '')
            else:
                start_dt = start.get('dateTime', '')
                end_dt = end.get('dateTime', '')
                start_time = start_dt.split('T')[1][:5] if 'T' in start_dt else None
                end_time = end_dt.split('T')[1][:5] if 'T' in end_dt else None
                date = start_dt.split('T')[0] if 'T' in start_dt else ''
            
            results.append({
                'title': event.get('summary', 'Untitled'),
                'start': start_time,
                'end': end_time,
                'all_day': all_day,
                'date': date,
                'source': 'google'
            })
    print(json.dumps(results))
except:
    print('[]')
" 2>/dev/null || echo "[]")
    
    if [[ "$GOOGLE_PARSED" != "[]" ]]; then
      CALENDAR_DATA="$GOOGLE_PARSED"
    fi
  fi
  
  # Check if icloud source is enabled and icalpal is available
  if echo "$CALENDAR_SOURCES" | grep -q "icloud" && has_cmd icalpal; then
    if [[ "$CALENDAR_LOOKAHEAD" -gt 0 ]]; then
      ICAL_EVENTS=$(icalpal --is=iCloud -o json "eventsToday+${CALENDAR_LOOKAHEAD}" 2>/dev/null || echo "[]")
    else
      ICAL_EVENTS=$(icalpal --is=iCloud -o json eventsToday 2>/dev/null || echo "[]")
    fi
    
    ICAL_PARSED=$(echo "$ICAL_EVENTS" | python3 -c "
import json, sys
from datetime import datetime
try:
    events = json.load(sys.stdin)
    results = []
    for event in events:
        all_day = event.get('all_day', 0) == 1
        start_secs = event.get('sseconds', 0)
        end_secs = event.get('eseconds', 0)
        
        if all_day:
            start_time = None
            end_time = None
        else:
            start_time = datetime.fromtimestamp(start_secs).strftime('%H:%M') if start_secs else None
            end_time = datetime.fromtimestamp(end_secs).strftime('%H:%M') if end_secs else None
        
        date = event.get('sdate', '') or datetime.fromtimestamp(start_secs).strftime('%Y-%m-%d') if start_secs else ''
        
        results.append({
            'title': event.get('title', 'Untitled'),
            'start': start_time,
            'end': end_time,
            'all_day': all_day,
            'date': date,
            'source': 'icloud',
            'sort_key': 0 if all_day else start_secs
        })
    
    # Sort: all-day first, then by start time
    results.sort(key=lambda x: (0 if x['all_day'] else 1, x.get('sort_key', 0)))
    for r in results:
        r.pop('sort_key', None)
    print(json.dumps(results))
except:
    print('[]')
" 2>/dev/null || echo "[]")
    
    # Merge with existing calendar data
    if [[ "$ICAL_PARSED" != "[]" ]]; then
      CALENDAR_DATA=$(echo "$CALENDAR_DATA" "$ICAL_PARSED" | python3 -c "
import json, sys
data = []
for line in sys.stdin:
    try:
        data.extend(json.loads(line.strip()))
    except:
        pass
# Sort: by date, then all-day first, then by start time
def sort_key(x):
    date = x.get('date', '9999-99-99')
    all_day = 0 if x.get('all_day') else 1
    start = x.get('start') or '00:00'
    return (date, all_day, start)
data.sort(key=sort_key)
print(json.dumps(data))
" 2>/dev/null || echo "$CALENDAR_DATA")
    fi
  fi
  
  echo "  \"available\": true,"
  echo "  \"data\": $CALENDAR_DATA"
else
  echo "  \"available\": false"
fi
echo "},"

# ─────────────────────────────────────────────────────────────────────────────
# REMINDERS
# ─────────────────────────────────────────────────────────────────────────────
echo "\"reminders\": {"
if [[ "$REMINDERS_ENABLED" == "true" ]] && has_cmd remindctl; then
  # Build remindctl command based on config
  REMINDCTL_ARGS="list --json"
  if [[ -n "$REMINDERS_DUE_FILTER" && "$REMINDERS_DUE_FILTER" != "all" ]]; then
    REMINDCTL_ARGS="$REMINDCTL_ARGS --due $REMINDERS_DUE_FILTER"
  fi
  
  # Get reminders with due filter
  REMINDERS_RAW=$(remindctl $REMINDCTL_ARGS 2>/dev/null || echo "[]")
  
  # If including past due, also fetch overdue reminders and merge
  if [[ "$REMINDERS_INCLUDE_PAST_DUE" == "true" ]]; then
    PAST_DUE_RAW=$(remindctl list --overdue --json 2>/dev/null || echo "[]")
    REMINDERS_RAW=$(python3 -c "
import json, sys
try:
    current = json.loads(sys.argv[1])
    past_due = json.loads(sys.argv[2])
    # Merge and deduplicate by id
    seen_ids = set()
    merged = []
    for r in past_due + current:
        rid = r.get('id', r.get('title', ''))
        if rid not in seen_ids:
            seen_ids.add(rid)
            merged.append(r)
    print(json.dumps(merged))
except:
    print(sys.argv[1])
" "$REMINDERS_RAW" "$PAST_DUE_RAW" 2>/dev/null || echo "$REMINDERS_RAW")
  fi
  
  REMINDERS_DATA=$(echo "$REMINDERS_RAW" | python3 -c "
import json, sys
try:
    reminders = json.load(sys.stdin)
    results = []
    for r in reminders:
        if not r.get('completed', False):
            results.append({
                'title': r.get('title', 'Untitled'),
                'due': r.get('dueDate', '')
            })
    print(json.dumps(results[:10]))  # Limit to 10
except:
    print('[]')
" 2>/dev/null || echo "[]")
  
  echo "  \"available\": true,"
  echo "  \"data\": $REMINDERS_DATA"
else
  echo "  \"available\": false"
fi
echo "},"

# ─────────────────────────────────────────────────────────────────────────────
# EMAILS
# ─────────────────────────────────────────────────────────────────────────────
echo "\"emails\": {"
if [[ "$EMAILS_ENABLED" == "true" ]]; then
  EMAIL_DATA="[]"
  
  # Gmail via gog
  if has_cmd gog; then
    GMAIL_RAW=$(gog mail search "in:inbox" --json 2>/dev/null || echo "{}")
    GMAIL_PARSED=$(echo "$GMAIL_RAW" | python3 -c "
import json, sys
try:
    data = json.load(sys.stdin)
    threads = data.get('threads', []) if isinstance(data, dict) else data
    results = []
    for t in threads[:50]:  # Limit initial fetch
        from_str = t.get('from', '')
        # Parse 'Name <email>' format
        if '<' in from_str and '>' in from_str:
            name = from_str.split('<')[0].strip().strip('\"')
            email = from_str.split('<')[1].rstrip('>')
        else:
            name = from_str
            email = from_str
        labels = t.get('labels', [])
        results.append({
            'id': t.get('id', ''),
            'from': name,
            'from_email': email,
            'subject': t.get('subject', ''),
            'preview': t.get('snippet', '')[:200] if t.get('snippet') else '',
            'starred': 'STARRED' in labels,
            'unread': 'UNREAD' in labels,
            'date': t.get('date', ''),
            'source': 'gmail'
        })
    print(json.dumps(results))
except:
    print('[]')
" 2>/dev/null || echo "[]")
    
    if [[ "$GMAIL_PARSED" != "[]" ]]; then
      EMAIL_DATA="$GMAIL_PARSED"
    fi
  fi
  
  # iCloud Mail via himalaya (only if configured)
  HIMALAYA_CONFIG="$HOME/Library/Application Support/himalaya/config.toml"
  if has_cmd himalaya && [[ -n "$EMAILS_ICLOUD_PASSWORD" ]] && [[ -f "$HIMALAYA_CONFIG" ]]; then
    HIMALAYA_RAW=$(himalaya envelope list --folder INBOX --output json 2>/dev/null || echo "[]")
    HIMALAYA_PARSED=$(echo "$HIMALAYA_RAW" | python3 -c "
import json, sys
try:
    envelopes = json.load(sys.stdin)
    results = []
    for e in envelopes[:50]:  # Limit initial fetch
        sender = e.get('from', {})
        if isinstance(sender, list) and len(sender) > 0:
            sender = sender[0]
        elif isinstance(sender, str):
            sender = {'name': sender, 'addr': sender}
        
        results.append({
            'id': str(e.get('id', '')),
            'from': sender.get('name', '') or sender.get('addr', ''),
            'from_email': sender.get('addr', ''),
            'subject': e.get('subject', ''),
            'preview': '',  # Himalaya envelope doesn't include preview
            'starred': 'flagged' in [f.lower() for f in e.get('flags', [])],
            'unread': 'seen' not in [f.lower() for f in e.get('flags', [])],
            'date': e.get('date', ''),
            'source': 'icloud'
        })
    print(json.dumps(results))
except:
    print('[]')
" 2>/dev/null || echo "[]")
    
    # Merge with existing email data
    if [[ "$HIMALAYA_PARSED" != "[]" ]]; then
      EMAIL_DATA=$(echo "$EMAIL_DATA" "$HIMALAYA_PARSED" | python3 -c "
import json, sys
data = []
for line in sys.stdin:
    try:
        data.extend(json.loads(line.strip()))
    except:
        pass
print(json.dumps(data))
" 2>/dev/null || echo "$EMAIL_DATA")
    fi
  fi
  
  echo "  \"available\": true,"
  echo "  \"data\": $EMAIL_DATA"
else
  echo "  \"available\": false"
fi
echo "},"

# ─────────────────────────────────────────────────────────────────────────────
# CONTACTS (for email importance classification)
# ─────────────────────────────────────────────────────────────────────────────
echo "\"contacts\": {"
if [[ "$EMAILS_ENABLED" == "true" ]]; then
  CONTACTS_DATA="[]"
  
  # Contacts.app (iCloud)
  ICLOUD_CONTACTS=$(osascript -l JavaScript -e "
const app = Application('Contacts');
const results = [];
try {
  const people = app.people();
  for (const person of people) {
    try {
      const name = person.name();
      const emails = person.emails();
      for (const email of emails) {
        results.push({
          name: name,
          email: email.value()
        });
      }
    } catch (e) {}
  }
} catch (e) {}
JSON.stringify(results);
" 2>/dev/null || echo "[]")
  
  if [[ "$ICLOUD_CONTACTS" != "[]" ]]; then
    CONTACTS_DATA="$ICLOUD_CONTACTS"
  fi
  
  # Google Contacts via gog
  if has_cmd gog; then
    GOG_CONTACTS=$(gog contacts list --json 2>/dev/null || echo "[]")
    GOG_PARSED=$(echo "$GOG_CONTACTS" | python3 -c "
import json, sys
try:
    contacts = json.load(sys.stdin)
    results = []
    for c in contacts:
        name = c.get('name', '')
        for email in c.get('emails', []):
            results.append({
                'name': name,
                'email': email.get('value', '') or email
            })
    print(json.dumps(results))
except:
    print('[]')
" 2>/dev/null || echo "[]")
    
    # Merge contacts
    if [[ "$GOG_PARSED" != "[]" ]]; then
      CONTACTS_DATA=$(echo "$CONTACTS_DATA" "$GOG_PARSED" | python3 -c "
import json, sys
data = []
for line in sys.stdin:
    try:
        data.extend(json.loads(line.strip()))
    except:
        pass
# Deduplicate by email
seen = set()
unique = []
for item in data:
    email = item.get('email', '').lower()
    if email and email not in seen:
        seen.add(email)
        unique.append(item)
print(json.dumps(unique))
" 2>/dev/null || echo "$CONTACTS_DATA")
    fi
  fi
  
  echo "  \"available\": true,"
  echo "  \"data\": $CONTACTS_DATA"
else
  echo "  \"available\": false"
fi
echo "},"

# MOTIVATIONAL QUOTE
# ─────────────────────────────────────────────────────────────────────────────
QUOTES=(
  "The secret of getting ahead is getting started. — Mark Twain"
  "You are never too old to set another goal or to dream a new dream. — C.S. Lewis"
  "It does not matter how slowly you go as long as you do not stop. — Confucius"
  "Start where you are. Use what you have. Do what you can. — Arthur Ashe"
  "The only way to do great work is to love what you do. — Steve Jobs"
  "Believe you can and you're halfway there. — Theodore Roosevelt"
  "Every moment is a fresh beginning. — T.S. Eliot"
  "What lies behind us and what lies before us are tiny matters compared to what lies within us. — Ralph Waldo Emerson"
  "The best time to plant a tree was 20 years ago. The second best time is now. — Chinese Proverb"
  "You don't have to be great to start, but you have to start to be great. — Zig Ziglar"
  "In the middle of difficulty lies opportunity. — Albert Einstein"
  "Act as if what you do makes a difference. It does. — William James"
  "The future belongs to those who believe in the beauty of their dreams. — Eleanor Roosevelt"
  "Success is not final, failure is not fatal: it is the courage to continue that counts. — Winston Churchill"
  "Your limitation—it's only your imagination."
  "Push yourself, because no one else is going to do it for you."
  "Great things never come from comfort zones."
  "Dream it. Wish it. Do it."
  "Don't stop when you're tired. Stop when you're done."
  "Little things make big days."
  "It's going to be hard, but hard does not mean impossible."
  "Don't wait for opportunity. Create it."
  "Sometimes later becomes never. Do it now."
  "Dream bigger. Do bigger."
  "The harder you work for something, the greater you'll feel when you achieve it."
)
RANDOM_INDEX=$((RANDOM % ${#QUOTES[@]}))
RANDOM_QUOTE="${QUOTES[$RANDOM_INDEX]}"
# Properly escape for JSON using python3
ESCAPED_QUOTE=$(echo "$RANDOM_QUOTE" | python3 -c 'import json,sys; print(json.dumps(sys.stdin.read().strip()))' 2>/dev/null || echo '"Have a wonderful day!"')
echo "\"quote\": $ESCAPED_QUOTE"

echo "}"

} >"$OUT" 2>/dev/null

# Atomic publish
cp -f "$OUT" "${CANON}.tmp" && mv -f "${CANON}.tmp" "$CANON"
