#!/usr/bin/env bash
# install.sh - Installs the daily-briefing skill

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BIN_DIR="${SCRIPT_DIR}/bin"
ORCHESTRATOR="${SCRIPT_DIR}/scripts/daily_briefing_orchestrator.sh"
RUNNER_SCRIPT="${BIN_DIR}/run_daily_briefing.sh"

echo "📦 Installing daily-briefing skill..."

# Create bin directory
mkdir -p "$BIN_DIR"

# Remove old runner script if exists (from previous install approach)
rm -rf "$RUNNER_SCRIPT" 2>/dev/null || true

# Make orchestrator executable
chmod +x "$ORCHESTRATOR"

# Create runner script that simply invokes the orchestrator
cat > "$RUNNER_SCRIPT" << 'RUNNER_EOF'
#!/bin/bash
# run_daily_briefing.sh - Runs the daily briefing data gatherer
# TCC permissions are granted to Terminal.app (or the calling process)

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
ORCHESTRATOR="${SCRIPT_DIR}/../scripts/daily_briefing_orchestrator.sh"

if [[ ! -f "$ORCHESTRATOR" ]]; then
  echo "Error: Orchestrator not found at $ORCHESTRATOR" >&2
  exit 1
fi

# Pass any arguments (e.g., --cleanup)
bash -l "$ORCHESTRATOR" "$@"
RUNNER_EOF

chmod +x "$RUNNER_SCRIPT"

echo "✅ Runner script created at: ${RUNNER_SCRIPT}"
echo ""
echo "📋 Next steps:"
echo ""
echo "   1. Grant Terminal.app access to Contacts, Calendar, and Reminders:"
echo "      System Settings > Privacy & Security > [Contacts/Calendars/Reminders]"
echo "      Add: Terminal (or your terminal app)"
echo ""
echo "   2. Test the skill:"
echo "      ${RUNNER_SCRIPT}"
echo "      cat /tmp/daily_briefing_data.json"
echo ""
echo "   3. Install optional CLI tools for enhanced data:"
echo "      brew install steipete/tap/gogcli      # Google services"
echo "      brew install ajrosen/tap/icalpal      # iCloud Calendar"
echo "      brew install steipete/tap/remindctl   # Apple Reminders"
echo "      brew install himalaya                 # iCloud Mail"
echo ""
echo "   4. Configure the skill in ~/.openclaw/openclaw.json"
echo ""
echo "🎉 Installation complete!"
