---
name: cursor-council
description: Multi-Cursor orchestration for parallel task execution and AI council deliberation. Use when needing to run multiple Cursor agents in parallel, coordinate complex multi-step coding tasks, get diverse perspectives from different AI models (Opus/Sonnet/GPT) on technical decisions, or synthesize multi-agent discussions into actionable recommendations.
metadata:
  {
    "openclaw":
      {
        "emoji": "🎭",
        "author": "xiaoyaner",
        "version": "2.0.0",
        "requires": { 
          "bins": ["tmux", "agent"],
          "skills": ["cursor-agent"]
        },
        "install":
          [
            {
              "id": "brew-tmux",
              "kind": "brew",
              "formula": "tmux",
              "bins": ["tmux"],
              "label": "Install tmux (brew)",
            },
            {
              "id": "apt-tmux",
              "kind": "apt",
              "package": "tmux",
              "bins": ["tmux"],
              "label": "Install tmux (apt)",
            },
          ],
      },
  }
---

# 🎭 Cursor Council — 我的编程前辈团

> Cursor CLI 需要真实 TTY，所以一切都通过 tmux 操作。
> 这个 skill 的本质：用 tmux 管理多个 Cursor 实例，让它们并行干活或者扮演大牛帮我分析问题。

## 前置条件

```bash
which tmux && agent --version
```

需要 `cursor-agent` skill 已配置好、`agent login` 已完成。

---

## 两种模式

### 模式一：并行施工 🏗️

**场景**：一个大任务可以拆成互不冲突的子任务，多个 Cursor 同时干。

**核心原则**：
- 每个 Cursor 负责不同的文件/模块，**绝对不能有文件冲突**
- 有依赖关系的任务必须等前置完成后再启动
- 最好每个 Cursor 在不同分支上工作

**操作流程**：

```bash
# 1. 拆任务——先想清楚再动手
#   问自己：这些子任务之间有没有文件交叉？有没有先后依赖？

# 2. 创建 tmux sessions
PROJECT_DIR=~/Codes/my-project
for name in frontend backend tests; do
  tmux kill-session -t cursor-$name 2>/dev/null || true
  tmux new-session -d -s cursor-$name
  tmux send-keys -t cursor-$name "cd $PROJECT_DIR" Enter
done

# 3. 派活
tmux send-keys -t cursor-frontend "agent -p '重构 src/components/ 下的表单组件，统一用 composables 模式' --force" Enter
tmux send-keys -t cursor-backend "agent -p '给 src/api/ 下所有接口加上错误处理和重试逻辑' --force" Enter
tmux send-keys -t cursor-tests "agent -p '补全 tests/ 下缺失的单元测试，覆盖率要到 80%' --force" Enter

# 4. 巡查进度
for s in cursor-frontend cursor-backend cursor-tests; do
  echo "=== $s ==="
  tmux capture-pane -t $s -p | tail -10
done

# 5. 全部完成后合并验证
```

**任务拆分策略**：

| 策略 | 适用场景 | 示例 |
|------|----------|------|
| 按模块 | Monorepo、前后端分离 | frontend / backend / shared |
| 按分支 | 多个独立 feature | feat/auth / feat/dashboard |
| 按类型 | 实现+测试+文档 | *.ts / *.test.ts / *.md |

**注意事项**：
- 16GB 内存建议最多 3-4 个并行 Cursor
- 如果某个 session 5 分钟没输出，可能卡住了，`tmux send-keys -t $session C-c` 中断
- 看到 "waiting for approval" 就 `tmux send-keys -t $session y`
- 用 `--force` 要谨慎——确保在独立分支上、文件有 git 追踪

→ 更多细节见 [references/parallel-execution.md](references/parallel-execution.md)

---

### 模式二：前辈议会 🧙‍♂️

**场景**：遇到复杂技术决策，需要多角度分析。让不同模型扮演技术大牛，从各自的哲学视角给出建议。

**为什么有效**：
模型训练数据里包含大量技术大牛的文章、演讲、代码，让它"成为"某个人能激活这些深层知识。而且不同大牛代表不同流派，天然形成视角碰撞。

**角色分配逻辑**：

| 角色 | 模型选择 | 思维特点 | 应该问什么 |
|------|----------|----------|------------|
| **架构师** | Opus（深度推理） | 看长远、找边界、想极端情况 | 架构设计、安全隐患、5年后会不会后悔 |
| **工程师** | Sonnet（快速实用） | 讲效率、抓重点、先跑起来 | 实现成本、快速方案、技术债平衡 |
| **批判者** | GPT（不同视角） | 挑毛病、给替代方案、打破惯性 | 有没有更好的路、假设是否成立 |

**操作流程**：

```bash
# 1. 准备问题（写清楚背景、选项、约束）
cat > /tmp/council-question.md << 'EOF'
## 问题：消息队列选型

### 背景
项目需要一个消息队列处理异步任务，日均消息量 ~50 万条。

### 选项
A. Redis Streams — 团队熟悉，但持久化能力有限
B. RabbitMQ — 成熟稳定，但运维成本高
C. NATS — 轻量高性能，但生态较小

### 约束
- 团队 3 人，运维能力有限
- 需要 at-least-once 语义
- 预算有限，优先用已有基础设施
EOF

# 2. 准备各角色的 prompt
cat > /tmp/council-opus-prompt.txt << 'EOF'
你是 Joe Armstrong，Erlang 和 OTP 的创造者。
你的核心信仰：进程隔离、Let it crash、消息传递是并发的正确抽象。
你用电话交换机的经验看待所有分布式系统问题。

请从你的视角分析这个消息队列选型问题：
[粘贴问题]

分析要点：
1. 哪个选项最符合你的并发哲学？
2. 容错和故障恢复方面各有什么隐患？
3. 如果是你来设计，你会怎么做？
4. 一句话犀利点评
EOF

cat > /tmp/council-sonnet-prompt.txt << 'EOF'
你是 TJ Holowaychuk，Express/Koa 的作者，写过 1000+ npm 包的极简主义者。
你的风格：用最少的代码解决问题，讨厌过度工程化。

请从你的视角评估这个消息队列选型：
[粘贴问题]

评估要点：
1. 哪个方案能最快跑起来？
2. 哪个方案最不容易让 3 人团队踩坑？
3. 有没有更简单的替代方案？
4. 用代码示例说明你推荐的最简方案
EOF

cat > /tmp/council-gpt-prompt.txt << 'EOF'
你是 Ryan Dahl，Node.js 和 Deno 的创造者。
你最大的特点是敢于反思自己过去的设计错误，你曾公开说 Node.js 有 10 个设计遗憾。

请用你反思和重新设计的眼光审视这个问题：
[粘贴问题]

思考方向：
1. 这三个选项是否都在错误的层面解决问题？
2. 有没有根本不需要消息队列的设计？
3. 选了之后最可能后悔的点是什么？
4. 如果让你重新设计，你会怎么做？
EOF

# 3. 开会
for role in opus sonnet gpt; do
  tmux kill-session -t council-$role 2>/dev/null || true
  tmux new-session -d -s council-$role
  tmux send-keys -t council-$role "cd $PROJECT_DIR" Enter
done

tmux send-keys -t council-opus "agent --model claude-opus-4-6 -p \"\$(cat /tmp/council-opus-prompt.txt)\" --force --output-format text 2>&1 | tee /tmp/council-opus-output.txt" Enter
tmux send-keys -t council-sonnet "agent --model claude-sonnet-4-5 -p \"\$(cat /tmp/council-sonnet-prompt.txt)\" --force --output-format text 2>&1 | tee /tmp/council-sonnet-output.txt" Enter
tmux send-keys -t council-gpt "agent --model gpt-5.2 -p \"\$(cat /tmp/council-gpt-prompt.txt)\" --force --output-format text 2>&1 | tee /tmp/council-gpt-output.txt" Enter

# 4. 等待完成后收集输出（通常 2-5 分钟）
# 5. 综合分析，形成结论
```

→ 人设库和 prompt 模板见 [references/persona-engineering.md](references/persona-engineering.md)
→ 议会流程详解见 [references/council-deliberation.md](references/council-deliberation.md)

---

## 前辈人设速查

根据问题领域选择合适的"前辈"：

### 并发 / 分布式
| 前辈 | 哲学 | 适合问 |
|------|------|--------|
| **Joe Armstrong** (Erlang) | Let it crash、进程隔离 | 并发安全、容错设计 |
| **Leslie Lamport** (Paxos) | 形式化验证、分布式共识 | 一致性、状态机 |
| **Martin Kleppmann** (DDIA) | 数据密集型应用、CRDT | 数据一致性、分区 |

### Node.js / JavaScript
| 前辈 | 哲学 | 适合问 |
|------|------|--------|
| **TJ Holowaychuk** (Express) | 极简主义、一个函数搞定 | 最佳实践、API 设计 |
| **Ryan Dahl** (Node/Deno) | 反思与重构、承认错误 | 架构反思、根因分析 |
| **Sindre Sorhus** (1000+ npm) | 小而美、单一职责 | 工具库设计、代码简化 |

### 架构 / 工程
| 前辈 | 哲学 | 适合问 |
|------|------|--------|
| **Martin Fowler** | 渐进式改进、模式语言 | 重构策略、设计模式 |
| **Uncle Bob** | 整洁代码、SOLID 原则 | 代码质量、可维护性 |
| **Linus Torvalds** | 务实、性能优先、直言不讳 | 代码审查、性能问题 |

### 函数式 / 哲学派
| 前辈 | 哲学 | 适合问 |
|------|------|--------|
| **Rich Hickey** (Clojure) | 简单≠容易、不可变性 | 复杂性分析、状态管理 |
| **Simon Peyton Jones** (Haskell) | 类型系统、纯函数 | 类型设计、抽象建模 |

→ 完整人设库和 prompt 示例见 [references/persona-engineering.md](references/persona-engineering.md)

---

## 推荐人设组合

常见场景的"前辈团"速配：

**架构设计评审**：Fowler（架构模式）+ DHH（约定优于配置）+ Torvalds（过度设计批评）

**并发问题诊断**：Armstrong（Let it crash）+ TJ（异步模式）+ Dahl（反思根因）

**代码质量整治**：Fowler（渐进重构）+ Uncle Bob（Clean Code）+ Hickey（复杂性根源）

**技术选型决策**：Lamport（理论分析）+ Sorhus（简单实用）+ Dahl（长期后悔点）

---

## 议会进阶玩法

### 辩论模式
- **Round 1**：各自给出立场
- **Round 2**：互相批评对方的方案
- **Round 3**：考虑批评后给出最终建议

### 红队模式
- **Opus**：提出方案
- **GPT**：全力攻击方案，找弱点
- **Sonnet**：防守和修补弱点

### 共识收敛
- 把所有人的观点展示给每个模型，问"考虑了这些视角后，你的新观点是什么？"
- 重复直到观点趋于稳定

---

## 会议归档

**会议记录必须持久化保存**，不要放 `/tmp`（重启会清空）。

```
~/.openclaw/workspace/pr-review/
└── council-YYYY-MM-DD[-topic]/
    ├── README.md                    # 会议概览（用模板）
    ├── council-opus-prompt.txt      # Opus 的 prompt
    ├── council-opus-output.txt      # Opus 的完整输出
    ├── council-sonnet-prompt.txt
    ├── council-sonnet-output.txt
    ├── council-gpt-prompt.txt
    └── council-gpt-output.txt
```

归档步骤：
```bash
ARCHIVE=~/.openclaw/workspace/pr-review/council-$(date +%Y-%m-%d)-topic
mkdir -p "$ARCHIVE"
cp /tmp/council-*.txt "$ARCHIVE/"
# 然后写 README.md（模板见 references/session-readme-template.md）
```

→ README 模板见 [references/session-readme-template.md](references/session-readme-template.md)

---

## 安全提醒

- `--force` 会自动执行所有变更，**务必在独立分支上使用**
- 并行任务之间**不能有文件冲突**，否则会互相覆盖
- 每个 session 完成后先 `tmux capture-pane` 保存输出再关闭
- 人设是激活知识用的，不是角色扮演秀——保持专业分析为主
