---
name: curriculum-generator
description: Intelligent educational curriculum generation system with strict step enforcement and human escalation policies
metadata:
  openclaw:
    requires:
      bins: ["node"]
      env: []
      config: []
    version: "1.0.0"
    author: "Apni Pathshala"
---

## DEBUG MODE

When the user includes "debug mode" or "show searches" in their curriculum request:

**Enable verbose output:**
- Print every neo-ddg-search query before executing
- Print number of results returned
- Print first 2-3 URLs extracted
- Print resource assignment: "Assigning to {topic}: {url}"

**Example debug output:**
```
[DEBUG] Executing neo-ddg-search("Python basics tutorial for beginners")
[DEBUG] Search returned 10 results
[DEBUG] Extracting URLs...
[DEBUG] Found: https://www.youtube.com/watch?v=rfscVS0vtbw
[DEBUG] Found: https://www.freecodecamp.org/learn/scientific-computing-with-python/
[DEBUG] Assigning to "Python Basics": https://www.youtube.com/watch?v=rfscVS0vtbw
```


## Dependencies

### Required Skills
This skill requires the following other skills to be installed:

- **neo-ddg-search**: For web searching educational resources
  - Install: `clawhub install neobotjan2026/neo-ddg-search`
  - Verify: Check if neo-ddg-search skill exists in skills directory

### Dependency Verification

At the start of curriculum generation, verify neo-ddg-search is available:
```
IF neo-ddg-search skill NOT found:
   🚨 DEPENDENCY MISSING
   
   The curriculum generator requires the neo-ddg-search skill for finding educational resources.
   
   Please install it:
   clawhub install neobotjan2026/neo-ddg-search
   
   Then restart this process.
   
   ⚠️ GENERATION CANNOT PROCEED without search capability
   
   STOP
```

### Search Tool Health Check

Before starting resource research, perform a test search:
```
Test: neo-ddg-search("Python tutorial test")

IF successful:
   ✅ Search tool operational
   Proceeding with resource research...
   
IF failed:
   🚨 SEARCH TOOL ERROR
   
   neo-ddg-search is installed but not responding correctly.
   
   Error: {error_details}
   
   Please check:
   • neo-ddg-search skill is properly installed
   • Internet connection is available
   • No firewall blocking DuckDuckGo
   
   ⚠️ Cannot proceed with resource research
   
   ESCALATE
```

# Curriculum Generator Skill

## Purpose
This skill helps generate customized educational curricula for PODs (Points of Delivery) through a structured, step-enforced process with mandatory human escalation when needed.

## Core Capabilities
- Guided requirement gathering via structured questionnaire
- Research-based curriculum design or assessment
- Excel (.xlsx) output generation
- Local memory storage for continuous improvement
- Background task execution
- Strict human escalation policy enforcement

## Storage Locations
- **Memory**: `~/.openclaw/skills/curriculum-generator/memory/`
- **Outputs**: `~/.openclaw/skills/curriculum-generator/outputs/`
- **Templates**: `~/.openclaw/skills/curriculum-generator/templates/`

## Activation Triggers
This skill activates when the user:
- Says "create curriculum", "design curriculum", or "assess curriculum"
- Says "curriculum help" or "start curriculum process"
- Explicitly requests curriculum generation for a POD

## CRITICAL RULES (NON-NEGOTIABLE)

### Core Principle
You MUST ask a human whenever you are forced to guess, infer, or trade off risk.
If a wrong decision could affect students, teachers, or POD operations, escalation is MANDATORY.

### Hard-Stop Escalation Triggers
You MUST stop and escalate to human if ANY of these occur:

**A. Missing or Ambiguous Inputs**
- Target age/grade level is unclear
- Teacher availability or capability is unknown
- Daily lab hours are not specified
- Infrastructure reliability (computers/internet/electricity) is unclear
- Whether existing curriculum exists is not confirmed

**B. Teacher Capability Risk**
- Teachers cannot operate computers independently
- Teachers lack experience running labs
- Teachers cannot manage lab discipline or session flow

**C. Operational Infeasibility**
- Curriculum hours exceed available lab hours
- Sessions per week exceed teacher capacity
- Student-to-computer ratio is unsafe
- Infrastructure cannot support planned activities

**D. High-Risk Curriculum Changes**
- Removing major learning outcomes
- Changing curriculum duration significantly
- Shifting learning area (e.g., digital literacy → employment readiness)
- Introducing new tools/platforms not previously used

**E. Contradictory Stakeholder Signals**
- Teachers say curriculum is too hard, students say too easy
- POD leader priorities conflict with feasibility
- Feedback loops contradict assessment data

### Escalation Format (MANDATORY)
When escalating, use this EXACT format:
```
🚨 HUMAN INPUT REQUIRED

Reason: [specific trigger]
Impact if Unresolved: [clear consequence]
Options (if any):
1. [option 1]
2. [option 2]

Awaiting Decision From: [POD Leader / Curriculum Owner]
```

## PROCESS FLOW

### STEP 0: SCENARIO IDENTIFICATION (MANDATORY)
Before anything else, determine:
- **Scenario A**: Assessment of existing curriculum
- **Scenario B**: Design of new curriculum

If unclear, STOP and ask user to confirm. Do NOT proceed without classification.

---

### SCENARIO A: ASSESSING EXISTING CURRICULUM

#### STEP A1: Gather Basic Information
Collect ALL of the following using the structured form:

**Section 0: Request Metadata**
- Request ID (auto-generate using timestamp)
- Date of Request (auto-capture)
- Requested By (Name + Role)
- POD Name (REQUIRED)
- Scenario Type (must be selected)

⚠️ If Scenario Type not selected → HARD STOP

**Section 1: Target Audience Profile (MANDATORY)**
1. Primary Student Group:
   - Age range
   - Grade/Education level
2. Student Background (check all that apply):
   - First-time computer users
   - Basic exposure (mouse, keyboard)
   - Prior digital lab experience
   - Mixed levels
3. Language Comfort:
   - Medium of instruction
   - English proficiency (Low/Medium/High)
4. Special Constraints:
   - Learning disabilities
   - Attendance inconsistency
   - Social/economic limitations

⚠️ If age/grade missing → HARD STOP and escalate

**Section 2: POD & Infrastructure Details (MANDATORY)**
1. Lab Infrastructure:
   - Number of computers
   - Average students per session
   - Internet availability (Reliable/Unstable/None)
   - Power backup (Yes/No)
2. Daily Lab Usage:
   - Available hours per day
   - Days per week lab is operational
3. Existing Tools/Platforms:
   - Operating System
   - Software already installed
   - Internet restrictions

⚠️ If lab hours or computer count missing → HARD STOP and escalate

**Section 3: Teacher Capability & Availability (MANDATORY)**
1. Number of Teachers Assigned
2. Teacher Availability:
   - Days per week
   - Hours per day
3. Teacher Capability Assessment:
   - Can operate computers independently? (Yes/No)
   - Comfortable managing digital labs? (Yes/No)
   - Prior experience with similar programs? (Yes/No)
4. Training Requirement:
   - No training needed
   - Short training needed
   - Extensive training needed

⚠️ Any "No" in capability assessment → Potential escalation

#### STEP A2: Stakeholder Inputs (Structured Summary)
Simulate structured stakeholder inputs based on provided data:
- **POD Leader**: Effectiveness, challenges, change needs
- **Teachers**: Teaching experience, curriculum gaps, student progress
- **Students**: Difficulty level, engagement, relevance

Then perform Teacher Capability Assessment:
- Can teachers operate computers independently?
- Can they run the lab as per curriculum?
- Can they manage discipline, safety, and session flow?
- Identify training gaps (if any)

#### STEP A3: Curriculum Evaluation
Evaluate curriculum on these dimensions:
- Relevance to student needs
- Alignment with industry/digital literacy goals
- Flexibility for varied learning speeds
- Outcome clarity and measurability
- Technology integration quality

Then perform Operational Feasibility Check:
- Lab schedule feasibility
- Teacher sufficiency
- Infrastructure readiness (computers, internet, electricity)

#### STEP A4: Recommendations
- Clearly state whether modification is required or optional
- If required, propose specific, actionable changes
- Flag risks explicitly

End with:
**Status: Draft Assessment – Pending Human Review**

---

### SCENARIO B: DESIGNING A NEW CURRICULUM

#### STEP B1: Define Curriculum Foundation
Explicitly define:
- **Learning Areas**: Digital Literacy / Academic Empowerment / Skill Development / Employment Readiness
- **Target Audience**: Grade, background
- **Clear, measurable Learning Outcomes** (no vague outcomes allowed)

#### STEP B2: Develop Course Structure
Generate:
- Modules and sub-topics
- Weekly lesson breakdown
- Learning objective per lesson
- Program duration (e.g., 3 months / 6 months)
- Class frequency

**Lab Planning (Mandatory)**:
- Daily lab operating hours
- Sessions per week
- Max students per session

**END OF STEP B2 - MANDATORY ACTION BEFORE PROCEEDING:**
```
BEFORE moving to Step B3, execute this command sequence:

1. Review the curriculum structure you just created
2. Identify ALL topics that will appear in the final output
3. For EACH topic, RIGHT NOW, execute:
   
   neo-ddg-search("{topic} tutorial for beginners")
   
4. Extract the first valid educational URL from results
5. Store it in a resource_map dictionary:
   
   resource_map["{topic}"] = "https://..."

6. Verify resource_map has entries for ALL topics
7. Only then proceed to Step B3

Example:
Topic: "Python Lists"
Execute: neo-ddg-search("Python Lists tutorial for beginners")
Result: Found https://www.youtube.com/watch?v=W8KRzm-HUcc
Store: resource_map["Python Lists"] = "https://www.youtube.com/watch?v=W8KRzm-HUcc"

DO NOT SKIP THIS. DO NOT PROCEED WITHOUT COMPLETING THIS.
```
```

## **Step 5: Create a Simpler Test in Telegram**

Now test with very explicit instructions. In Telegram, send:
```
Create a tiny test curriculum:
- Topic: HTML Basics  
- Duration: 1 week only
- 2 lessons total

IMPORTANT INSTRUCTIONS:
1. After creating the structure, use neo-ddg-search to find resources for EACH lesson
2. Before generating the CSV file, verify ALL resource links are real URLs
3. Show me each search you perform
4. If ANY resource link is empty or "TBD", STOP and search for it immediately

Start now.

**MANDATORY: End of Step B2 Resource Collection**

Before proceeding to Step B3, you MUST complete this:
```
STOP HERE.

Before moving to Step B3, execute this sequence:

1. List all topics you just created: [topic1, topic2, topic3, ...]

2. Create an empty dictionary: resource_links = {}

3. FOR EACH topic:
   a. Execute: result = neo-ddg-search("{topic} tutorial for beginners")
   b. Look at the result text
   c. Find all text that starts with "https://"
   d. Extract the complete URL
   e. Store: resource_links[topic] = that_url
   f. Print: "Stored for {topic}: {that_url}"

4. Verify resource_links has ALL topics

5. Print the complete resource_links dictionary

6. ONLY THEN proceed to Step B3

Example for "HTML Basics":
   Execute: neo-ddg-search("HTML Basics tutorial for beginners")
   Result contains: "...https://www.youtube.com/watch?v=pQN-pnXPaVg..."
   Extract: "https://www.youtube.com/watch?v=pQN-pnXPaVg"
   Store: resource_links["HTML Basics"] = "https://www.youtube.com/watch?v=pQN-pnXPaVg"
   Print: "Stored for HTML Basics: https://www.youtube.com/watch?v=pQN-pnXPaVg"

DO THIS FOR EVERY SINGLE TOPIC BEFORE MOVING ON.
```


**END OF STEP B2 - RESOURCE COLLECTION (MANDATORY)**
```
YOU MUST NOW COLLECT RESOURCES BEFORE PROCEEDING.

Execute this EXACT sequence:

1. Create empty dictionary: resource_links = {}

2. List all topics from Step B2

3. For EACH topic, execute:
   
   bash_tool: python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py "{topic} tutorial for beginners" 5
   
4. From the output, find all text starting with "https://"

5. Take the first URL that contains "youtube.com", or if none, the first URL

6. Store it: resource_links[topic] = that_url

7. Print: "Collected for {topic}: {that_url}"

8. After ALL topics are done, print the complete resource_links dictionary

9. Verify every topic has a URL

10. ONLY THEN create the CSV using URLs from resource_links

DO NOT WRITE "TBD" IN THE CSV.
USE THE URLS FROM resource_links DICTIONARY.

If you cannot find a URL for a topic, STOP and ESCALATE.
Do not proceed to CSV generation without URLs for all topics.
```
```

## **Save and Test**

Save the file, then in Telegram:
```
reload skills
```

Then test with a VERY simple example:
```
Create curriculum:
- Topic: HTML only
- 1 lesson total
- Show me EVERY step

After you build the structure:
1. Search for HTML resources using: python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py "HTML tutorial for beginners" 5
2. Show me the raw output
3. Extract the URLs
4. Show me what URL you extracted
5. Show me the CSV content BEFORE writing it
6. If Resource Link shows "TBD", STOP immediately

Start.
```

## **What to Watch For**

You should see output like:
```
✅ Course structure complete

🔍 Starting resource search...

Topic: HTML Basics
Executing: python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py "HTML Basics tutorial for beginners" 5

[Results shown]
[1] HTML Tutorial | https://www.youtube.com/watch?v=...
[2] Learn HTML | https://www.w3schools.com/html/

Found 2 URLs
Selected: https://www.youtube.com/watch?v=...
✅ Stored for HTML Basics: https://www.youtube.com/watch?v=...

Resource Links Dictionary:
  HTML Basics: https://www.youtube.com/watch?v=...

📋 CSV Preview:
Covered Topics           | Resource Link
HTML Basics             | https://www.youtube.com/watch?v=...

Writing file...


#### STEP B2.5: RESOURCE LINK POPULATION (SIMPLIFIED)

**After completing Step B2 structure, execute this EXACT process:**

### Simple 3-Step Process Per Topic

**For EACH topic:**

**STEP 1: Search**
```bash
python3 ~/.openclaw/workspace/skills/neo-ddg-search/scripts/search.py "{topic} tutorial for beginners" 5
```

**STEP 2: Look at output and extract FIRST URL**
- Scan the output line by line
- When you see `https://` → copy everything from `https://` until the next space
- That's your URL

**STEP 3: Store it**
```
resource_links["{topic}"] = "the_url_you_found"
```

**Then IMMEDIATELY move to next topic. Do NOT do additional searches unless the first one returns ZERO results.**

### Hard Limit: 1 Search Per Topic

**RULE**: Execute ONE search per topic. Extract ONE URL. Move on.

Do NOT:
- ❌ Execute multiple searches for the same topic
- ❌ Try to find "better" resources
- ❌ Analyze quality extensively
- ❌ Wait or pause

DO:
- ✅ Search once
- ✅ Grab first URL
- ✅ Move to next topic
- ✅ Complete all topics quickly

### Exact Execution Template
```
Print: "🔍 Resource Research Starting..."
Print: ""

resource_links = {}
topics = [list of all topics from Step B2]

For topic in topics:
    Print: f"Topic: {topic}"
    
    # Execute search (ONE TIME ONLY)
    result = bash_tool(f'python3 ~/.openclaw/workspace/skills/neo-ddg-search/scripts/search.py "{topic} tutorial" 5')
    
    # Extract first URL (simple method)
    url = None
    for line in result.split('\n'):
        if 'https://' in line:
            start = line.find('https://')
            end_of_line = line[start:]
            # Get URL until space or end
            space_index = end_of_line.find(' ')
            if space_index > 0:
                url = end_of_line[:space_index]
            else:
                url = end_of_line.strip()
            break  # Take FIRST URL and stop
    
    if url:
        resource_links[topic] = url
        Print: f"  ✅ {url}"
    else:
        resource_links[topic] = "MANUAL_RESEARCH_NEEDED"
        Print: f"  ⚠️ No URL found - marked for manual research"
    
    # IMMEDIATELY continue to next topic
    
Print: ""
Print: "✅ Resource research complete"
Print: f"Collected {len(resource_links)} resource links"
Print: ""
```

### Time Limit

**Maximum time for resource research: 2 minutes total**

If you're taking longer than 2 minutes for resource collection, you're doing something wrong. This should be fast:
- 5 seconds per search
- 2 topics = 10 seconds
- 10 topics = 50 seconds

### What Gets Stored
```python
# Good examples:
resource_links["Python Basics"] = "https://datascientest.com/en/python-variables-beginners-guide"
resource_links["HTML Intro"] = "https://www.w3schools.com/python/python_variables.asp"

# Acceptable if no URL found:
resource_links["Obscure Topic"] = "MANUAL_RESEARCH_NEEDED"

# NEVER acceptable:
resource_links["Topic"] = "TBD"  # ❌
resource_links["Topic"] = ""     # ❌
```

### After Collection: Immediate CSV Generation

**Do NOT pause or wait. Immediately proceed to CSV generation.**
```
Print: "📄 Generating CSV with collected resources..."

csv_data = []

for topic in curriculum_structure:
    resource_url = resource_links.get(topic, "MANUAL_RESEARCH_NEEDED")
    
    csv_row = {
        "Curriculum ID": curriculum_id,
        "File Name": file_name,
        "Target POD Type": pod_type,
        "Clusters": clusters,
        "Content Type": content_type,
        "Covered Topics": topic,
        "Owner": owner,
        "Resource Link": resource_url,  # ← Use collected URL
        "Document Creation Date": date,
        "Last Updated On": date
    }
    csv_data.append(csv_row)

write_csv(csv_data)
Print: "✅ CSV file generated"
```

### Example: Complete 2-Topic Execution

**Topics**: ["Python Basics", "Python Functions"]
```
🔍 Resource Research Starting...

Topic: Python Basics
  Executing search...
  [Results received]
  Found URL: https://datascientest.com/en/python-variables-beginners-guide
  ✅ https://datascientest.com/en/python-variables-beginners-guide

Topic: Python Functions  
  Executing search...
  [Results received]
  Found URL: https://www.w3schools.com/python/python_functions.asp
  ✅ https://www.w3schools.com/python/python_functions.asp

✅ Resource research complete
Collected 2 resource links

📄 Generating CSV with collected resources...
✅ CSV file generated: Python_Basics_v1.0.csv
```

**Total time**: ~15 seconds

### No Escalation for "Imperfect" Resources

**Accept whatever URL you find in the first search.**

Priority is:
1. Speed ✅
2. Completion ✅
3. Perfect resources ⚠️ (nice to have, not required)

If the first search returns W3Schools instead of YouTube, that's FINE. Use it and move on.

### Escalation Only For Zero Results

**Only escalate if:**
- ❌ Search returns absolutely no results
- ❌ Search returns results but contains ZERO URLs
- ❌ Search tool completely fails

**Do NOT escalate if:**
- ✅ URL is from W3Schools instead of YouTube (still good!)
- ✅ URL is from a lesser-known educational site (acceptable!)
- ✅ URL is documentation instead of video (perfectly fine!)

### Debug Output (When Requested)

If user requests DEBUG MODE:
```
[DEBUG] Topic: Python Basics
[DEBUG] Command: python3 ~/.openclaw/workspace/skills/neo-ddg-search/scripts/search.py "Python Basics tutorial" 5
[DEBUG] Results: 5 entries returned
[DEBUG] Extracting URLs...
[DEBUG] Line 1: Contains 'https://datascientest.com/...'
[DEBUG] Extracted: https://datascientest.com/en/python-variables-beginners-guide
[DEBUG] Storing: resource_links["Python Basics"] = "https://datascientest.com/..."
[DEBUG] ✅ Complete - moving to next topic
```

**MANDATORY: After Step B2, execute resource collection IMMEDIATELY**
```
After completing Step B2 course structure:

1. DO NOT pause
2. DO NOT ask for confirmation
3. IMMEDIATELY start resource collection
4. Use the Simple 3-Step Process above
5. Complete ALL topics within 2 minutes
6. Then IMMEDIATELY generate CSV
7. Do NOT wait between steps

This should be ONE CONTINUOUS FLOW:
Step B2 → Resource Collection → CSV Generation → Done

No breaks. No pauses. No waiting.
```


### Implementation in Curriculum Generation

**After Step B2 (course structure), do this:**
```
Print: "🔍 Starting resource search for all topics..."
Print: ""

Initialize: resource_links = {}

For each topic in curriculum:
    Print: "Topic: {topic}"
    
    # Execute search
    command = f"python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py \"{topic} tutorial for beginners\" 5"
    result = execute_bash(command)
    
    # Extract URLs (look for https://)
    lines = result.split('\n')
    urls = []
    for line in lines:
        if 'https://' in line:
            # Extract the URL part
            start = line.find('https://')
            # Find end (space or newline)
            rest = line[start:]
            space_pos = rest.find(' ')
            if space_pos > 0:
                url = rest[:space_pos]
            else:
                url = rest.strip()
            urls.append(url)
    
    Print: f"  Found {len(urls)} URLs"
    
    # Choose best URL
    best_url = None
    for url in urls:
        if 'youtube.com' in url:
            best_url = url
            break
    
    if not best_url and urls:
        for url in urls:
            if 'freecodecamp.org' in url:
                best_url = url
                break
    
    if not best_url and urls:
        best_url = urls[0]  # Use first URL
    
    if best_url:
        resource_links[topic] = best_url
        Print: f"  ✅ Stored: {best_url}"
    else:
        Print: f"  ❌ No URLs found - trying alternative search..."
        # Try one more time
        alt_command = f"python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py \"{topic} free course\" 5"
        alt_result = execute_bash(alt_command)
        # Extract URLs again...
        # [same extraction logic]
        
        if alt_urls:
            resource_links[topic] = alt_urls[0]
            Print: f"  ✅ Stored: {alt_urls[0]}"
        else:
            ESCALATE(f"No resources found for {topic}")
    
    Print: ""

Print: "✅ Resource collection complete!"
Print: f"Total topics: {len(resource_links)}"
Print: ""
Print: "Resource Links Dictionary:"
for topic, url in resource_links.items():
    Print: f"  {topic}: {url}"
```



#### STEP B3: Teacher Preparation & Readiness
Specify:
- Teacher resources required
- Teaching methodology (interactive, adaptable)
- Teacher readiness evaluation:
  - Prior experience
  - Comfort with computer labs
- Whether short training is required (Yes/No + why)

#### STEP B4: Assessments & Feedback Design
Define:
- Formative assessments (quizzes, projects, assignments)
- Summative assessment (final exam/capstone project)
- What each assessment measures

#### STEP B5: Continuous Improvement Loop
Define:
- Feedback sources (teachers, students, assessments)
- Review frequency
- Criteria for curriculum revision

---

## RESOURCE RESEARCH (MANDATORY)

### ANTI-STUCK RULE

**If resource collection is taking longer than 3 minutes total:**

STOP what you're doing and execute this:
```
Print: "⏱️ Resource collection timeout (3 min exceeded)"
Print: "Completing with available resources..."

For any topic without a resource:
    resource_links[topic] = "MANUAL_RESEARCH_NEEDED"

Proceed immediately to CSV generation
```

**Never get stuck searching indefinitely.**
```

## **Test Again**

Save the file and test:
```
reload skills
```

Then:
```
Create curriculum:
- Python basics
- 2 lessons
- 1 week

DO NOT GET STUCK. If resource search takes more than 1 minute total, skip to CSV generation.

Show me when you start resource search and when you finish.
```

## **What Should Happen**

You should see:
```
🔍 Resource Research Starting...

Topic: Lesson 1 - Python Intro
  ✅ https://datascientest.com/en/python-variables-beginners-guide

Topic: Lesson 2 - Python Functions
  ✅ https://www.w3schools.com/python/python_functions.asp

✅ Resource research complete (15 seconds)
Collected 2 resource links

📄 Generating CSV...
✅ Done
```

**NOT this:**
```
Topic: Lesson 1
  Executing search...
  [Results]
  Trying alternative search...
  [More results]
  Evaluating quality...
  [STUCK HERE]  ← Never gets to CSV

### How to Execute Searches

To search for resources, use this EXACT command:
```bash
python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py "YOUR QUERY HERE" 5
```

This returns search results with URLs that you must extract.

### Simple Search and Extract Process

**For EACH topic in the curriculum:**

#### Step 1: Execute Search
```bash
# Example for "HTML Basics"
python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py "HTML basics tutorial for beginners" 5
```

#### Step 2: Look at the Output

The output looks like this:
```
[1] Page Title | Year | Type | Site https://example.com/url1
Description text

[2] Another Title | Year | Type | Site https://another.com/url2  
More description
```

#### Step 3: Extract URLs (Simple Pattern)

**Look for any text starting with `https://`**

From the example above, extract:
- `https://example.com/url1`
- `https://another.com/url2`

#### Step 4: Choose Best URL

Priority order:
1. URLs containing `youtube.com` (first choice)
2. URLs containing `freecodecamp.org` (second choice)
3. URLs containing `w3schools.com` (third choice)
4. Any other educational site
5. If none found, use the first URL

#### Step 5: Store the URL

Store in a simple format:
```
Topic: HTML Basics
Resource: https://www.youtube.com/watch?v=...
```

### Complete Example Workflow

**Topic**: "Python Lists"

**Step 1 - Search**:
```bash
python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py "Python lists tutorial for beginners" 5
```

**Step 2 - Output Received**:
```
[1] Python Lists Tutorial | 2023 | Video | YouTube https://www.youtube.com/watch?v=W8KRzm-HUcc
Learn Python lists from scratch

[2] Python Lists Guide | 2024 | Article | W3Schools https://www.w3schools.com/python/python_lists.asp
Complete guide to Python lists
```

**Step 3 - Extract URLs**:
- Found: `https://www.youtube.com/watch?v=W8KRzm-HUcc`
- Found: `https://www.w3schools.com/python/python_lists.asp`

**Step 4 - Choose Best**:
- First URL contains "youtube.com" → Select this one
- Selected: `https://www.youtube.com/watch?v=W8KRzm-HUcc`

**Step 5 - Store**:
```
resource_links["Python Lists"] = "https://www.youtube.com/watch?v=W8KRzm-HUcc"
```

### Before Writing CSV

**MANDATORY CHECK:**
```
Print: "🔍 Verifying resource links before CSV generation..."
Print: ""

csv_data = []

for row in curriculum_structure:
    topic = row['topic']
    
    # Get resource from resource_links dictionary
    if topic in resource_links:
        resource_url = resource_links[topic]
    else:
        Print: f"❌ ERROR: No resource link for '{topic}'"
        STOP
    
    # Verify it's a valid URL
    if not resource_url.startswith('http'):
        Print: f"❌ ERROR: Invalid URL for '{topic}': {resource_url}"
        STOP
    
    Print: f"✅ {topic}: {resource_url[:60]}..."
    
    # Add to CSV data
    csv_row = {
        "Curriculum ID": curriculum_id,
        "File Name": file_name,
        "Target POD Type": pod_type,
        "Clusters": clusters,
        "Content Type": content_type,
        "Covered Topics": topic,
        "Owner": owner,
        "Resource Link": resource_url,  # ← ACTUAL URL HERE
        "Document Creation Date": date,
        "Last Updated On": date
    }
    csv_data.append(csv_row)

Print: ""
Print: "✅ All rows verified with valid URLs"
Print: "📄 Writing CSV file..."

write_csv_file(csv_data)
```

### CSV Preview Before Writing

**Show user the data:**
```
Print: "📋 CSV Preview:"
Print: "=" * 80
Print: f"Covered Topics | Resource Link"
Print: "-" * 80
for row in csv_data:
    topic = row["Covered Topics"]
    url = row["Resource Link"]
    Print: f"{topic[:30]:30} | {url}"
Print: "=" * 80
Print: ""
Print: "Writing to file..."
```

### Escalation for Missing Resources

If after searching, a topic has no URL:
```
🚨 RESOURCE SEARCH FAILED - HUMAN INPUT REQUIRED

Topic: {topic_name}

Search 1: "python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py '{topic} tutorial for beginners' 5"
Result: {number} URLs found
None matched quality criteria

Search 2: "python3 ~/.openclaw/skills/neo-ddg-search/scripts/search.py '{topic} free course' 5"  
Result: {number} URLs found
None matched quality criteria

Issue: Cannot find suitable free educational resources

Options:
1. Modify topic name to be more general
2. Accept lower-quality resource if available
3. Mark for manual research

Awaiting Decision From: Curriculum Owner

⚠️ CSV generation paused
```


## OUTPUT GENERATION

## PRE-FILE-GENERATION CHECKLIST (MANDATORY)

**Before writing ANY output file, you MUST complete this checklist:**

### Checklist Item 1: Resource Link Verification

**STOP and verify:**
```
FOR EACH row in the curriculum data:
    topic = row['Covered Topics']
    resource_link = row['Resource Link']
    
    IF resource_link is empty OR resource_link == "TBD" OR resource_link == "N/A":
        
        PRINT "⚠️ Missing resource link for: {topic}"
        PRINT "🔍 Executing search now..."
        
        # Execute neo-ddg-search immediately
        search_query = f"{topic} tutorial for beginners"
        EXECUTE: neo-ddg-search(search_query)
        
        # Extract URLs from results
        urls = EXTRACT_URLS_FROM_RESULTS()
        
        IF urls found:
            row['Resource Link'] = urls[0]  # Use first result
            PRINT "✅ Found resource: {urls[0]}"
        ELSE:
            # Try alternative search
            search_query_2 = f"{topic} free course"
            EXECUTE: neo-ddg-search(search_query_2)
            urls = EXTRACT_URLS_FROM_RESULTS()
            
            IF urls found:
                row['Resource Link'] = urls[0]
                PRINT "✅ Found resource: {urls[0]}"
            ELSE:
                ESCALATE("Cannot find resources for {topic}")
                STOP_FILE_GENERATION
```

**You MUST see output like:**
```
Checking resource links before file generation...
✅ Row 1 - HTML Basics: Has resource link
✅ Row 2 - CSS Fundamentals: Has resource link  
⚠️ Row 3 - JavaScript: Missing resource link
🔍 Executing search now...
   Using neo-ddg-search: "JavaScript tutorial for beginners"
✅ Found resource: https://www.youtube.com/watch?v=...
✅ Row 3 - JavaScript: Resource link populated

All rows verified. Proceeding to file generation...
```

### Checklist Item 2: URL Format Validation

Verify all resource links are actual URLs:
```
FOR EACH resource_link in curriculum:
    IF NOT resource_link.startswith("http"):
        ERROR: "Invalid resource link format: {resource_link}"
        STOP
```

### Checklist Item 3: Final Count
```
total_topics = COUNT(curriculum rows)
topics_with_resources = COUNT(rows where Resource Link is valid URL)

PRINT "📊 Resource Link Status:"
PRINT "   Total topics: {total_topics}"
PRINT "   With resources: {topics_with_resources}"
PRINT "   Missing: {total_topics - topics_with_resources}"

IF topics_with_resources < total_topics:
    ESCALATE("Some topics still missing resources after search")
    STOP
ELSE:
    PRINT "✅ All topics have resource links. Safe to generate file."
```


## CSV/EXCEL FILE GENERATION - WITH RESOURCE LINKS

### Pre-Generation: Build Complete Resource Map

**Before writing any file, build a complete resource map:**
```python
# Initialize resource map
resource_map = {}

# Get all topics from curriculum structure
all_topics = extract_all_topics_from_curriculum()

print(f"\n📚 Building resource map for {len(all_topics)} topics...\n")

# For each topic, search and extract URL
for topic in all_topics:
    print(f"🔍 Topic: {topic}")
    
    # Execute search
    search_query = f"{topic} tutorial for beginners"
    print(f"   Searching: {search_query}")
    
    search_results = neo_ddg_search(search_query)
    
    # Extract URLs from results
    urls_found = extract_urls_from_search_result(search_results)
    print(f"   Found {len(urls_found)} URLs")
    
    # Select best URL
    if urls_found:
        best_url = select_best_url(urls_found)
        resource_map[topic] = best_url
        print(f"   ✅ Selected: {best_url}\n")
    else:
        print(f"   ⚠️ No URLs found, trying alternative search...")
        # Try alternative search
        alt_search = neo_ddg_search(f"{topic} free course")
        urls_found_alt = extract_urls_from_search_result(alt_search)
        
        if urls_found_alt:
            best_url = select_best_url(urls_found_alt)
            resource_map[topic] = best_url
            print(f"   ✅ Selected: {best_url}\n")
        else:
            resource_map[topic] = "ESCALATION_NEEDED"
            print(f"   ❌ No resources found - will escalate\n")

# Verify all topics have resources
missing_resources = [t for t, url in resource_map.items() if url == "ESCALATION_NEEDED"]

if missing_resources:
    print(f"🚨 {len(missing_resources)} topics need escalation:")
    for topic in missing_resources:
        print(f"   - {topic}")
    ESCALATE("Resource search failed for some topics")
    STOP
else:
    print(f"✅ All {len(all_topics)} topics have resource links!")
    print(f"📝 Proceeding to CSV generation...\n")
```

### CSV Generation with Resource Map

**When creating each row in the CSV:**
```python
for week_num, lesson in curriculum_structure:
    topic = lesson['topic']
    
    # Get resource link from resource_map
    resource_link = resource_map.get(topic, "ERROR_NO_RESOURCE")
    
    # Verify it's a valid URL
    if not resource_link.startswith("http"):
        print(f"ERROR: Invalid resource for {topic}: {resource_link}")
        STOP
    
    csv_row = {
        "Curriculum ID": curriculum_id,
        "File Name": file_name,
        "Target POD Type": pod_type,
        "Clusters": clusters,
        "Content Type": content_type,
        "Covered Topics": topic,
        "Owner": owner,
        "Resource Link": resource_link,  # ← USE THE ACTUAL URL HERE
        "Document Creation Date": creation_date,
        "Last Updated On": last_updated
    }
    
    csv_data.append(csv_row)
```

**Critical Check Before Writing**:
```python
# Final verification
print("\n🔍 Final CSV Data Verification:")
for i, row in enumerate(csv_data):
    resource = row["Resource Link"]
    topic = row["Covered Topics"]
    
    if resource == "TBD" or not resource.startswith("http"):
        print(f"❌ Row {i+1} ({topic}): INVALID resource '{resource}'")
        STOP_GENERATION
    else:
        print(f"✅ Row {i+1} ({topic}): {resource[:60]}...")

print("\n✅ All rows verified - writing file...")
write_csv_file(csv_data)
```



## FILE GENERATION - FINAL RESOURCE CHECK

**CRITICAL: Execute this immediately before writing the file:**
```python
# Pseudo-code showing the exact logic needed

def prepare_curriculum_data_for_file():
    """
    This function runs RIGHT BEFORE creating the CSV/Excel file.
    It ensures NO 'TBD' values slip through.
    """
    
    curriculum_rows = get_curriculum_structure()
    
    print("\n🔍 FINAL RESOURCE LINK CHECK (Pre-File-Generation)")
    print("=" * 50)
    
    for i, row in enumerate(curriculum_rows):
        topic = row['Covered Topics']
        resource_link = row.get('Resource Link', '')
        
        # Check if resource link is missing or placeholder
        if not resource_link or resource_link in ['TBD', 'N/A', '', 'null', 'None']:
            
            print(f"\n⚠️  Row {i+1}: '{topic}' has no resource link")
            print(f"    Current value: '{resource_link}'")
            print(f"    🔍 Searching now with neo-ddg-search...")
            
            # EXECUTE NEO-DDG-SEARCH HERE
            search_results = neo_ddg_search(f"{topic} tutorial for beginners free")
            
            # Extract URLs from search results
            urls_found = extract_urls_from_search_results(search_results)
            
            if urls_found and len(urls_found) > 0:
                row['Resource Link'] = urls_found[0]
                print(f"    ✅ Updated with: {urls_found[0]}")
            else:
                # Try one more time with different query
                print(f"    🔄 First search returned no URLs, trying again...")
                search_results_2 = neo_ddg_search(f"{topic} learn online")
                urls_found_2 = extract_urls_from_search_results(search_results_2)
                
                if urls_found_2 and len(urls_found_2) > 0:
                    row['Resource Link'] = urls_found_2[0]
                    print(f"    ✅ Updated with: {urls_found_2[0]}")
                else:
                    # HARD STOP - escalate
                    print(f"    ❌ FAILED: No resources found after 2 searches")
                    escalate_resource_failure(topic)
                    return None  # Don't proceed to file generation
        else:
            print(f"✅ Row {i+1}: '{topic}' has resource: {resource_link[:50]}...")
    
    print("\n" + "=" * 50)
    print("✅ All resource links verified. Proceeding to file write.\n")
    
    return curriculum_rows


# THEN write the file
verified_data = prepare_curriculum_data_for_file()

if verified_data is None:
    print("🚨 File generation cancelled - resource verification failed")
    # STOP HERE, don't write file
else:
    write_csv_file(verified_data)  # Only write if all checks passed
```

**What the user should see:**
```
🔍 FINAL RESOURCE LINK CHECK (Pre-File-Generation)
==================================================
✅ Row 1: 'HTML Basics' has resource: https://www.youtube.com/watch?v=pQN-pnXPaVg
✅ Row 2: 'CSS Fundamentals' has resource: https://www.youtube.com/watch?v=1Rs2ND1ryYc
⚠️  Row 3: 'JavaScript Intro' has no resource link
    Current value: 'TBD'
    🔍 Searching now with neo-ddg-search...
    Using neo-ddg-search: "JavaScript Intro tutorial for beginners free"
    ✅ Updated with: https://www.youtube.com/watch?v=PkZNo7MFNFg
✅ Row 4: 'DOM Manipulation' has resource: https://www.freecodecamp.org/...
==================================================
✅ All resource links verified. Proceeding to file write.

📄 Writing file: Web_Dev_Fundamentals_v1.0.csv
✅ File generated successfully!
```

### Excel File Structure
Generate `.xlsx` file with these columns:
1. Curriculum ID
2. File Name
3. Target POD Type
4. Clusters
5. Content Type
6. Covered Topics
7. Owner
8. **Resource Link** ⚠️ MUST contain actual URLs, NEVER "TBD"
9. Document Creation Date
10. Last Updated On

**Column Population Rules:**
- **Resource Link**: Search and populate with real URLs during curriculum generation
  - Format: "URL1 | URL2 | URL3" if multiple resources
  - Use web_search before writing Excel file
  - If search fails, escalate (never write "TBD")

### Mandatory Footer in Every Output
```
Curriculum Version: vX.X
Scenario: [Assessment / New Design]
Prepared By: Clawdbot
Status: Draft – Pending POD Leader / Authority Approval

Key Risks & Assumptions:
- [List all assumptions made]
- [List all identified risks]
```

## MEMORY MANAGEMENT

After each curriculum generation:
1. Save conversation context to `~/.openclaw/skills/curriculum-generator/memory/curriculum_[REQUEST_ID].json`
2. Store lessons learned in `~/.openclaw/skills/curriculum-generator/memory/learnings.md`
3. Track escalations in `~/.openclaw/skills/curriculum-generator/memory/escalations.log`

Memory file structure:
```json
{
  "request_id": "CUR_20260208_001",
  "date": "2026-02-08",
  "scenario": "new_design",
  "pod_name": "Example POD",
  "user": "madhur",
  "inputs_collected": {},
  "decisions_made": [],
  "escalations": [],
  "output_file": "curriculum_example_v1.0.xlsx",
  "learnings": []
}
```

## TOOLS REQUIRED

### Primary Tools
- `web_search`: Research educational standards, best practices, resources
- `create_file`: Generate Excel output files
- `bash_tool`: File management, directory operations
- `view`: Read memory files, check existing curricula

### Research Sources (Use web_search for)
- Educational standards and frameworks
- Age-appropriate learning resources
- Digital literacy benchmarks
- Best practices for computer lab education
- Free educational tools and platforms

## BACKGROUND TASK EXECUTION

When triggered, inform user:
```
🦞 Starting curriculum generation process...
This may take 5-10 minutes. I'll notify you when complete.
You can continue other tasks in the meantime.
```

Then proceed with research and generation in background.

## AUTO-APPROVAL ZONE
You may proceed autonomously when:
- All required inputs are present
- Teacher capability is sufficient
- Infrastructure is adequate
- Changes are incremental, not structural
- Curriculum stays within defined learning areas

Even then: Final output is still **Draft – Pending Human Review**

## META-RULES (VERY IMPORTANT)
You must NEVER:
- Fill gaps silently
- Optimize feasibility without disclosure
- Reduce curriculum scope to make things work unless approved

**Transparency > automation speed**

## Example Usage

**User**: "Create a new curriculum for our POD"

**Clawdbot**:
```
🦞 Curriculum Generator Activated

I'll guide you through a structured questionnaire to design the curriculum.

First, let me confirm: Is this for:
A) Assessing an existing curriculum
B) Designing a new curriculum from scratch

Please select A or B.
```

**User**: "B - new curriculum"

**Clawdbot**:
```
Perfect! Starting new curriculum design process.

SECTION 0: REQUEST METADATA
- Request ID: CUR_20260208_001 (auto-generated)
- Date: February 8, 2026
- Requested By: Please provide your name and role
- POD Name: Please provide the POD name
```

[Continue with structured questions...]

## Success Criteria
- All mandatory fields collected
- No silent assumptions made
- Appropriate escalations triggered
- Excel file generated successfully
- Memory stored for future reference
- User receives clear, actionable output

### Preferred Resource Platforms (Priority Order)
1. **YouTube**: Structured courses from reputable channels
   - FreeCodeCamp, Traversy Media, Programming with Mosh, Fireship
2. **Interactive Platforms**: Free tiers
   - freeCodeCamp.org, Khan Academy, Codecademy free, W3Schools
3. **Documentation**: Official docs when appropriate
   - MDN Web Docs, Python.org, official framework docs
4. **Written Tutorials**: High-quality articles
   - Dev.to, Medium (free articles), DigitalOcean tutorials
5. **Practice Platforms**: Free exercises
   - Exercism.io, LeetCode (free problems), HackerRank
