#!/usr/bin/env python3
"""
Auto-update agent MEMORY.md with learned trading rules.
"""

import argparse
import json
from datetime import datetime
from pathlib import Path
import re

DATA_DIR = Path(__file__).parent.parent / "data"
RULES_FILE = DATA_DIR / "learned_rules.json"


def load_rules():
    """Load learned rules from file."""
    if not RULES_FILE.exists():
        return None
    with open(RULES_FILE, "r") as f:
        return json.load(f)


def generate_memory_section(rules_data):
    """Generate markdown section for MEMORY.md."""
    if not rules_data or not rules_data.get("rules"):
        return None
    
    rules = rules_data["rules"]
    generated_at = rules_data.get("generated_at", datetime.now().isoformat())
    
    section = f"""

---

## 🧠 LEARNED RULES (Auto-Generated)

*Last updated: {datetime.fromisoformat(generated_at).strftime('%Y-%m-%d %H:%M')}*
*Based on {rules_data.get('total_rules', len(rules))} patterns from trade history*

### ✅ PREFER (High Win Rate Patterns)

"""
    
    prefer_rules = [r for r in rules if r["type"] == "PREFER"]
    avoid_rules = [r for r in rules if r["type"] == "AVOID"]
    
    if prefer_rules:
        for r in prefer_rules:
            conf = "HIGH" if r["confidence"] == "HIGH" else "MEDIUM"
            section += f"- **{r['rule']}** ({r['evidence']}) [{conf}]\n"
    else:
        section += "- *No high-confidence PREFER rules yet*\n"
    
    section += "\n### 🚫 AVOID (Low Win Rate Patterns)\n\n"
    
    if avoid_rules:
        for r in avoid_rules:
            conf = "HIGH" if r["confidence"] == "HIGH" else "MEDIUM"
            section += f"- **{r['rule']}** ({r['evidence']}) [{conf}]\n"
    else:
        section += "- *No high-confidence AVOID rules yet*\n"
    
    section += """
### 📋 How to Use These Rules

1. **Before opening a trade:** Check if any AVOID rules apply
2. **When conditions match PREFER:** Consider the setup more seriously
3. **Confidence levels:** HIGH = 10+ trades, MEDIUM = 5-9 trades

> ⚠️ These rules are learned from YOUR trading history. They reflect your actual performance, not theoretical strategies.

"""
    
    return section


def update_memory_file(memory_path, dry_run=False):
    """Update MEMORY.md with learned rules."""
    rules_data = load_rules()
    
    if not rules_data:
        print("❌ No learned rules found. Run generate_rules.py first!")
        return False
    
    new_section = generate_memory_section(rules_data)
    if not new_section:
        print("❌ Could not generate rules section.")
        return False
    
    memory_path = Path(memory_path)
    
    if not memory_path.exists():
        print(f"❌ Memory file not found: {memory_path}")
        return False
    
    with open(memory_path, "r") as f:
        content = f.read()
    
    # Remove existing learned rules section if present
    pattern = r'\n---\n\n## 🧠 LEARNED RULES \(Auto-Generated\).*?(?=\n---\n|\n## [^🧠]|\Z)'
    content = re.sub(pattern, '', content, flags=re.DOTALL)
    
    # Remove trailing whitespace
    content = content.rstrip()
    
    # Add new section
    new_content = content + new_section
    
    if dry_run:
        print("📄 DRY RUN - Would add this section:\n")
        print("=" * 50)
        print(new_section)
        print("=" * 50)
        return True
    
    # Write updated content
    with open(memory_path, "w") as f:
        f.write(new_content)
    
    print(f"✅ Updated: {memory_path}")
    print(f"   Added {len(rules_data['rules'])} learned rules")
    return True


def main():
    parser = argparse.ArgumentParser(description="Update MEMORY.md with learned rules")
    parser.add_argument("--memory-path", required=True, help="Path to MEMORY.md")
    parser.add_argument("--dry-run", action="store_true", help="Preview changes without writing")
    
    args = parser.parse_args()
    
    success = update_memory_file(args.memory_path, args.dry_run)
    
    if success and not args.dry_run:
        print("\n🧠 Agent memory updated with learned trading rules!")
        print("   The agent will now consider these patterns in future trades.")


if __name__ == "__main__":
    main()
