#!/usr/bin/env python3
"""
Log trades with full context for self-learning analysis.
"""

import argparse
import json
import os
import sys
from datetime import datetime
from pathlib import Path
import uuid

DATA_DIR = Path(__file__).parent.parent / "data"
TRADES_FILE = DATA_DIR / "trades.json"


def load_trades():
    """Load existing trades from file."""
    if TRADES_FILE.exists():
        with open(TRADES_FILE, "r") as f:
            return json.load(f)
    return {"trades": [], "metadata": {"created": datetime.now().isoformat()}}


def save_trades(data):
    """Save trades to file."""
    DATA_DIR.mkdir(parents=True, exist_ok=True)
    data["metadata"]["updated"] = datetime.now().isoformat()
    with open(TRADES_FILE, "w") as f:
        json.dump(data, f, indent=2)


def log_trade(args):
    """Log a new trade."""
    data = load_trades()
    
    trade = {
        "id": str(uuid.uuid4())[:8],
        "timestamp": datetime.now().isoformat(),
        "symbol": args.symbol.upper(),
        "direction": args.direction.upper(),
        "entry": float(args.entry),
        "exit": float(args.exit),
        "pnl_percent": float(args.pnl_percent),
        "result": args.result.upper(),
        "leverage": int(args.leverage) if args.leverage else None,
        "reason": args.reason,
        "indicators": json.loads(args.indicators) if args.indicators else {},
        "market_context": json.loads(args.market_context) if args.market_context else {},
        "notes": args.notes
    }
    
    # Extract day/hour from timestamp for analysis
    dt = datetime.now()
    trade["day_of_week"] = dt.strftime("%A").lower()
    trade["hour"] = dt.hour
    
    data["trades"].append(trade)
    save_trades(data)
    
    emoji = "✅" if trade["result"] == "WIN" else "❌"
    print(f"{emoji} Trade logged: {trade['symbol']} {trade['direction']} | PnL: {trade['pnl_percent']:+.2f}% | ID: {trade['id']}")
    return trade


def list_trades(args):
    """List recent trades."""
    data = load_trades()
    trades = data["trades"]
    
    if not trades:
        print("No trades logged yet.")
        return
    
    # Get last N trades
    n = args.last if args.last else len(trades)
    recent = trades[-n:]
    
    print(f"\n📊 Last {len(recent)} trades:\n")
    print(f"{'ID':<10} {'Date':<12} {'Symbol':<10} {'Dir':<6} {'PnL %':<8} {'Result':<6}")
    print("-" * 60)
    
    for t in recent:
        dt = datetime.fromisoformat(t["timestamp"]).strftime("%m/%d %H:%M")
        emoji = "✅" if t["result"] == "WIN" else "❌"
        print(f"{t['id']:<10} {dt:<12} {t['symbol']:<10} {t['direction']:<6} {t['pnl_percent']:+.2f}%   {emoji}")


def show_stats(args):
    """Show overall statistics."""
    data = load_trades()
    trades = data["trades"]
    
    if not trades:
        print("No trades logged yet.")
        return
    
    wins = [t for t in trades if t["result"] == "WIN"]
    losses = [t for t in trades if t["result"] == "LOSS"]
    
    total_pnl = sum(t["pnl_percent"] for t in trades)
    avg_win = sum(t["pnl_percent"] for t in wins) / len(wins) if wins else 0
    avg_loss = sum(t["pnl_percent"] for t in losses) / len(losses) if losses else 0
    
    win_rate = len(wins) / len(trades) * 100 if trades else 0
    
    print(f"""
📈 TRADING STATISTICS
{'='*40}

Total Trades:    {len(trades)}
Wins:            {len(wins)} ({win_rate:.1f}%)
Losses:          {len(losses)} ({100-win_rate:.1f}%)

Total PnL:       {total_pnl:+.2f}%
Avg Win:         {avg_win:+.2f}%
Avg Loss:        {avg_loss:+.2f}%

Profit Factor:   {abs(avg_win/avg_loss):.2f}x (if > 1 = profitable)
""")
    
    # By direction
    longs = [t for t in trades if t["direction"] == "LONG"]
    shorts = [t for t in trades if t["direction"] == "SHORT"]
    
    if longs:
        long_wins = len([t for t in longs if t["result"] == "WIN"])
        print(f"LONG Win Rate:   {long_wins/len(longs)*100:.1f}% (n={len(longs)})")
    
    if shorts:
        short_wins = len([t for t in shorts if t["result"] == "WIN"])
        print(f"SHORT Win Rate:  {short_wins/len(shorts)*100:.1f}% (n={len(shorts)})")


def main():
    parser = argparse.ArgumentParser(description="Log and analyze trades")
    parser.add_argument("--list", action="store_true", help="List trades")
    parser.add_argument("--last", type=int, help="Show last N trades")
    parser.add_argument("--stats", action="store_true", help="Show statistics")
    
    # Trade logging arguments
    parser.add_argument("--symbol", help="Trading pair (e.g., BTCUSDT)")
    parser.add_argument("--direction", choices=["LONG", "SHORT", "long", "short"], help="Trade direction")
    parser.add_argument("--entry", type=float, help="Entry price")
    parser.add_argument("--exit", type=float, help="Exit price")
    parser.add_argument("--pnl_percent", "--pnl", type=float, help="PnL percentage")
    parser.add_argument("--leverage", type=int, help="Leverage used")
    parser.add_argument("--reason", help="Reason for entry")
    parser.add_argument("--indicators", help="JSON string with indicators")
    parser.add_argument("--market_context", help="JSON string with market context")
    parser.add_argument("--result", choices=["WIN", "LOSS", "win", "loss"], help="Trade result")
    parser.add_argument("--notes", help="Additional notes")
    
    args = parser.parse_args()
    
    if args.stats:
        show_stats(args)
    elif args.list or args.last:
        list_trades(args)
    elif args.symbol and args.direction and args.entry and args.exit and args.pnl_percent and args.result:
        log_trade(args)
    else:
        parser.print_help()
        print("\n⚠️  To log a trade, provide: --symbol, --direction, --entry, --exit, --pnl_percent, --result")


if __name__ == "__main__":
    main()
