#!/usr/bin/env python3
"""
Generate trading rules from historical trade data.
"""

import json
from datetime import datetime
from pathlib import Path
from collections import defaultdict

DATA_DIR = Path(__file__).parent.parent / "data"
TRADES_FILE = DATA_DIR / "trades.json"
RULES_FILE = DATA_DIR / "learned_rules.json"


def load_trades():
    """Load trades from file."""
    if not TRADES_FILE.exists():
        return []
    with open(TRADES_FILE, "r") as f:
        data = json.load(f)
    return data.get("trades", [])


def calculate_win_rate(trades):
    """Calculate win rate for a list of trades."""
    if not trades:
        return 0, 0
    wins = len([t for t in trades if t["result"] == "WIN"])
    return wins / len(trades) * 100, len(trades)


def generate_rules(trades, min_trades=5, confidence_threshold=65):
    """Generate trading rules based on patterns."""
    rules = []
    overall_wr, _ = calculate_win_rate(trades)
    
    # Rule: Direction preference
    directions = defaultdict(list)
    for t in trades:
        directions[t["direction"]].append(t)
    
    for direction, dir_trades in directions.items():
        if len(dir_trades) >= min_trades:
            wr, n = calculate_win_rate(dir_trades)
            if wr >= confidence_threshold:
                rules.append({
                    "type": "PREFER",
                    "category": "direction",
                    "condition": f"direction == {direction}",
                    "rule": f"PREFER {direction} positions",
                    "evidence": f"Win rate: {wr:.0f}% over {n} trades",
                    "win_rate": wr,
                    "sample_size": n,
                    "confidence": "HIGH" if n >= 10 else "MEDIUM"
                })
            elif wr <= (100 - confidence_threshold):
                rules.append({
                    "type": "AVOID",
                    "category": "direction",
                    "condition": f"direction == {direction}",
                    "rule": f"AVOID {direction} positions",
                    "evidence": f"Win rate: {wr:.0f}% over {n} trades",
                    "win_rate": wr,
                    "sample_size": n,
                    "confidence": "HIGH" if n >= 10 else "MEDIUM"
                })
    
    # Rule: Day of week
    days = defaultdict(list)
    for t in trades:
        day = t.get("day_of_week")
        if day:
            days[day].append(t)
    
    for day, day_trades in days.items():
        if len(day_trades) >= min_trades:
            wr, n = calculate_win_rate(day_trades)
            if wr >= confidence_threshold + 10:  # Higher threshold for day rules
                rules.append({
                    "type": "PREFER",
                    "category": "timing",
                    "condition": f"day_of_week == {day}",
                    "rule": f"PREFER trading on {day.title()}",
                    "evidence": f"Win rate: {wr:.0f}% over {n} trades",
                    "win_rate": wr,
                    "sample_size": n,
                    "confidence": "HIGH" if n >= 10 else "MEDIUM"
                })
            elif wr <= (100 - confidence_threshold - 10):
                rules.append({
                    "type": "AVOID",
                    "category": "timing",
                    "condition": f"day_of_week == {day}",
                    "rule": f"AVOID trading on {day.title()}",
                    "evidence": f"Win rate: {wr:.0f}% over {n} trades",
                    "win_rate": wr,
                    "sample_size": n,
                    "confidence": "HIGH" if n >= 10 else "MEDIUM"
                })
    
    # Rule: Leverage
    high_lev = [t for t in trades if t.get("leverage") and t["leverage"] >= 10]
    low_lev = [t for t in trades if t.get("leverage") and t["leverage"] < 10]
    
    if len(high_lev) >= min_trades:
        wr, n = calculate_win_rate(high_lev)
        if wr <= 45:  # High leverage with low win rate
            rules.append({
                "type": "AVOID",
                "category": "risk",
                "condition": "leverage >= 10",
                "rule": "AVOID leverage >= 10x",
                "evidence": f"Win rate: {wr:.0f}% over {n} high-leverage trades",
                "win_rate": wr,
                "sample_size": n,
                "confidence": "HIGH" if n >= 10 else "MEDIUM"
            })
    
    # Rule: RSI-based
    rsi_trades = [t for t in trades if t.get("indicators", {}).get("rsi") is not None]
    
    oversold = [t for t in rsi_trades if t["indicators"]["rsi"] < 30]
    overbought = [t for t in rsi_trades if t["indicators"]["rsi"] > 70]
    
    if len(oversold) >= min_trades:
        # Check LONG on oversold
        oversold_longs = [t for t in oversold if t["direction"] == "LONG"]
        if len(oversold_longs) >= 3:
            wr, n = calculate_win_rate(oversold_longs)
            if wr >= confidence_threshold:
                rules.append({
                    "type": "PREFER",
                    "category": "indicator",
                    "condition": "RSI < 30 AND direction == LONG",
                    "rule": "PREFER LONG when RSI < 30 (oversold)",
                    "evidence": f"Win rate: {wr:.0f}% over {n} trades",
                    "win_rate": wr,
                    "sample_size": n,
                    "confidence": "HIGH" if n >= 10 else "MEDIUM"
                })
    
    if len(overbought) >= min_trades:
        # Check SHORT on overbought
        overbought_shorts = [t for t in overbought if t["direction"] == "SHORT"]
        if len(overbought_shorts) >= 3:
            wr, n = calculate_win_rate(overbought_shorts)
            if wr >= confidence_threshold:
                rules.append({
                    "type": "PREFER",
                    "category": "indicator",
                    "condition": "RSI > 70 AND direction == SHORT",
                    "rule": "PREFER SHORT when RSI > 70 (overbought)",
                    "evidence": f"Win rate: {wr:.0f}% over {n} trades",
                    "win_rate": wr,
                    "sample_size": n,
                    "confidence": "HIGH" if n >= 10 else "MEDIUM"
                })
    
    # Sort rules by confidence and win rate
    rules.sort(key=lambda x: (x["confidence"] == "HIGH", x["win_rate"]), reverse=True)
    
    return rules


def save_rules(rules):
    """Save rules to file."""
    DATA_DIR.mkdir(parents=True, exist_ok=True)
    output = {
        "generated_at": datetime.now().isoformat(),
        "total_rules": len(rules),
        "rules": rules
    }
    with open(RULES_FILE, "w") as f:
        json.dump(output, f, indent=2)
    return RULES_FILE


def main():
    trades = load_trades()
    
    if not trades:
        print("❌ No trades logged yet. Log some trades first!")
        return
    
    if len(trades) < 5:
        print(f"⚠️  Only {len(trades)} trades. Need at least 5 for rule generation.")
        print("   Keep logging trades to discover patterns!")
        return
    
    rules = generate_rules(trades)
    
    print(f"""
🧠 LEARNED TRADING RULES
{'='*50}
Generated from {len(trades)} trades
{'='*50}
""")
    
    if not rules:
        print("📊 Not enough data to generate confident rules yet.")
        print("   Keep trading! Patterns will emerge with more data.")
        return
    
    # Group by type
    prefer_rules = [r for r in rules if r["type"] == "PREFER"]
    avoid_rules = [r for r in rules if r["type"] == "AVOID"]
    
    if prefer_rules:
        print("✅ PREFER (high win rate patterns):\n")
        for r in prefer_rules:
            conf = "🔥" if r["confidence"] == "HIGH" else "📊"
            print(f"   {conf} {r['rule']}")
            print(f"      └─ {r['evidence']}\n")
    
    if avoid_rules:
        print("🚫 AVOID (low win rate patterns):\n")
        for r in avoid_rules:
            conf = "⚠️" if r["confidence"] == "HIGH" else "📊"
            print(f"   {conf} {r['rule']}")
            print(f"      └─ {r['evidence']}\n")
    
    # Save rules
    rules_file = save_rules(rules)
    print(f"💾 Rules saved to: {rules_file}")


if __name__ == "__main__":
    main()
