#!/usr/bin/env python3
"""
Analyze trade patterns to discover what works and what doesn't.
"""

import argparse
import json
from datetime import datetime
from pathlib import Path
from collections import defaultdict

DATA_DIR = Path(__file__).parent.parent / "data"
TRADES_FILE = DATA_DIR / "trades.json"


def load_trades():
    """Load trades from file."""
    if not TRADES_FILE.exists():
        return []
    with open(TRADES_FILE, "r") as f:
        data = json.load(f)
    return data.get("trades", [])


def calculate_win_rate(trades):
    """Calculate win rate for a list of trades."""
    if not trades:
        return 0, 0
    wins = len([t for t in trades if t["result"] == "WIN"])
    return wins / len(trades) * 100, len(trades)


def analyze_by_field(trades, field, min_trades=3):
    """Analyze win rate grouped by a field."""
    groups = defaultdict(list)
    
    for t in trades:
        value = t.get(field)
        if value is not None:
            groups[value].append(t)
    
    results = []
    for value, group_trades in groups.items():
        if len(group_trades) >= min_trades:
            win_rate, n = calculate_win_rate(group_trades)
            avg_pnl = sum(t["pnl_percent"] for t in group_trades) / n
            results.append({
                "value": value,
                "win_rate": win_rate,
                "n": n,
                "avg_pnl": avg_pnl
            })
    
    return sorted(results, key=lambda x: x["win_rate"], reverse=True)


def analyze_by_indicator(trades, indicator_name, ranges, min_trades=3):
    """Analyze win rate by indicator ranges."""
    results = []
    
    for range_name, (low, high) in ranges.items():
        matching = []
        for t in trades:
            ind_value = t.get("indicators", {}).get(indicator_name)
            if ind_value is not None and low <= ind_value < high:
                matching.append(t)
        
        if len(matching) >= min_trades:
            win_rate, n = calculate_win_rate(matching)
            results.append({
                "range": range_name,
                "win_rate": win_rate,
                "n": n
            })
    
    return results


def generate_insights(trades, min_trades=3):
    """Generate actionable insights from trade data."""
    insights = []
    overall_win_rate, total = calculate_win_rate(trades)
    
    # Analyze by direction
    direction_stats = analyze_by_field(trades, "direction", min_trades)
    for stat in direction_stats:
        diff = stat["win_rate"] - overall_win_rate
        if abs(diff) > 10:  # Significant difference
            emoji = "✅" if diff > 0 else "⚠️"
            action = "PREFER" if diff > 0 else "CAUTION"
            insights.append({
                "type": "direction",
                "emoji": emoji,
                "action": action,
                "message": f"{stat['value']} trades (win rate: {stat['win_rate']:.0f}%, n={stat['n']})",
                "impact": diff
            })
    
    # Analyze by day of week
    day_stats = analyze_by_field(trades, "day_of_week", min_trades)
    for stat in day_stats:
        diff = stat["win_rate"] - overall_win_rate
        if abs(diff) > 15:  # Significant difference
            emoji = "✅" if diff > 0 else "🚫"
            action = "PREFER" if diff > 0 else "AVOID"
            insights.append({
                "type": "day",
                "emoji": emoji,
                "action": action,
                "message": f"Trading on {stat['value'].title()} (win rate: {stat['win_rate']:.0f}%, n={stat['n']})",
                "impact": diff
            })
    
    # Analyze by leverage
    leverage_stats = analyze_by_field(trades, "leverage", min_trades)
    for stat in leverage_stats:
        if stat["value"] and stat["value"] >= 10:
            if stat["win_rate"] < 50:
                insights.append({
                    "type": "leverage",
                    "emoji": "⚠️",
                    "action": "CAUTION",
                    "message": f"High leverage ({stat['value']}x) trades (win rate: {stat['win_rate']:.0f}%, n={stat['n']})",
                    "impact": stat["win_rate"] - overall_win_rate
                })
    
    # Analyze by RSI ranges
    rsi_ranges = {
        "oversold (<30)": (0, 30),
        "neutral (30-70)": (30, 70),
        "overbought (>70)": (70, 100)
    }
    
    for t in trades:
        rsi = t.get("indicators", {}).get("rsi")
        if rsi is not None:
            rsi_stats = analyze_by_indicator(trades, "rsi", rsi_ranges, min_trades)
            for stat in rsi_stats:
                diff = stat["win_rate"] - overall_win_rate
                if abs(diff) > 15:
                    emoji = "✅" if diff > 0 else "🚫"
                    action = "PREFER" if diff > 0 else "AVOID"
                    insights.append({
                        "type": "rsi",
                        "emoji": emoji,
                        "action": action,
                        "message": f"RSI {stat['range']} (win rate: {stat['win_rate']:.0f}%, n={stat['n']})",
                        "impact": diff
                    })
            break  # Only analyze once
    
    # Sort by impact
    insights.sort(key=lambda x: abs(x["impact"]), reverse=True)
    return insights


def main():
    parser = argparse.ArgumentParser(description="Analyze trading patterns")
    parser.add_argument("--symbol", help="Filter by symbol")
    parser.add_argument("--direction", help="Filter by direction")
    parser.add_argument("--min-trades", type=int, default=3, help="Minimum trades for analysis")
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    
    args = parser.parse_args()
    
    trades = load_trades()
    
    if not trades:
        print("❌ No trades logged yet. Use log_trade.py to log your first trade!")
        return
    
    # Apply filters
    if args.symbol:
        trades = [t for t in trades if t["symbol"] == args.symbol.upper()]
    if args.direction:
        trades = [t for t in trades if t["direction"] == args.direction.upper()]
    
    if len(trades) < args.min_trades:
        print(f"⚠️  Not enough trades ({len(trades)}) for meaningful analysis. Need at least {args.min_trades}.")
        return
    
    # Overall stats
    overall_win_rate, total = calculate_win_rate(trades)
    total_pnl = sum(t["pnl_percent"] for t in trades)
    
    print(f"""
🧠 TRADE PATTERN ANALYSIS
{'='*50}

📊 Overall Performance:
   Total Trades: {total}
   Win Rate: {overall_win_rate:.1f}%
   Total PnL: {total_pnl:+.2f}%
""")
    
    # By direction
    print("📈 By Direction:")
    for stat in analyze_by_field(trades, "direction", args.min_trades):
        emoji = "✅" if stat["win_rate"] >= 50 else "❌"
        print(f"   {emoji} {stat['value']}: {stat['win_rate']:.0f}% win rate (n={stat['n']}, avg PnL: {stat['avg_pnl']:+.2f}%)")
    
    # By day
    day_stats = analyze_by_field(trades, "day_of_week", args.min_trades)
    if day_stats:
        print("\n📅 By Day of Week:")
        for stat in day_stats:
            emoji = "✅" if stat["win_rate"] >= 50 else "❌"
            print(f"   {emoji} {stat['value'].title()}: {stat['win_rate']:.0f}% (n={stat['n']})")
    
    # By symbol
    symbol_stats = analyze_by_field(trades, "symbol", args.min_trades)
    if len(symbol_stats) > 1:
        print("\n💰 By Symbol:")
        for stat in symbol_stats:
            emoji = "✅" if stat["win_rate"] >= 50 else "❌"
            print(f"   {emoji} {stat['value']}: {stat['win_rate']:.0f}% (n={stat['n']})")
    
    # Insights
    insights = generate_insights(trades, args.min_trades)
    if insights:
        print(f"\n{'='*50}")
        print("🎯 KEY INSIGHTS:")
        print(f"{'='*50}\n")
        for i in insights[:5]:  # Top 5 insights
            print(f"{i['emoji']} {i['action']}: {i['message']}")
        print()
    
    if args.json:
        output = {
            "overall": {"win_rate": overall_win_rate, "total_trades": total, "total_pnl": total_pnl},
            "by_direction": analyze_by_field(trades, "direction", args.min_trades),
            "by_day": day_stats,
            "insights": insights
        }
        print("\n📄 JSON Output:")
        print(json.dumps(output, indent=2))


if __name__ == "__main__":
    main()
