#!/usr/bin/env python3
"""
Test script for Crypto Levels Analyzer
"""

import sys
import time
from analyze_levels import CryptoLevelsAnalyzer


def test_pair(analyzer, pair):
    """Test a single pair"""
    print(f"\n{'='*60}")
    print(f"Testing: {pair}")
    print(f"{'='*60}")
    
    start_time = time.time()
    analysis = analyzer.analyze(pair)
    end_time = time.time()
    
    if analysis:
        output = analyzer.format_output(analysis)
        print(output)
        print(f"\n⏱️  Analysis time: {end_time - start_time:.2f}s")
        return True
    else:
        print(f"❌ Failed to analyze {pair}")
        return False


def main():
    """Main test function"""
    print("🧪 Crypto Levels Analyzer - Test Suite")
    print("=" * 60)
    
    # Create analyzer
    analyzer = CryptoLevelsAnalyzer(data_source="coingecko")
    
    # Test pairs
    test_pairs = [
        "BTC-USDT",
        "ETH-USDT",
        "SOL-USDT",
        "BNB-USDT",
        "XRP-USDT"
    ]
    
    results = []
    
    for pair in test_pairs:
        success = test_pair(analyzer, pair)
        results.append((pair, success))
        time.sleep(1)  # Rate limiting
    
    # Summary
    print(f"\n{'='*60}")
    print("TEST SUMMARY")
    print(f"{'='*60}")
    
    for pair, success in results:
        status = "✅ PASS" if success else "❌ FAIL"
        print(f"{status}: {pair}")
    
    total = len(results)
    passed = sum(1 for _, s in results if s)
    
    print(f"\nTotal: {passed}/{total} passed ({passed/total*100:.1f}%)")
    
    if passed == total:
        print("\n🎉 All tests passed!")
        sys.exit(0)
    else:
        print(f"\n⚠️  {total - passed} test(s) failed")
        sys.exit(1)


if __name__ == "__main__":
    main()
