#!/usr/bin/env python3
"""
Mock Crypto Levels Analyzer for testing without network access
"""

import random
import sys
from datetime import datetime


class MockCryptoLevelsAnalyzer:
    def __init__(self):
        # Mock price data for common pairs
        self.mock_prices = {
            "BTC": 67500,
            "ETH": 3450,
            "SOL": 175,
            "BNB": 580,
            "XRP": 0.52,
            "ADA": 0.48,
            "DOGE": 0.085,
            "DOT": 7.2,
            "AVAX": 35.5,
            "MATIC": 0.58
        }
    
    def normalize_pair(self, pair: str) -> str:
        """Normalize pair format"""
        pair = pair.replace(" ", "").upper()
        
        if "-" in pair:
            base, quote = pair.split("-")
        elif "/" in pair:
            base, quote = pair.split("/")
        elif pair.endswith("USDT"):
            base = pair[:-4]
            quote = "USDT"
        else:
            base = pair[:-4] if pair.endswith("USDT") else pair
            quote = "USDT"
        
        return base, quote
    
    def get_mock_price(self, symbol: str) -> float:
        """Get mock price with some variation"""
        base_price = self.mock_prices.get(symbol.upper(), 1000)
        # Add 1-5% random variation
        variation = random.uniform(0.95, 1.05)
        return round(base_price * variation, 2)
    
    def calculate_levels(self, current_price: float) -> dict:
        """Calculate mock support/resistance levels"""
        # Support levels (below current price)
        support1 = round(current_price * 0.97, 2)  # -3%
        support2 = round(current_price * 0.95, 2)  # -5%
        support3 = round(current_price * 0.92, 2)  # -8%
        
        # Resistance levels (above current price)
        resistance1 = round(current_price * 1.03, 2)  # +3%
        resistance2 = round(current_price * 1.05, 2)  # +5%
        resistance3 = round(current_price * 1.08, 2)  # +8%
        
        # RSI (random between 30-70)
        rsi = round(random.uniform(35, 65), 1)
        
        # Moving averages
        ma50 = round(current_price * random.uniform(0.98, 1.02), 2)
        ma100 = round(current_price * random.uniform(0.96, 1.04), 2)
        
        # 24h change (random between -5% to +5%)
        change_24h = round(random.uniform(-5, 5), 2)
        
        return {
            "current_price": current_price,
            "change_24h": change_24h,
            "resistance": [resistance1, resistance2, resistance3],
            "support": [support1, support2, support3],
            "rsi": rsi,
            "ma50": ma50,
            "ma100": ma100,
            "recent_high": round(current_price * 1.06, 2),
            "recent_low": round(current_price * 0.94, 2)
        }
    
    def analyze(self, pair: str) -> dict:
        """Main analysis function"""
        base, quote = self.normalize_pair(pair)
        
        if quote != "USDT":
            print(f"⚠️  Only USDT pairs are supported. Using {base}-USDT")
        
        print(f"🔍 Analyzing {base}-USDT (MOCK DATA)...")
        
        # Get mock price
        current_price = self.get_mock_price(base)
        
        # Calculate levels
        analysis = self.calculate_levels(current_price)
        
        analysis["symbol"] = base
        analysis["pair"] = f"{base}-USDT"
        analysis["timestamp"] = datetime.now().isoformat()
        analysis["mock"] = True
        
        return analysis
    
    def format_output(self, analysis: dict) -> str:
        """Format analysis as readable output"""
        if not analysis:
            return "❌ Analysis failed"
        
        symbol = analysis.get("symbol", "Unknown")
        current_price = analysis.get("current_price", 0)
        change_24h = analysis.get("change_24h", 0)
        
        resistance = analysis.get("resistance", [])
        support = analysis.get("support", [])
        
        rsi = analysis.get("rsi")
        ma50 = analysis.get("ma50")
        ma100 = analysis.get("ma100")
        
        # Format change indicator
        change_color = "🟢" if change_24h >= 0 else "🔴"
        change_sign = "+" if change_24h >= 0 else ""
        
        # Build output
        output = []
        output.append(f"📊 {symbol}-USDT 技术分析 (模拟数据)")
        output.append("")
        output.append(f"💰 当前价格: ${current_price:,.2f}")
        output.append(f"📈 24h变化: {change_color} {change_sign}{change_24h:.2f}%")
        output.append("")
        
        # Resistance levels
        if resistance:
            output.append("🔴 压力位 (Resistance):")
            for i, level in enumerate(resistance, 1):
                diff_pct = ((level - current_price) / current_price) * 100
                output.append(f"   • R{i}: ${level:,.2f} (+{diff_pct:.2f}%)")
        else:
            output.append("🔴 压力位: 暂无明显阻力")
        
        output.append("")
        
        # Support levels
        if support:
            output.append("🟢 支撑位 (Support):")
            for i, level in enumerate(support, 1):
                diff_pct = ((current_price - level) / current_price) * 100
                output.append(f"   • S{i}: ${level:,.2f} (-{diff_pct:.2f}%)")
        else:
            output.append("🟢 支撑位: 暂无明显支撑")
        
        output.append("")
        
        # Technical indicators
        output.append("📊 技术指标:")
        if rsi:
            rsi_status = "超买" if rsi > 70 else "超卖" if rsi < 30 else "中性"
            rsi_color = "🔴" if rsi > 70 else "🟢" if rsi < 30 else "🟡"
            output.append(f"   {rsi_color} RSI: {rsi} ({rsi_status})")
        
        if ma50:
            ma50_status = "支撑" if current_price > ma50 else "阻力"
            output.append(f"   📈 MA50: ${ma50:,.2f} ({ma50_status})")
        
        if ma100:
            ma100_status = "支撑" if current_price > ma100 else "阻力"
            output.append(f"   📈 MA100: ${ma100:,.2f} ({ma100_status})")
        
        output.append("")
        
        # Trading insights
        output.append("💡 交易建议:")
        
        if rsi and rsi < 30:
            output.append("   • RSI超卖，可能有反弹机会")
            output.append("   • 关注支撑位附近的买入信号")
        elif rsi and rsi > 70:
            output.append("   • RSI超买，可能有回调风险")
            output.append("   • 关注压力位附近的卖出信号")
        else:
            output.append("   • 市场处于中性区间")
            output.append("   • 建议等待明确突破信号")
        
        # Market sentiment
        if change_24h > 5:
            output.append("   • 短期情绪: 看涨")
        elif change_24h < -5:
            output.append("   • 短期情绪: 看跌")
        else:
            output.append("   • 短期情绪: 中性")
        
        output.append("")
        output.append("⚠️  风险提示: 本分析仅供参考，不构成投资建议。加密货币交易风险极高，请谨慎投资。")
        output.append("📝 注意: 此为模拟数据，仅用于演示功能。实际使用时需要网络连接。")
        
        return "\n".join(output)


def main():
    """Main function"""
    if len(sys.argv) < 2:
        print("Usage: python3 mock_analyzer.py <pair>")
        print("Example: python3 mock_analyzer.py BTC-USDT")
        sys.exit(1)
    
    pair = sys.argv[1]
    
    # Create analyzer
    analyzer = MockCryptoLevelsAnalyzer()
    
    # Analyze
    analysis = analyzer.analyze(pair)
    
    if analysis:
        # Format and print output
        output = analyzer.format_output(analysis)
        print(output)
        
        # Also save as JSON for debugging
        try:
            import json
            with open("/tmp/crypto_analysis_mock.json", "w") as f:
                json.dump(analysis, f, indent=2, ensure_ascii=False)
        except:
            pass
        
        sys.exit(0)
    else:
        print("❌ Analysis failed")
        sys.exit(1)


if __name__ == "__main__":
    main()
