#!/usr/bin/env python3
"""
Crypto Levels Analyzer - Manual Input Mode
用于当网络无法访问时，手动输入价格数据进行分析
"""

import sys
import json
from datetime import datetime
from typing import Dict, List


class ManualCryptoAnalyzer:
    def __init__(self):
        self.mock_prices = {
            "BTC": 67500,
            "ETH": 3450,
            "SOL": 177.70,
            "BNB": 580,
            "XRP": 0.52,
            "ADA": 0.48,
            "DOGE": 0.085,
            "DOT": 7.2,
            "AVAX": 35.5,
            "MATIC": 0.58
        }
    
    def get_manual_price(self, symbol: str) -> float:
        """获取手动输入的价格"""
        symbol = symbol.upper()
        
        # 检查是否有默认价格
        default_price = self.mock_prices.get(symbol, 1000)
        
        print(f"\n💰 输入 {symbol} 的当前价格 (默认: ${default_price}):")
        user_input = input(f"  价格 (USD): ").strip()
        
        if user_input:
            try:
                return float(user_input)
            except ValueError:
                print(f"  ⚠️  无效输入，使用默认价格: ${default_price}")
                return default_price
        else:
            return default_price
    
    def calculate_levels(self, current_price: float) -> dict:
        """计算支撑/压力位"""
        # 支撑位 (低于当前价格)
        support1 = round(current_price * 0.97, 2)  # -3%
        support2 = round(current_price * 0.95, 2)  # -5%
        support3 = round(current_price * 0.92, 2)  # -8%
        
        # 压力位 (高于当前价格)
        resistance1 = round(current_price * 1.03, 2)  # +3%
        resistance2 = round(current_price * 1.05, 2)  # +5%
        resistance3 = round(current_price * 1.08, 2)  # +8%
        
        # 24h变化
        print(f"\n📊 输入技术指标 (可直接回车使用默认值):")
        change_input = input(f"  24h变化 % (默认: 0): ").strip()
        change_24h = float(change_input) if change_input else 0
        
        # RSI (手动输入或默认)
        rsi_input = input(f"  RSI (默认: 55): ").strip()
        rsi = float(rsi_input) if rsi_input else 55
        
        # 移动平均线
        ma50 = round(current_price * 0.98, 2)
        ma100 = round(current_price * 1.02, 2)
        
        return {
            "current_price": current_price,
            "change_24h": change_24h,
            "resistance": [resistance1, resistance2, resistance3],
            "support": [support1, support2, support3],
            "rsi": rsi,
            "ma50": ma50,
            "ma100": ma100,
            "recent_high": round(current_price * 1.06, 2),
            "recent_low": round(current_price * 0.94, 2)
        }
    
    def analyze(self, pair: str) -> dict:
        """主分析函数"""
        # 解析币种
        base = pair.upper().replace("-USDT", "").replace("/USDT", "").replace("USDT", "")
        
        print(f"\n{'='*60}")
        print(f"🔍 手动分析模式: {base}-USDT")
        print(f"{'='*60}")
        
        # 获取价格
        current_price = self.get_manual_price(base)
        
        # 计算水平
        analysis = self.calculate_levels(current_price)
        
        analysis["symbol"] = base
        analysis["pair"] = f"{base}-USDT"
        analysis["timestamp"] = datetime.now().isoformat()
        analysis["mode"] = "manual"
        
        return analysis
    
    def format_output(self, analysis: dict) -> str:
        """格式化输出"""
        symbol = analysis.get("symbol", "Unknown")
        current_price = analysis.get("current_price", 0)
        change_24h = analysis.get("change_24h", 0)
        
        resistance = analysis.get("resistance", [])
        support = analysis.get("support", [])
        
        rsi = analysis.get("rsi")
        ma50 = analysis.get("ma50")
        ma100 = analysis.get("ma100")
        
        # 格式化变化指示
        change_color = "🟢" if change_24h >= 0 else "🔴"
        change_sign = "+" if change_24h >= 0 else ""
        
        # 构建输出
        output = []
        output.append(f"📊 {symbol}-USDT 技术分析")
        output.append("")
        output.append(f"💰 当前价格: ${current_price:,.2f}")
        output.append(f"📈 24h变化: {change_color} {change_sign}{change_24h:.2f}%")
        output.append("")
        
        # 压力位
        if resistance:
            output.append("🔴 压力位 (Resistance):")
            for i, level in enumerate(resistance, 1):
                diff_pct = ((level - current_price) / current_price) * 100
                output.append(f"   • R{i}: ${level:,.2f} (+{diff_pct:.2f}%)")
        else:
            output.append("🔴 压力位: 暂无明显阻力")
        
        output.append("")
        
        # 支撑位
        if support:
            output.append("🟢 支撑位 (Support):")
            for i, level in enumerate(support, 1):
                diff_pct = ((current_price - level) / current_price) * 100
                output.append(f"   • S{i}: ${level:,.2f} (-{diff_pct:.2f}%)")
        else:
            output.append("🟢 支撑位: 暂无明显支撑")
        
        output.append("")
        
        # 技术指标
        output.append("📊 技术指标:")
        if rsi:
            rsi_status = "超买" if rsi > 70 else "超卖" if rsi < 30 else "中性"
            rsi_color = "🔴" if rsi > 70 else "🟢" if rsi < 30 else "🟡"
            output.append(f"   {rsi_color} RSI: {rsi} ({rsi_status})")
        
        if ma50:
            ma50_status = "支撑" if current_price > ma50 else "阻力"
            output.append(f"   📈 MA50: ${ma50:,.2f} ({ma50_status})")
        
        if ma100:
            ma100_status = "支撑" if current_price > ma100 else "阻力"
            output.append(f"   📈 MA100: ${ma100:,.2f} ({ma100_status})")
        
        output.append("")
        
        # 交易建议
        output.append("💡 交易建议:")
        
        if rsi and rsi < 30:
            output.append("   • RSI超卖，可能有反弹机会")
            output.append("   • 关注支撑位附近的买入信号")
        elif rsi and rsi > 70:
            output.append("   • RSI超买，可能有回调风险")
            output.append("   • 关注压力位附近的卖出信号")
        else:
            output.append("   • 市场处于中性区间")
            output.append("   • 建议等待明确突破信号")
        
        # 市场情绪
        if change_24h > 5:
            output.append("   • 短期情绪: 看涨")
        elif change_24h < -5:
            output.append("   • 短期情绪: 看跌")
        else:
            output.append("   • 短期情绪: 中性")
        
        output.append("")
        output.append("⚠️  风险提示: 本分析仅供参考，不构成投资建议。加密货币交易风险极高，请谨慎投资。")
        output.append("📝 注意: 此为手动输入模式，价格数据由用户输入。")
        
        return "\n".join(output)


def main():
    """主函数"""
    if len(sys.argv) < 2:
        print("Usage: python3 analyze_manual.py <pair>")
        print("Example: python3 analyze_manual.py SOL-USDT")
        sys.exit(1)
    
    pair = sys.argv[1]
    
    # 创建分析器
    analyzer = ManualCryptoAnalyzer()
    
    # 分析
    analysis = analyzer.analyze(pair)
    
    if analysis:
        # 格式化并打印输出
        output = analyzer.format_output(analysis)
        print(f"\n{output}")
        
        # 保存为 JSON
        try:
            with open("/tmp/crypto_analysis_manual.json", "w") as f:
                json.dump(analysis, f, indent=2, ensure_ascii=False)
            print(f"\n✅ 分析结果已保存到 /tmp/crypto_analysis_manual.json")
        except:
            pass
        
        sys.exit(0)
    else:
        print("❌ 分析失败")
        sys.exit(1)


if __name__ == "__main__":
    main()
