#!/usr/bin/env python3
"""
Crypto Levels Analyzer
Analyzes support and resistance levels for cryptocurrency pairs
"""

import requests
import json
import sys
import time
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Tuple
import math


class CryptoLevelsAnalyzer:
    def __init__(self, data_source="coingecko"):
        self.data_source = data_source
        self.base_urls = {
            "coingecko": "https://api.coingecko.com/api/v3",
            "binance": "https://api.binance.com/api/v3",
            "coinmarketcap": "https://pro-api.coinmarketcap.com/v1"
        }
        self.base_url = self.base_urls.get(data_source, self.base_urls["coingecko"])
        
    def normalize_pair(self, pair: str) -> str:
        """Normalize pair format"""
        # Remove spaces and convert to uppercase
        pair = pair.replace(" ", "").upper()
        
        # Handle different formats
        if "-" in pair:
            base, quote = pair.split("-")
        elif "/" in pair:
            base, quote = pair.split("/")
        elif pair.endswith("USDT"):
            base = pair[:-4]
            quote = "USDT"
        else:
            # Assume format like BTCUSDT
            base = pair[:-4] if pair.endswith("USDT") else pair
            quote = "USDT"
        
        return base, quote
    
    def get_coin_id(self, symbol: str) -> Optional[str]:
        """Get CoinGecko coin ID from symbol"""
        # Mapping of common symbols to CoinGecko IDs
        symbol_map = {
            "BTC": "bitcoin",
            "ETH": "ethereum",
            "BNB": "binancecoin",
            "SOL": "solana",
            "XRP": "ripple",
            "ADA": "cardano",
            "DOGE": "dogecoin",
            "DOT": "polkadot",
            "AVAX": "avalanche-2",
            "MATIC": "polygon",
            "LINK": "chainlink",
            "UNI": "uniswap",
            "ATOM": "cosmos",
            "LTC": "litecoin",
            "BCH": "bitcoin-cash",
            "XLM": "stellar",
            "SHIB": "shiba-inu",
            "TRX": "tron",
            "ETC": "ethereum-classic",
            "FIL": "filecoin",
            "AAVE": "aave",
            "COMP": "compound-governance-token",
            "MKR": "maker",
            "SNX": "havven",
            "SUSHI": "sushi",
            "YFI": "yearn-finance",
            "CRV": "curve-dao-token",
            "BAL": "balancer",
            "OP": "optimism",
            "ARB": "arbitrum",
            "FET": "fetch-ai",
            "RNDR": "render-token",
            "GRT": "the-graph",
            "NEAR": "near",
            "APT": "aptos",
            "SUI": "sui",
            "TON": "toncoin",
            "INJ": "injective-protocol",
            "XMR": "monero",
            "ZEC": "zcash",
            "DASH": "dash",
            "PEPE": "pepe",
            "BONK": "bonk",
            "WIF": "dogwifhat",
            "FLOKI": "floki",
            "SAND": "the-sandbox",
            "MANA": "decentraland",
            "AXS": "axie-infinity",
            "GALA": "gala",
            "ENJ": "enjin"
        }
        
        return symbol_map.get(symbol.upper())
    
    def fetch_price_data(self, symbol: str, days: int = 30) -> Optional[Dict]:
        """Fetch historical price data"""
        coin_id = self.get_coin_id(symbol)
        
        if not coin_id:
            print(f"❌ Unknown symbol: {symbol}")
            return None
        
        try:
            if self.data_source == "coingecko":
                url = f"{self.base_url}/coins/{coin_id}/market_chart"
                params = {
                    "vs_currency": "usd",
                    "days": days,
                    "interval": "hourly"
                }
                
                response = requests.get(url, params=params, timeout=10)
                
                if response.status_code == 200:
                    data = response.json()
                    return {
                        "prices": data.get("prices", []),
                        "market_caps": data.get("market_caps", []),
                        "total_volumes": data.get("total_volumes", [])
                    }
                else:
                    print(f"❌ API error: {response.status_code}")
                    return None
                    
        except Exception as e:
            print(f"❌ Fetch error: {e}")
            return None
        
        return None
    
    def fetch_current_price(self, symbol: str) -> Optional[float]:
        """Fetch current price"""
        coin_id = self.get_coin_id(symbol)
        
        if not coin_id:
            return None
        
        try:
            if self.data_source == "coingecko":
                url = f"{self.base_url}/simple/price"
                params = {
                    "ids": coin_id,
                    "vs_currencies": "usd"
                }
                
                response = requests.get(url, params=params, timeout=5)
                
                if response.status_code == 200:
                    data = response.json()
                    return data.get(coin_id, {}).get("usd")
                    
        except Exception as e:
            print(f"❌ Price fetch error: {e}")
        
        return None
    
    def calculate_support_resistance(self, price_data: Dict, current_price: float) -> Dict:
        """Calculate support and resistance levels"""
        if not price_data or "prices" not in price_data:
            return {}
        
        prices = [p[1] for p in price_data["prices"]]
        volumes = [v[2] for v in price_data["total_volumes"]] if price_data.get("total_volumes") else [0] * len(prices)
        
        # Calculate recent highs and lows
        recent_prices = prices[-50:]  # Last 50 periods
        
        # Find local maxima and minima
        local_maxima = []
        local_minima = []
        
        for i in range(5, len(recent_prices) - 5):
            if (recent_prices[i] > recent_prices[i-1] and 
                recent_prices[i] > recent_prices[i+1] and
                recent_prices[i] > max(recent_prices[i-5:i]) and
                recent_prices[i] > max(recent_prices[i+1:i+6])):
                local_maxima.append(recent_prices[i])
            
            if (recent_prices[i] < recent_prices[i-1] and 
                recent_prices[i] < recent_prices[i+1] and
                recent_prices[i] < min(recent_prices[i-5:i]) and
                recent_prices[i] < min(recent_prices[i+1:i+6])):
                local_minima.append(recent_prices[i])
        
        # Sort and get unique levels
        local_maxima = sorted(set([round(m, 2) for m in local_maxima]))
        local_minima = sorted(set([round(m, 2) for m in local_minima]))
        
        # Calculate moving averages
        ma50 = self.calculate_ma(prices, 50)
        ma100 = self.calculate_ma(prices, 100)
        ma200 = self.calculate_ma(prices, 200)
        
        # Calculate RSI
        rsi = self.calculate_rsi(prices[-50:])
        
        # Calculate Fibonacci levels
        recent_high = max(recent_prices)
        recent_low = min(recent_prices)
        fib_levels = self.calculate_fibonacci(recent_high, recent_low)
        
        # Determine primary levels
        resistance_levels = []
        support_levels = []
        
        # Resistance: recent highs + MAs
        if local_maxima:
            resistance_levels.extend(local_maxima[-3:])  # Last 3 highs
        
        if ma50 and current_price < ma50:
            resistance_levels.append(round(ma50, 2))
        if ma100 and current_price < ma100:
            resistance_levels.append(round(ma100, 2))
        
        # Add Fibonacci resistance
        if fib_levels:
            resistance_levels.extend([round(r, 2) for r in fib_levels['resistance'][:2]])
        
        # Support: recent lows + MAs
        if local_minima:
            support_levels.extend(local_minima[-3:])  # Last 3 lows
        
        if ma50 and current_price > ma50:
            support_levels.append(round(ma50, 2))
        if ma100 and current_price > ma100:
            support_levels.append(round(ma100, 2))
        
        # Add Fibonacci support
        if fib_levels:
            support_levels.extend([round(s, 2) for s in fib_levels['support'][:2]])
        
        # Remove duplicates and sort
        resistance_levels = sorted(set(resistance_levels))
        support_levels = sorted(set(support_levels))
        
        # Filter levels close to current price
        resistance_levels = [r for r in resistance_levels if r > current_price]
        support_levels = [s for s in support_levels if s < current_price]
        
        # Keep top 3 levels
        resistance_levels = resistance_levels[:3]
        support_levels = support_levels[:3]
        
        # Calculate 24h change
        if len(prices) >= 24:
            change_24h = ((current_price - prices[-24]) / prices[-24]) * 100
        else:
            change_24h = 0
        
        return {
            "current_price": current_price,
            "change_24h": change_24h,
            "resistance": resistance_levels,
            "support": support_levels,
            "rsi": rsi,
            "ma50": round(ma50, 2) if ma50 else None,
            "ma100": round(ma100, 2) if ma100 else None,
            "ma200": round(ma200, 2) if ma200 else None,
            "recent_high": round(recent_high, 2),
            "recent_low": round(recent_low, 2)
        }
    
    def calculate_ma(self, prices: List[float], period: int) -> Optional[float]:
        """Calculate moving average"""
        if len(prices) < period:
            return None
        
        return sum(prices[-period:]) / period
    
    def calculate_rsi(self, prices: List[float], period: int = 14) -> Optional[float]:
        """Calculate RSI"""
        if len(prices) < period + 1:
            return None
        
        deltas = [prices[i] - prices[i-1] for i in range(1, len(prices))]
        
        gains = [d for d in deltas if d > 0]
        losses = [-d for d in deltas if d < 0]
        
        if not gains or not losses:
            return 50
        
        avg_gain = sum(gains[-period:]) / period
        avg_loss = sum(losses[-period:]) / period
        
        if avg_loss == 0:
            return 100
        
        rs = avg_gain / avg_loss
        rsi = 100 - (100 / (1 + rs))
        
        return round(rsi, 2)
    
    def calculate_fibonacci(self, high: float, low: float) -> Dict:
        """Calculate Fibonacci retracement levels"""
        diff = high - low
        
        return {
            "support": [
                high - diff * 0.236,
                high - diff * 0.382,
                high - diff * 0.5,
                high - diff * 0.618
            ],
            "resistance": [
                low + diff * 0.236,
                low + diff * 0.382,
                low + diff * 0.5,
                low + diff * 0.618
            ]
        }
    
    def analyze(self, pair: str) -> Optional[Dict]:
        """Main analysis function"""
        base, quote = self.normalize_pair(pair)
        
        if quote != "USDT":
            print(f"⚠️  Only USDT pairs are supported. Using {base}-USDT")
        
        print(f"🔍 Analyzing {base}-USDT...")
        
        # Fetch current price
        current_price = self.fetch_current_price(base)
        if not current_price:
            print(f"❌ Could not fetch price for {base}")
            return None
        
        # Fetch historical data
        price_data = self.fetch_price_data(base, days=30)
        if not price_data:
            print(f"❌ Could not fetch historical data for {base}")
            return None
        
        # Calculate levels
        analysis = self.calculate_support_resistance(price_data, current_price)
        
        if analysis:
            analysis["symbol"] = base
            analysis["pair"] = f"{base}-USDT"
            analysis["timestamp"] = datetime.now().isoformat()
        
        return analysis
    
    def format_output(self, analysis: Dict) -> str:
        """Format analysis as readable output"""
        if not analysis:
            return "❌ Analysis failed"
        
        symbol = analysis.get("symbol", "Unknown")
        current_price = analysis.get("current_price", 0)
        change_24h = analysis.get("change_24h", 0)
        
        resistance = analysis.get("resistance", [])
        support = analysis.get("support", [])
        
        rsi = analysis.get("rsi")
        ma50 = analysis.get("ma50")
        ma100 = analysis.get("ma100")
        
        # Format change indicator
        change_color = "🟢" if change_24h >= 0 else "🔴"
        change_sign = "+" if change_24h >= 0 else ""
        
        # Build output
        output = []
        output.append(f"📊 {symbol}-USDT 技术分析")
        output.append("")
        output.append(f"💰 当前价格: ${current_price:,.2f}")
        output.append(f"📈 24h变化: {change_color} {change_sign}{change_24h:.2f}%")
        output.append("")
        
        # Resistance levels
        if resistance:
            output.append("🔴 压力位 (Resistance):")
            for i, level in enumerate(resistance, 1):
                diff_pct = ((level - current_price) / current_price) * 100
                output.append(f"   • R{i}: ${level:,.2f} (+{diff_pct:.2f}%)")
        else:
            output.append("🔴 压力位: 暂无明显阻力")
        
        output.append("")
        
        # Support levels
        if support:
            output.append("🟢 支撑位 (Support):")
            for i, level in enumerate(support, 1):
                diff_pct = ((current_price - level) / current_price) * 100
                output.append(f"   • S{i}: ${level:,.2f} (-{diff_pct:.2f}%)")
        else:
            output.append("🟢 支撑位: 暂无明显支撑")
        
        output.append("")
        
        # Technical indicators
        output.append("📊 技术指标:")
        if rsi:
            rsi_status = "超买" if rsi > 70 else "超卖" if rsi < 30 else "中性"
            rsi_color = "🔴" if rsi > 70 else "🟢" if rsi < 30 else "🟡"
            output.append(f"   {rsi_color} RSI: {rsi} ({rsi_status})")
        
        if ma50:
            ma50_status = "支撑" if current_price > ma50 else "阻力"
            output.append(f"   📈 MA50: ${ma50:,.2f} ({ma50_status})")
        
        if ma100:
            ma100_status = "支撑" if current_price > ma100 else "阻力"
            output.append(f"   📈 MA100: ${ma100:,.2f} ({ma100_status})")
        
        output.append("")
        
        # Trading insights
        output.append("💡 交易建议:")
        
        if rsi and rsi < 30:
            output.append("   • RSI超卖，可能有反弹机会")
            output.append("   • 关注支撑位附近的买入信号")
        elif rsi and rsi > 70:
            output.append("   • RSI超买，可能有回调风险")
            output.append("   • 关注压力位附近的卖出信号")
        else:
            output.append("   • 市场处于中性区间")
            output.append("   • 建议等待明确突破信号")
        
        # Market sentiment
        if change_24h > 5:
            output.append("   • 短期情绪: 看涨")
        elif change_24h < -5:
            output.append("   • 短期情绪: 看跌")
        else:
            output.append("   • 短期情绪: 中性")
        
        output.append("")
        output.append("⚠️  风险提示: 本分析仅供参考，不构成投资建议。加密货币交易风险极高，请谨慎投资。")
        
        return "\n".join(output)


def main():
    """Main function"""
    if len(sys.argv) < 2:
        print("Usage: python3 analyze_levels.py <pair>")
        print("Example: python3 analyze_levels.py BTC-USDT")
        sys.exit(1)
    
    pair = sys.argv[1]
    
    # Create analyzer
    analyzer = CryptoLevelsAnalyzer(data_source="coingecko")
    
    # Analyze
    analysis = analyzer.analyze(pair)
    
    if analysis:
        # Format and print output
        output = analyzer.format_output(analysis)
        print(output)
        
        # Also save as JSON for debugging
        try:
            with open("/tmp/crypto_analysis.json", "w") as f:
                json.dump(analysis, f, indent=2, ensure_ascii=False)
        except:
            pass
        
        sys.exit(0)
    else:
        print("❌ Analysis failed")
        sys.exit(1)


if __name__ == "__main__":
    main()
