# Technical Analysis Guide

## Overview

This guide explains the methodology behind support and resistance level calculation in the Crypto Levels skill.

## Core Concepts

### Support and Resistance

#### Support Level
- **Definition**: Price level where buying interest is strong enough to overcome selling pressure
- **Characteristics**:
  - Price tends to bounce up from this level
  - High volume at this price
  - Multiple touches increase significance
- **Psychology**: "This is a good price to buy"

#### Resistance Level
- **Definition**: Price level where selling pressure overcomes buying interest
- **Characteristics**:
  - Price tends to reverse down from this level
  - High volume at this price
  - Multiple touches increase significance
- **Psychology**: "This is a good price to sell"

## Calculation Methods

### 1. Price Action Analysis

#### Recent Highs and Lows
```python
# Primary Support/Resistance
Primary_Support = min(low_prices[-5:])  # Last 5 periods low
Primary_Resistance = max(high_prices[-5:])  # Last 5 periods high
```

#### Multiple Time Frame Analysis
- **Short-term**: Last 5-10 periods
- **Medium-term**: Last 20-50 periods
- **Long-term**: Last 100-200 periods

### 2. Volume Profile

#### Volume at Price Levels
```python
# Calculate volume-weighted price levels
def calculate_volume_levels(prices, volumes, bins=20):
    # Group prices into bins
    # Calculate total volume per bin
    # Return levels with highest volume
    pass
```

#### Volume Confirmation
- **High Volume at Support**: Strong buying interest
- **High Volume at Resistance**: Strong selling pressure
- **Low Volume**: Weak levels, prone to break

### 3. Moving Averages

#### Common Periods
```python
MA_periods = {
    'short': [5, 10, 20],      # Short-term
    'medium': [50, 100],       # Medium-term
    'long': [200]              # Long-term
}
```

#### MA as Dynamic Support/Resistance
- **Price above MA**: MA acts as support
- **Price below MA**: MA acts as resistance
- **MA crossovers**: Signal potential level changes

### 4. Fibonacci Retracement

#### Key Levels
```python
fib_levels = {
    'support': [0.236, 0.382, 0.5, 0.618],
    'resistance': [1.0, 1.236, 1.382, 1.618]
}
```

#### Calculation
```python
def fibonacci_levels(high, low, trend='up'):
    if trend == 'up':
        # Retracement from low to high
        diff = high - low
        return {
            's1': high - diff * 0.236,
            's2': high - diff * 0.382,
            's3': high - diff * 0.5,
            's4': high - diff * 0.618,
            'r1': high + diff * 0.236,
            'r2': high + diff * 0.382
        }
    else:
        # Retracement from high to low
        diff = high - low
        return {
            'r1': low + diff * 0.236,
            'r2': low + diff * 0.382,
            'r3': low + diff * 0.5,
            'r4': low + diff * 0.618,
            's1': low - diff * 0.236,
            's2': low - diff * 0.382
        }
```

### 5. Pivot Points

#### Classic Pivot Points
```python
def classic_pivots(high, low, close):
    pivot = (high + low + close) / 3
    
    r1 = 2 * pivot - low
    s1 = 2 * pivot - high
    
    r2 = pivot + (high - low)
    s2 = pivot - (high - low)
    
    r3 = high + 2 * (pivot - low)
    s3 = low - 2 * (high - pivot)
    
    return {
        'pivot': pivot,
        'resistance': [r1, r2, r3],
        'support': [s1, s2, s3]
    }
```

#### Camarilla Pivots
```python
def camarilla_pivots(high, low, close):
    diff = high - low
    
    return {
        'support': [
            close - diff * 1.1/12,
            close - diff * 1.1/6,
            close - diff * 1.1/4
        ],
        'resistance': [
            close + diff * 1.1/12,
            close + diff * 1.1/6,
            close + diff * 1.1/4
        ]
    }
```

### 6. Trend Line Analysis

#### Drawing Trend Lines
```python
def trend_line_support(prices, lookback=20):
    # Find local minima
    local_minima = find_local_minima(prices, lookback)
    
    # Fit linear regression
    if len(local_minima) >= 2:
        slope, intercept = linear_regression(local_minima)
        return slope, intercept
    return None, None
```

#### Channel Analysis
- **Ascending Channel**: Higher highs and higher lows
- **Descending Channel**: Lower highs and lower lows
- **Horizontal Channel**: Equal highs and lows

## Technical Indicators

### RSI (Relative Strength Index)

#### Calculation
```python
def calculate_rsi(prices, period=14):
    deltas = np.diff(prices)
    gains = np.where(deltas > 0, deltas, 0)
    losses = np.where(deltas < 0, -deltas, 0)
    
    avg_gain = np.mean(gains[-period:])
    avg_loss = np.mean(losses[-period:])
    
    rs = avg_gain / avg_loss
    rsi = 100 - (100 / (1 + rs))
    
    return rsi
```

#### Interpretation
- **RSI > 70**: Overbought (potential resistance)
- **RSI < 30**: Oversold (potential support)
- **RSI 50**: Neutral

### MACD (Moving Average Convergence Divergence)

#### Calculation
```python
def calculate_macd(prices, fast=12, slow=26, signal=9):
    ema_fast = ema(prices, fast)
    ema_slow = ema(prices, slow)
    
    macd_line = ema_fast - ema_slow
    signal_line = ema(macd_line, signal)
    histogram = macd_line - signal_line
    
    return {
        'macd': macd_line,
        'signal': signal_line,
        'histogram': histogram
    }
```

#### Interpretation
- **MACD > Signal**: Bullish (support likely)
- **MACD < Signal**: Bearish (resistance likely)
- **Histogram increasing**: Momentum strengthening

### Bollinger Bands

#### Calculation
```python
def bollinger_bands(prices, period=20, std_dev=2):
    sma = moving_average(prices, period)
    std = np.std(prices[-period:])
    
    upper = sma + (std * std_dev)
    lower = sma - (std * std_dev)
    
    return {
        'upper': upper,
        'middle': sma,
        'lower': lower,
        'width': (upper - lower) / sma
    }
```

#### Interpretation
- **Price at lower band**: Potential support
- **Price at upper band**: Potential resistance
- **Band contraction**: Low volatility (breakout imminent)
- **Band expansion**: High volatility

## Volume Analysis

### Volume Profile

#### Point of Control (POC)
- Price level with highest volume
- Acts as strong support/resistance

#### Value Area
- 70% of volume around POC
- Range where price spends most time

### Volume Indicators

#### On-Balance Volume (OBV)
```python
def calculate_obv(prices, volumes):
    obv = [volumes[0]]
    for i in range(1, len(prices)):
        if prices[i] > prices[i-1]:
            obv.append(obv[-1] + volumes[i])
        elif prices[i] < prices[i-1]:
            obv.append(obv[-1] - volumes[i])
        else:
            obv.append(obv[-1])
    return obv
```

#### Volume Weighted Average Price (VWAP)
```python
def calculate_vwap(prices, volumes):
    total_pv = sum(p * v for p, v in zip(prices, volumes))
    total_v = sum(volumes)
    return total_pv / total_v
```

## Market Structure

### Breakouts and Breakdowns

#### Breakout Confirmation
```python
def is_breakout(price, resistance, volume, threshold=0.02):
    # Price above resistance
    price_above = price > resistance * (1 + threshold)
    
    # Volume confirmation
    volume_confirm = volume > average_volume * 1.5
    
    # Time confirmation (sustained)
    time_confirm = price > resistance for n_periods
    
    return price_above and volume_confirm and time_confirm
```

#### Breakdown Confirmation
```python
def is_breakdown(price, support, volume, threshold=0.02):
    # Price below support
    price_below = price < support * (1 - threshold)
    
    # Volume confirmation
    volume_confirm = volume > average_volume * 1.5
    
    return price_below and volume_confirm
```

### Support/Resistance Role Reversal

#### Principle
- **Broken support becomes resistance**
- **Broken resistance becomes support**

#### Detection
```python
def role_reversal(old_support, current_price, threshold=0.05):
    if current_price < old_support * (1 - threshold):
        # Support broken, now acts as resistance
        return 'resistance'
    elif current_price > old_support * (1 + threshold):
        # Retested and held, still support
        return 'support'
    return 'uncertain'
```

## Time Frame Analysis

### Multi-Time Frame Approach

#### Top-Down Analysis
1. **Monthly/Weekly**: Long-term trend
2. **Daily**: Medium-term structure
3. **4H/1H**: Short-term entry

#### Time Frame Confluence
- **Strong levels**: Multiple time frames align
- **Weak levels**: Only one time frame

### Optimal Time Frames

#### For Day Trading
- **Primary**: 1H, 4H
- **Secondary**: 15M, 1D
- **Confirmation**: 15M + 1H + 4H

#### For Swing Trading
- **Primary**: 4H, 1D
- **Secondary**: 1H, 1W
- **Confirmation**: 4H + 1D + 1W

#### For Position Trading
- **Primary**: 1D, 1W
- **Secondary**: 1M, 4H
- **Confirmation**: 1D + 1W + 1M

## Risk Management

### Position Sizing

#### Kelly Criterion
```python
def kelly_criterion(win_rate, win_loss_ratio):
    # win_rate: probability of winning
    # win_loss_ratio: average win / average loss
    return win_rate - (1 - win_rate) / win_loss_ratio
```

#### Fixed Fractional
```python
def fixed_fractional(account_size, risk_per_trade=0.02):
    return account_size * risk_per_trade
```

### Stop Loss Placement

#### Below Support
```python
def stop_loss_below_support(support, price, atr=None):
    if atr:
        # Use ATR for dynamic stop
        return support - (atr * 1.5)
    else:
        # Fixed percentage
        return support * 0.98
```

#### Trailing Stop
```python
def trailing_stop(current_price, highest_price, trail_percent=0.05):
    return highest_price * (1 - trail_percent)
```

### Take Profit Levels

#### Resistance Based
```python
def take_profit_at_resistance(resistance_levels, risk_reward_ratio=2):
    # Target 2:1 risk/reward
    return [level for level in resistance_levels 
            if level >= entry_price + (risk * risk_reward_ratio)]
```

#### Partial Profits
```python
def partial_profit-taking(levels, position_size):
    # Take profit at multiple levels
    return {
        'level1': position_size * 0.3,  # 30% at first resistance
        'level2': position_size * 0.3,  # 30% at second resistance
        'level3': position_size * 0.4   # 40% at third resistance
    }
```

## Market Psychology

### Support/Resistance Psychology

#### Support Psychology
- "This is a good price to buy"
- "The price won't go lower"
- "Fear of missing out (FOMO) kicks in"

#### Resistance Psychology
- "This is a good price to sell"
- "The price won't go higher"
- "Greed turns to fear"

### Emotional Levels

#### Round Numbers
- **BTC**: $50,000, $60,000, $70,000
- **ETH**: $3,000, $4,000, $5,000
- **Psychology**: Humans prefer round numbers

#### Psychological Levels
- **All-time highs**: Strong resistance
- **Previous lows**: Strong support
- **Fibonacci levels**: Self-fulfilling prophecy

## Advanced Techniques

### Order Flow Analysis

#### Order Book Levels
- **Bid walls**: Large buy orders (support)
- **Ask walls**: Large sell orders (resistance)
- **Iceberg orders**: Hidden large orders

#### Liquidation Levels
- **Long liquidations**: Below support (cascade down)
- **Short liquidations**: Above resistance (cascade up)

### Market Profile

#### TPO (Time-Price Opportunity)
- **Point of Control (POC)**: Highest TPO count
- **Value Area**: 70% of TPOs
- **High/Low**: Range extremes

#### Volume Profile
- **High Volume Nodes (HVN)**: Strong levels
- **Low Volume Nodes (LVN)**: Weak levels (breakthrough easily)

### Elliott Wave Theory

#### Impulsive Waves (5 waves)
- **Wave 1**: New trend
- **Wave 2**: Retracement (support)
- **Wave 3**: Strongest move
- **Wave 4**: Retracement (support)
- **Wave 5**: Final push

#### Corrective Waves (3 waves)
- **Wave A**: Down
- **Wave B**: Retracement (resistance)
- **Wave C**: Final down

## Algorithm Implementation

### Level Ranking System

```python
def rank_levels(levels, weights):
    """
    Rank support/resistance levels by strength
    """
    scores = {}
    
    for level in levels:
        score = 0
        
        # Volume score (0-30)
        score += min(level['volume'] / avg_volume * 10, 30)
        
        # Touch score (0-30)
        score += min(level['touches'] * 5, 30)
        
        # Time frame score (0-20)
        score += level['timeframes'] * 5
        
        # Volume profile score (0-20)
        score += level['volume_profile'] * 20
        
        scores[level['price']] = score
    
    return sorted(scores.items(), key=lambda x: x[1], reverse=True)
```

### Confidence Scoring

```python
def calculate_confidence(level, context):
    """
    Calculate confidence in a level
    """
    confidence = 0
    
    # Multiple time frame confirmation
    if context['timeframe_confluence']:
        confidence += 30
    
    # Volume confirmation
    if context['volume_spike']:
        confidence += 25
    
    # Historical significance
    if context['historical_touches'] >= 3:
        confidence += 20
    
    # Technical indicator alignment
    if context['indicator_alignment']:
        confidence += 15
    
    # Market structure
    if context['market_structure']:
        confidence += 10
    
    return min(confidence, 100)
```

## Limitations and Caveats

### Market Conditions

#### Trending Markets
- **Support/Resistance**: More reliable
- **Breakouts**: More common
- **False breaks**: Less frequent

#### Ranging Markets
- **Support/Resistance**: Very reliable
- **Bounces**: More predictable
- **Range boundaries**: Strong

#### Volatile Markets
- **Support/Resistance**: Less reliable
- **False breaks**: More common
- **Wider stops needed**

### Black Swan Events

#### Unexpected Events
- **Regulatory news**: Can break any level
- **Exchange hacks**: Immediate price impact
- **Market manipulation**: Can invalidate analysis

#### Risk Mitigation
- **Position sizing**: Never risk too much
- **Stop losses**: Always use stops
- **Diversification**: Don't put all eggs in one basket

## Continuous Improvement

### Backtesting

#### Historical Data
```python
def backtest_strategy(levels, historical_data):
    """
    Test strategy on historical data
    """
    results = []
    
    for period in historical_data:
        # Simulate trades
        # Calculate win rate
        # Calculate risk/reward
        pass
    
    return results
```

### Optimization

#### Parameter Tuning
- **Lookback periods**: Test different lengths
- **Volume thresholds**: Optimize for different pairs
- **Time frames**: Find optimal combinations

#### Walk-Forward Testing
- **In-sample**: Optimize parameters
- **Out-of-sample**: Validate performance
- **Rolling window**: Continuous validation

## Resources

### Books
- "Technical Analysis of the Financial Markets" - John Murphy
- "Market Wizards" - Jack Schwager
- "Trading in the Zone" - Mark Douglas

### Online Resources
- Investopedia Technical Analysis
- BabyPips School of Pipsology
- TradingView Chart Patterns

### Tools
- TradingView: Charting and analysis
- CoinGecko: Crypto data
- Binance: Exchange and API

## Disclaimer

**This is educational content only.** Trading involves substantial risk. Past performance does not guarantee future results. Always do your own research and consider professional advice.

---

**Last Updated**: 2026-02-05
