# Crypto Levels Configuration Guide

## Overview

Configure the Crypto Levels skill to use different data sources, update intervals, and analysis parameters.

## Configuration File

### Location
```
/home/openclaw/.openclaw/openclaw.json
```

### Basic Configuration

```json
{
  "crypto-levels": {
    "enabled": true,
    "dataSource": "coingecko",
    "updateInterval": 60,
    "cacheDuration": 300,
    "defaultTimeframe": "4h",
    "apiKey": {
      "coingecko": "",
      "coinmarketcap": "",
      "binance": ""
    }
  }
}
```

## Data Source Configuration

### 1. CoinGecko (Recommended - Free)

**Pros:**
- Free tier available
- 10,000+ cryptocurrencies
- Real-time data
- No API key required for basic use

**Cons:**
- Rate limit: 50 calls/minute
- Limited historical data

**Configuration:**
```json
{
  "crypto-levels": {
    "dataSource": "coingecko",
    "apiKey": {
      "coingecko": ""  // Optional for higher limits
    }
  }
}
```

**API Key (Optional):**
- Sign up at: https://www.coingecko.com/en/api
- Free tier: 50 calls/minute
- Pro tier: 500 calls/minute ($79/month)

### 2. Binance API

**Pros:**
- Real-time exchange data
- High liquidity pairs
- No API key needed for public data
- Very reliable

**Cons:**
- Limited to Binance listed pairs
- No historical data for some pairs

**Configuration:**
```json
{
  "crypto-levels": {
    "dataSource": "binance",
    "apiKey": {
      "binance": ""  // Optional for higher limits
    }
  }
}
```

**API Key (Optional):**
- Create at: https://www.binance.com/en/my/settings/api-management
- Basic: 1200 requests/minute
- IP restrictions recommended

### 3. CoinMarketCap

**Pros:**
- Professional data
- 5,000+ cryptocurrencies
- Good documentation

**Cons:**
- Requires API key
- Free tier limited to 333 calls/day
- More expensive for higher tiers

**Configuration:**
```json
{
  "crypto-levels": {
    "dataSource": "coinmarketcap",
    "apiKey": {
      "coinmarketcap": "YOUR_API_KEY"
    }
  }
}
```

**API Key Required:**
- Sign up at: https://coinmarketcap.com/api/
- Free: 333 calls/day
- Hobbyist: $29/month (10,000 calls/day)
- Professional: $99/month (100,000 calls/day)

## Advanced Configuration

### Update Intervals

#### Real-time (Default)
```json
{
  "updateInterval": 60  // Update every 60 seconds
}
```

#### Fast Updates
```json
{
  "updateInterval": 30  // Update every 30 seconds
}
```

#### Conservative
```json
{
  "updateInterval": 300  // Update every 5 minutes
}
```

### Cache Settings

#### Short Cache (Fast Response)
```json
{
  "cacheDuration": 60  // Cache for 1 minute
}
```

#### Medium Cache (Balanced)
```json
{
  "cacheDuration": 300  // Cache for 5 minutes
}
```

#### Long Cache (Reduced API Calls)
```json
{
  "cacheDuration": 1800  // Cache for 30 minutes
}
```

### Time Frame Settings

#### Default Time Frame
```json
{
  "defaultTimeframe": "4h"  // 4-hour charts
}
```

**Available Time Frames:**
- `1m` - 1 minute (for high-frequency trading)
- `5m` - 5 minutes
- `15m` - 15 minutes
- `1h` - 1 hour
- `4h` - 4 hours (recommended)
- `1d` - 1 day
- `1w` - 1 week

### Analysis Parameters

#### Support/Resistance Calculation
```json
{
  "analysis": {
    "lookbackPeriod": 30,      // Days to analyze
    "volumeThreshold": 0.1,    // Volume multiplier
    "priceDeviation": 0.02,    // 2% deviation
    "minLevels": 3,            // Minimum levels per side
    "maxLevels": 5             // Maximum levels per side
  }
}
```

#### Technical Indicators
```json
{
  "indicators": {
    "rsi": {
      "enabled": true,
      "period": 14
    },
    "macd": {
      "enabled": true,
      "fastPeriod": 12,
      "slowPeriod": 26,
      "signalPeriod": 9
    },
    "movingAverages": {
      "enabled": true,
      "periods": [50, 100, 200]
    },
    "bollingerBands": {
      "enabled": true,
      "period": 20,
      "stdDev": 2
    }
  }
}
```

## Multi-Source Configuration

### Fallback Strategy
```json
{
  "crypto-levels": {
    "dataSource": "coingecko",
    "fallbackSources": ["binance", "coinmarketcap"],
    "maxRetries": 3,
    "timeout": 10
  }
}
```

### Weighted Sources
```json
{
  "crypto-levels": {
    "sources": [
      {
        "name": "coingecko",
        "weight": 0.5,
        "priority": 1
      },
      {
        "name": "binance",
        "weight": 0.3,
        "priority": 2
      },
      {
        "name": "coinmarketcap",
        "weight": 0.2,
        "priority": 3
      }
    ]
  }
}
```

## Environment Variables

### Alternative to JSON Config

```bash
# Data Source
export CRYPTO_LEVELS_DATA_SOURCE="coingecko"

# API Keys
export COINGECKO_API_KEY="your_key"
export COINMARKETCAP_API_KEY="your_key"
export BINANCE_API_KEY="your_key"

# Intervals
export CRYPTO_LEVELS_UPDATE_INTERVAL="60"
export CRYPTO_LEVELS_CACHE_DURATION="300"

# Timeframe
export CRYPTO_LEVELS_DEFAULT_TIMEFRAME="4h"
```

## Performance Optimization

### For High Traffic
```json
{
  "crypto-levels": {
    "dataSource": "binance",
    "updateInterval": 120,
    "cacheDuration": 600,
    "maxConcurrentRequests": 5,
    "requestTimeout": 15
  }
}
```

### For Low Traffic
```json
{
  "crypto-levels": {
    "dataSource": "coingecko",
    "updateInterval": 300,
    "cacheDuration": 1800,
    "maxConcurrentRequests": 2,
    "requestTimeout": 10
  }
}
```

## Security Settings

### API Key Management
```json
{
  "crypto-levels": {
    "security": {
      "encryptKeys": true,
      "keyStorage": "env",  // or "file", "vault"
      "rotateKeys": true,
      "keyRotationDays": 30
    }
  }
}
```

### Rate Limiting
```json
{
  "crypto-levels": {
    "rateLimit": {
      "enabled": true,
      "requestsPerMinute": 50,
      "burstSize": 10,
      "cooldownPeriod": 60
    }
  }
}
```

## Logging Configuration

### Debug Mode
```json
{
  "crypto-levels": {
    "logging": {
      "level": "debug",
      "file": "/var/log/openclaw/crypto-levels.log",
      "maxSize": "10m",
      "maxFiles": 5
    }
  }
}
```

### Production Mode
```json
{
  "crypto-levels": {
    "logging": {
      "level": "info",
      "file": "/var/log/openclaw/crypto-levels.log",
      "maxSize": "5m",
      "maxFiles": 3
    }
  }
}
```

## Error Handling

### Retry Logic
```json
{
  "crypto-levels": {
    "errorHandling": {
      "maxRetries": 3,
      "retryDelay": 1000,
      "backoffMultiplier": 2,
      "circuitBreaker": {
        "enabled": true,
        "failureThreshold": 5,
        "resetTimeout": 60000
      }
    }
  }
}
```

### Fallback Behavior
```json
{
  "crypto-levels": {
    "fallback": {
      "enabled": true,
      "useCachedData": true,
      "cacheOnFailure": true,
      "staleDataThreshold": 300
    }
  }
}
```

## Testing Configuration

### Test Mode
```json
{
  "crypto-levels": {
    "test": {
      "enabled": false,
      "mockData": true,
      "simulateDelay": 500,
      "useTestData": false
    }
  }
}
```

### Development Mode
```json
{
  "crypto-levels": {
    "development": {
      "strictValidation": true,
      "verboseLogging": true,
      "mockExternalAPIs": false
    }
  }
}
```

## Complete Example

### Production Configuration
```json
{
  "crypto-levels": {
    "enabled": true,
    "dataSource": "coingecko",
    "fallbackSources": ["binance"],
    "updateInterval": 60,
    "cacheDuration": 300,
    "defaultTimeframe": "4h",
    "apiKey": {
      "coingecko": "CG-xxxxxxxxxxxx",
      "binance": ""
    },
    "analysis": {
      "lookbackPeriod": 30,
      "volumeThreshold": 0.1,
      "priceDeviation": 0.02,
      "minLevels": 3,
      "maxLevels": 5
    },
    "indicators": {
      "rsi": { "enabled": true, "period": 14 },
      "macd": { "enabled": true },
      "movingAverages": { "enabled": true, "periods": [50, 100, 200] }
    },
    "rateLimit": {
      "enabled": true,
      "requestsPerMinute": 50
    },
    "logging": {
      "level": "info",
      "file": "/var/log/openclaw/crypto-levels.log"
    }
  }
}
```

## Validation

### Test Configuration
```bash
# Validate config syntax
python3 -m json.tool /home/openclaw/.openclaw/openclaw.json

# Test data source connection
python3 crypto-levels/scripts/test_connection.py

# Run diagnostics
python3 crypto-levels/scripts/diagnostics.py
```

## Troubleshooting

### Common Issues

#### "API key required"
- Add API key to configuration
- Check environment variables
- Verify key has correct permissions

#### "Rate limit exceeded"
- Increase update interval
- Use caching
- Consider paid API tier

#### "No data available"
- Check pair format
- Verify data source coverage
- Try different data source

### Debug Mode
```json
{
  "crypto-levels": {
    "logging": {
      "level": "debug"
    }
  }
}
```

## Best Practices

### 1. API Key Security
- Never commit API keys to version control
- Use environment variables in production
- Rotate keys regularly
- Use IP restrictions when available

### 2. Rate Limit Management
- Monitor API usage
- Implement caching
- Use multiple sources
- Respect rate limits

### 3. Error Handling
- Implement retry logic
- Use circuit breakers
- Log errors appropriately
- Provide graceful degradation

### 4. Performance
- Cache aggressively
- Use appropriate update intervals
- Monitor response times
- Optimize queries

## Monitoring

### Metrics to Track
- API response times
- Cache hit rates
- Error rates
- Data freshness
- User query volume

### Alerting
```json
{
  "crypto-levels": {
    "monitoring": {
      "alerts": {
        "enabled": true,
        "email": "admin@example.com",
        "slack": "https://hooks.slack.com/...",
        "thresholds": {
          "errorRate": 0.05,
          "responseTime": 5000,
          "cacheHitRate": 0.7
        }
      }
    }
  }
}
```

## Updates

This configuration guide is regularly updated. Check for new features and best practices.

**Last Updated**: 2026-02-05
