#!/usr/bin/env bash
set -euo pipefail

# generate_report.sh — Compile recent scan results into a markdown report
# Usage: generate_report.sh [--help] [--days N] [--output FILE]

show_help() {
    cat <<'EOF'
Usage: generate_report.sh [OPTIONS]

Generate a markdown security report from recent scan results.

Options:
  --days N       Include scans from last N days (default: 7)
  --output FILE  Write report to file (default: stdout)
  -h, --help     Show this help

Environment:
  CRUSTY_LOG_DIR      Scan log directory (default: /tmp/crusty_logs)
EOF
    exit 0
}

LOG_DIR="${CRUSTY_LOG_DIR:-${CLAWGUARD_LOG_DIR:-/tmp/crusty_logs}}"
DAYS=7
OUTPUT=""

while [[ $# -gt 0 ]]; do
    case "$1" in
        --days) DAYS="$2"; shift 2 ;;
        --output) OUTPUT="$2"; shift 2 ;;
        -h|--help) show_help ;;
        *) shift ;;
    esac
done

TIMESTAMP=$(date -u +"%Y-%m-%d %H:%M UTC")
CUTOFF=$(date -d "-${DAYS} days" +%s 2>/dev/null || date -v-${DAYS}d +%s 2>/dev/null || echo 0)

# Collect scan results
TOTAL_SCANS=0
THREATS_FOUND=0
CLEAN_SCANS=0
ERROR_SCANS=0
THREAT_DETAILS=""

if [[ -d "$LOG_DIR" ]]; then
    for log in "$LOG_DIR"/scan_*.json; do
        [[ -f "$log" ]] || continue
        
        # Check if within date range
        log_ts=$(stat -c %Y "$log" 2>/dev/null || stat -f %m "$log" 2>/dev/null || echo 0)
        [[ "$log_ts" -ge "$CUTOFF" ]] || continue
        
        ((TOTAL_SCANS++)) || true
        
        status=$(python3 -c "import json; d=json.load(open('$log')); print(d.get('status','unknown'))" 2>/dev/null || echo "unknown")
        threats=$(python3 -c "import json; d=json.load(open('$log')); print(d.get('threats_found',0))" 2>/dev/null || echo "0")
        
        case "$status" in
            clean) ((CLEAN_SCANS++)) || true ;;
            infected)
                ((THREATS_FOUND += threats)) || true
                # Extract threat details
                details=$(python3 -c "
import json
d = json.load(open('$log'))
for t in d.get('threats', []):
    print(f\"- **{t.get('threat_name','Unknown')}** in \`{t.get('file','?')}\` (action: {t.get('action','none')})\")
" 2>/dev/null || true)
                THREAT_DETAILS+="$details\n"
                ;;
            *) ((ERROR_SCANS++)) || true ;;
        esac
    done
fi

# Calculate posture score
if [[ $TOTAL_SCANS -eq 0 ]]; then
    POSTURE="N/A (no scans)"
    POSTURE_EMOJI="❓"
elif [[ $THREATS_FOUND -eq 0 ]]; then
    POSTURE="Excellent (100/100)"
    POSTURE_EMOJI="🟢"
elif [[ $THREATS_FOUND -le 2 ]]; then
    POSTURE="Fair (60/100)"
    POSTURE_EMOJI="🟡"
else
    POSTURE="Poor (20/100)"
    POSTURE_EMOJI="🔴"
fi

# Generate report
REPORT="# 🛡️ Crusty Security Report

**Generated:** $TIMESTAMP
**Period:** Last $DAYS days

## Summary

| Metric | Value |
|--------|-------|
| Total Scans | $TOTAL_SCANS |
| Clean | $CLEAN_SCANS |
| Threats Found | $THREATS_FOUND |
| Errors | $ERROR_SCANS |
| Security Posture | $POSTURE_EMOJI $POSTURE |

## Threats

"

if [[ $THREATS_FOUND -gt 0 ]]; then
    REPORT+="$(echo -e "$THREAT_DETAILS")

"
else
    REPORT+="✅ No threats detected in the reporting period.

"
fi

REPORT+="## Recommendations

"

if [[ $TOTAL_SCANS -eq 0 ]]; then
    REPORT+="- ⚠️ No scans have been run. Schedule regular scans.
- Run: \`bash scripts/scan_file.sh -r /data/workspace\`
"
fi

if [[ $THREATS_FOUND -gt 0 ]]; then
    REPORT+="- 🚨 Review and remediate detected threats immediately.
- Check quarantine directory for isolated files.
- Consider running a full system scan.
"
fi

# Check ClamAV
if command -v clamscan &>/dev/null; then
    REPORT+="- ✅ ClamAV is installed.
"
else
    REPORT+="- ⚠️ ClamAV is not installed. Run \`scripts/install_clamav.sh\`.
"
fi

# (VirusTotal removed — ClamAV only)

REPORT+="
---
*Report generated by Crusty Security*"

if [[ -n "$OUTPUT" ]]; then
    echo "$REPORT" > "$OUTPUT"
    echo "Report written to $OUTPUT" >&2
else
    echo "$REPORT"
fi
