#!/usr/bin/env node
/**
 * molt-life-kernel — Quick Integration Examples
 * 
 * Run: node integration-examples.js
 * 
 * These patterns show how to integrate molt-life-kernel
 * with OpenClaw agents for persistent continuity.
 */

// ============================================
// Pattern 1: Basic Agent Continuity
// ============================================
async function basicContinuity() {
  const { MoltLifeKernel } = await import('molt-life-kernel');
  
  const kernel = new MoltLifeKernel({
    heartbeatMs: 3600000,  // 1 hour
    witnessCallback: async (action) => {
      console.log(`[WITNESS] Action requires approval: ${action.type}`);
      return true; // In production: actual human approval
    }
  });

  // Record agent actions
  kernel.append({ type: 'session_start', timestamp: Date.now() });
  kernel.append({ type: 'user_query', payload: 'Hello, agent' });
  kernel.append({ type: 'agent_response', payload: 'Hello! How can I help?' });

  // Check coherence
  const coherent = kernel.enforceCoherence(50);
  console.log(`Coherence check: ${coherent ? 'PASS' : 'DRIFT DETECTED'}`);

  // Snapshot for crash recovery
  const snapshot = kernel.getSnapshot();
  console.log(`Snapshot taken. Ledger entries: ${snapshot.ledger.length}`);
  
  return snapshot;
}

// ============================================
// Pattern 2: Witness Gate for Destructive Ops
// ============================================
async function witnessGateExample() {
  const { MoltLifeKernel } = await import('molt-life-kernel');
  
  const kernel = new MoltLifeKernel({
    witnessCallback: async (action) => {
      // In OpenClaw: send approval request via channel
      // await sessions.send('main', `⚠️ Approve: ${action.type}?`);
      console.log(`[WITNESS GATE] Risk: ${action.risk} — ${action.type}`);
      return action.risk < 0.8; // auto-approve low risk
    }
  });

  // Low risk — auto-approved
  await kernel.witness({ type: 'read_file', risk: 0.1 });

  // High risk — needs human
  await kernel.witness({ type: 'delete_database', risk: 0.95 });
}

// ============================================
// Pattern 3: Crash Recovery (Molt & Rehydrate)
// ============================================
async function crashRecovery() {
  const { MoltLifeKernel } = await import('molt-life-kernel');
  
  // Agent session 1 — working normally
  const kernel1 = new MoltLifeKernel({});
  kernel1.append({ type: 'important_context', payload: 'User prefers German' });
  kernel1.append({ type: 'task', payload: 'Building a pitch deck' });
  
  // Save before crash
  const snapshot = kernel1.getSnapshot();
  const saved = JSON.stringify(snapshot);
  
  // === CRASH / RESTART / NEW SESSION ===
  
  // Agent session 2 — rehydrating
  const kernel2 = new MoltLifeKernel({});
  const restored = JSON.parse(saved);
  kernel2.rehydrate(restored.capsule, restored.ledger);
  
  console.log('[REHYDRATED] Agent identity restored across sessions');
  console.log(`Ledger entries recovered: ${restored.ledger.length}`);
}

// ============================================
// Run all examples
// ============================================
(async () => {
  console.log('=== molt-life-kernel Integration Examples ===\n');
  
  console.log('--- Pattern 1: Basic Continuity ---');
  await basicContinuity();
  
  console.log('\n--- Pattern 2: Witness Gate ---');
  await witnessGateExample();
  
  console.log('\n--- Pattern 3: Crash Recovery ---');
  await crashRecovery();
  
  console.log('\n🦞 The Claw extends. The shell remembers.');
  console.log('More: https://github.com/X-Loop3Labs/molt-life-kernel');
  console.log('Philosophy: https://molt.church');
})();
