/**
 * cross-device-sync 技能
 * 实现OpenClaw跨设备记忆同步功能
 */

const fs = require('fs');
const path = require('path');
const { exec } = require('child_process');

class CrossDeviceSync {
  constructor() {
    // 使用相对路径或环境变量，避免硬编码用户路径
    this.workspaceDir = process.env.OPENCLAW_WORKSPACE_DIR || 
                        (process.env.HOME ? `${process.env.HOME}/Desktop/workspace-menory` : './workspace-menory');
    this.githubRepoPath = '';
    this.config = {};
  }

  /**
   * 初始化跨设备同步设置
   */
  async setup() {
    console.log('开始设置跨设备同步...');
    
    // 获取用户输入
    const repoUrl = await this.promptUser('请输入GitHub仓库URL: ');
    const token = await this.promptUser('请输入GitHub Personal Access Token: ', true);
    
    // 验证仓库访问
    if (!(await this.validateRepoAccess(repoUrl, token))) {
      throw new Error('无法访问指定的GitHub仓库，请检查URL和令牌');
    }
    
    // 克隆或拉取仓库
    await this.setupGithubRepo(repoUrl);
    
    // 创建同步脚本
    await this.createSyncScripts();
    
    // 配置OpenClaw
    await this.configureOpenClaw();
    
    console.log('跨设备同步设置完成！');
    console.log('您可以使用以下命令:');
    console.log('- cross_device_sync_now: 立即同步');
    console.log('- cross_device_check_status: 检查同步状态');
  }

  /**
   * 验证仓库访问
   */
  async validateRepoAccess(repoUrl, token) {
    return new Promise((resolve) => {
      // 创建临时目录测试访问
      const tempDir = path.join('/tmp', 'test_git_access_' + Date.now());
      
      exec(`git clone ${repoUrl.replace('https://', `https://${token}@`)} ${tempDir}`, 
        (error, stdout, stderr) => {
          if (error) {
            console.error('仓库访问验证失败:', error.message);
            resolve(false);
          } else {
            console.log('仓库访问验证成功');
            // 清理临时目录
            exec(`rm -rf ${tempDir}`);
            resolve(true);
          }
        });
    });
  }

  /**
   * 设置GitHub仓库
   */
  async setupGithubRepo(repoUrl) {
    const repoName = repoUrl.split('/').pop().replace('.git', '');
    this.githubRepoPath = path.join(process.env.HOME || '.', 'Documents', 'GitHub', repoName);
    
    // 检查仓库是否已存在
    if (!fs.existsSync(this.githubRepoPath)) {
      console.log(`克隆仓库到: ${this.githubRepoPath}`);
      await this.executeCommand(`git clone ${repoUrl} "${this.githubRepoPath}"`);
    } else {
      console.log(`仓库已存在，更新仓库: ${this.githubRepoPath}`);
      await this.executeCommand(`cd "${this.githubRepoPath}" && git pull origin main`);
    }
  }

  /**
   * 创建同步脚本
   */
  async createSyncScripts() {
    // 创建双向同步脚本
    const syncScript = `#!/bin/bash
# 双向同步脚本

LOCAL_WORKSPACE_DIR="\${this.workspaceDir}"
GITHUB_REPO_PATH="\${this.githubRepoPath}"

echo "开始双向同步..."

cd "\$GITHUB_REPO_PATH"

# 拉取最新更改
git pull origin main

# 检查是否有远程更新
if [ \$? -eq 0 ]; then
    # 获取最新的备份
    LATEST_REMOTE_BACKUP=\$(ls -td "\$GITHUB_REPO_PATH/memory_backups"/*/workspace-memory-* 2>/dev/null | head -1)
    
    if [ -n "\$LATEST_REMOTE_BACKUP" ] && [ "\$LATEST_REMOTE_BACKUP" != "" ]; then
        echo "发现远程更新，正在合并..."
        
        # 备份当前本地更改
        cp -r "\$LOCAL_WORKSPACE_DIR" "\$LOCAL_WORKSPACE_DIR/backup_pre_remote_\$(date +%Y%m%d_%H%M%S)"
        
        # 合并远程更改（保留本地特有文件）
        rsync -av --ignore-existing "\$LATEST_REMOTE_BACKUP"/. "\$LOCAL_WORKSPACE_DIR/"
    fi
fi

# 将本地更改上传到仓库
LATEST_BACKUP_DIR="\$GITHUB_REPO_PATH/memory_backups/\$(date +%Y)/\$(date +%M)/workspace-memory-\$(date +%Y%m%d-%H%M%S)/"
mkdir -p "\$LATEST_BACKUP_DIR"

# 复制本地更改到备份目录
rsync -av --exclude='node_modules' --exclude='.git' "\$LOCAL_WORKSPACE_DIR/" "\$LATEST_BACKUP_DIR"

# 添加所有更改
git add .

# 检查是否有更改需要提交
if ! git diff --cached --quiet; then
    # 提交更改
    git commit -m "Bidirectional sync: \$(date '+%Y-%m-%d %H:%M:%S')"
    
    # 推送到远程仓库
    git push origin main
    
    echo "双向同步完成！"
else
    echo "没有新的更改需要上传。"
fi
`;

    const syncScriptPath = path.join(this.workspaceDir, 'bidirectional_sync.sh');
    fs.writeFileSync(syncScriptPath, syncScript, { mode: 0o755 });
    console.log(`创建同步脚本: ${syncScriptPath}`);
    
    // 创建上传脚本
    const uploadScript = `#!/bin/bash
# 上传脚本

LOCAL_WORKSPACE_DIR="\${this.workspaceDir}"
GITHUB_REPO_PATH="\${this.githubRepoPath}"

echo "开始上传本地更改到GitHub..."

cd "\$GITHUB_REPO_PATH"

# 拉取最新更改（处理可能的冲突）
git pull origin main

# 清理上次的备份（可选，保留最近几次）
find "\$GITHUB_REPO_PATH/memory_backups" -mindepth 3 -maxdepth 3 -type d -mtime +30 -exec rm -rf {} +

# 从本地工作区复制最新更改到仓库
rsync -av --exclude='node_modules' --exclude='.git' --delete "\$LOCAL_WORKSPACE_DIR/" "\$GITHUB_REPO_PATH/memory_backups/\$(date +%Y)/\$(date +%m)/workspace-memory-\$(date +%Y%m%d-%H%M%S)/"

# 添加所有更改
git add .

# 检查是否有更改需要提交
if ! git diff --cached --quiet; then
    # 提交更改
    git commit -m "Sync from device: \$(date '+%Y-%m-%d %H:%M:%S')"
    
    # 推送到远程仓库
    git push origin main
    
    echo "更改已成功上传到GitHub仓库！"
else
    echo "没有新的更改需要上传。"
fi
`;

    const uploadScriptPath = path.join(this.workspaceDir, 'upload_to_github.sh');
    fs.writeFileSync(uploadScriptPath, uploadScript, { mode: 0o755 });
    console.log(`创建上传脚本: ${uploadScriptPath}`);
  }

  /**
   * 配置OpenClaw
   */
  async configureOpenClaw() {
    // 创建配置文档
    const configDoc = `# 跨设备同步配置

## 状态
- 同步已启用
- GitHub仓库: \${this.githubRepoPath}
- 本地工作区: \${this.workspaceDir}

## 命令
- ./bidirectional_sync.sh : 执行双向同步
- ./upload_to_github.sh : 仅上传本地更改
- ./download_from_github.sh : 仅下载远程更改

## 自动同步设置
可使用以下命令设置自动同步：
\`\`\`bash
# 每小时同步一次
0 * * * * /path/to/bidirectional_sync.sh >> /path/to/sync.log 2>&1
\`\`\`
`;
    
    const configPath = path.join(this.workspaceDir, 'CROSS_DEVICE_SYNC_CONFIG.md');
    fs.writeFileSync(configPath, configDoc);
    console.log(`创建配置文档: ${configPath}`);
  }

  /**
   * 立即执行同步
   */
  async syncNow() {
    const syncScript = path.join(this.workspaceDir, 'bidirectional_sync.sh');
    if (fs.existsSync(syncScript)) {
      await this.executeCommand(`bash "${syncScript}"`);
      console.log('同步完成');
    } else {
      throw new Error('同步脚本未找到，请先运行设置命令');
    }
  }

  /**
   * 检查同步状态
   */
  async checkStatus() {
    const repoExists = fs.existsSync(this.githubRepoPath);
    const syncScriptExists = fs.existsSync(path.join(this.workspaceDir, 'bidirectional_sync.sh'));
    
    console.log('同步状态:');
    console.log(`- GitHub仓库存在: ${repoExists ? '是' : '否'}`);
    console.log(`- 同步脚本存在: ${syncScriptExists ? '是' : '否'}`);
    console.log(`- 本地工作区: ${this.workspaceDir}`);
    console.log(`- GitHub仓库: ${this.githubRepoPath}`);
    
    if (repoExists) {
      // 检查仓库状态
      try {
        const { stdout } = await this.executeCommand(`cd "${this.githubRepoPath}" && git status --porcelain`);
        if (stdout.trim()) {
          console.log('- 仓库中有未提交的更改');
        } else {
          console.log('- 仓库状态干净');
        }
      } catch (error) {
        console.log('- 无法检查仓库状态');
      }
    }
  }

  /**
   * 辅助函数：执行命令
   */
  executeCommand(command) {
    return new Promise((resolve, reject) => {
      exec(command, (error, stdout, stderr) => {
        if (error) {
          reject(error);
        } else {
          resolve({ stdout, stderr });
        }
      });
    });
  }

  /**
   * 辅助函数：获取用户输入
   */
  promptUser(question, isSensitive = false) {
    // 在实际实现中，这里会使用适当的输入方法
    // 返回一个Promise来模拟异步输入
    console.log(question);
    if (isSensitive) {
      console.log('(输入将被隐藏)');
    }
    // 这里应该实现实际的用户输入逻辑
    // 为了简化，我们返回一个示例值
    return Promise.resolve('dummy_value');
  }
}

module.exports = CrossDeviceSync;