#!/bin/bash
# setup-cron.sh - Setup cron jobs for automated backups
# Usage: ./setup-cron.sh <type> <source_dir> <backup_dir> <schedule> [extra_args]
# Types: daily, versioned, cleanup

set -e

JOB_TYPE="${1:-}"
SOURCE_DIR="${2:-}"
BACKUP_DIR="${3:-}"
SCHEDULE="${4:-}"
EXTRA_ARGS="${5:-}"

# Validate arguments
if [ -z "$JOB_TYPE" ] || [ -z "$SCHEDULE" ]; then
    echo "Usage: $0 <type> <source_dir> <backup_dir> <schedule> [extra_args]"
    echo ""
    echo "Types:"
    echo "  daily     - Daily directory backup"
    echo "  versioned - Version-triggered backup"
    echo "  cleanup   - Cleanup old backups"
    echo ""
    echo "Schedule format: cron expression (e.g., '0 2 * * *' for daily at 2 AM)"
    echo ""
    echo "Examples:"
    echo "  $0 daily /home/data /backups/data '0 2 * * *'"
    echo "  $0 versioned /opt/app /backups/app '0 */6 * * *'"
    echo "  $0 cleanup /backups/data '' '0 3 * * *' 7"
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
CRON_TAG="# cron-backup-$JOB_TYPE"

# Build command based on type
case "$JOB_TYPE" in
    daily)
        if [ -z "$SOURCE_DIR" ] || [ -z "$BACKUP_DIR" ]; then
            echo "Error: source_dir and backup_dir required for daily backups"
            exit 1
        fi
        BACKUP_NAME=$(basename "$SOURCE_DIR")
        CMD="$SCRIPT_DIR/backup.sh $SOURCE_DIR $BACKUP_DIR $BACKUP_NAME"
        ;;
    versioned)
        if [ -z "$SOURCE_DIR" ] || [ -z "$BACKUP_DIR" ]; then
            echo "Error: source_dir and backup_dir required for versioned backups"
            exit 1
        fi
        # For versioned, SOURCE_DIR is also the version source (command or directory)
        CMD="$SCRIPT_DIR/backup-versioned.sh $SOURCE_DIR $SOURCE_DIR $BACKUP_DIR"
        ;;
    cleanup)
        if [ -z "$SOURCE_DIR" ]; then
            echo "Error: backup_dir required for cleanup"
            exit 1
        fi
        RETENTION_DAYS="${EXTRA_ARGS:-7}"
        CMD="$SCRIPT_DIR/cleanup.sh $SOURCE_DIR $RETENTION_DAYS"
        ;;
    *)
        echo "Error: Unknown job type: $JOB_TYPE"
        exit 1
        ;;
esac

# Build cron entry
CRON_ENTRY="$SCHEDULE $CMD >> $BACKUP_DIR/.backup.log 2>&1 $CRON_TAG"

echo "Setting up cron job..."
echo "  Type: $JOB_TYPE"
echo "  Schedule: $SCHEDULE"
echo "  Command: $CMD"

# Check if job already exists
if crontab -l 2>/dev/null | grep -q "$CRON_TAG"; then
    echo "Warning: A cron job with tag $CRON_TAG already exists"
    echo "Remove it first with: crontab -e"
    exit 1
fi

# Add to crontab
(crontab -l 2>/dev/null || echo "") | grep -v "$CRON_TAG" | { cat; echo "$CRON_ENTRY"; } | crontab -

echo "✓ Cron job added successfully"
echo ""
echo "View all cron jobs: crontab -l"
echo "Remove this job: crontab -e (delete the line with $CRON_TAG)"
