#!/bin/bash
# cleanup.sh - Remove old backups
# Usage: ./cleanup.sh <backup_dir> [retention_days] [min_backups]

set -e

BACKUP_DIR="${1:-}"
RETENTION_DAYS="${2:-7}"
MIN_BACKUPS="${3:-3}"

# Validate arguments
if [ -z "$BACKUP_DIR" ]; then
    echo "Usage: $0 <backup_dir> [retention_days] [min_backups]"
    echo "Example: $0 /backups/data 7 3"
    echo "  (keeps backups for 7 days, minimum 3 backups)"
    exit 1
fi

if [ ! -d "$BACKUP_DIR" ]; then
    echo "Error: Backup directory does not exist: $BACKUP_DIR"
    exit 1
fi

echo "Cleaning up old backups..."
echo "  Directory: $BACKUP_DIR"
echo "  Retention: $RETENTION_DAYS days"
echo "  Minimum backups to keep: $MIN_BACKUPS"

# Count current backups
CURRENT_COUNT=$(find "$BACKUP_DIR" -name "*.tar.gz" -type f | wc -l)
echo "  Current backups: $CURRENT_COUNT"

if [ "$CURRENT_COUNT" -le "$MIN_BACKUPS" ]; then
    echo "✓ Backup count ($CURRENT_COUNT) at or below minimum ($MIN_BACKUPS), skipping cleanup"
    exit 0
fi

# Find and remove old backups
DELETED_COUNT=0
while IFS= read -r file; do
    if [ -n "$file" ]; then
        echo "  Deleting: $(basename "$file")"
        rm -f "$file"
        DELETED_COUNT=$((DELETED_COUNT + 1))
        
        # Check minimum count
        REMAINING=$((CURRENT_COUNT - DELETED_COUNT))
        if [ "$REMAINING" -le "$MIN_BACKUPS" ]; then
            echo "  Reached minimum backup count, stopping cleanup"
            break
        fi
    fi
done < <(find "$BACKUP_DIR" -name "*.tar.gz" -type f -mtime +$RETENTION_DAYS 2>/dev/null | sort)

if [ "$DELETED_COUNT" -eq 0 ]; then
    echo "✓ No old backups to delete"
else
    echo "✓ Deleted $DELETED_COUNT old backup(s)"
fi

# Show remaining backups
REMAINING=$(find "$BACKUP_DIR" -name "*.tar.gz" -type f | wc -l)
echo "  Remaining backups: $REMAINING"
