---
name: cron-backup
description: Set up scheduled automated backups with version tracking and cleanup. Use when users need to (1) Schedule periodic backups of directories or files, (2) Monitor version changes and backup on updates, (3) Automatically clean up old backups to save space, (4) Create backup strategies for configuration files, code repositories, or user data.
---

# Cron Backup

Automated backup scheduling with version detection and intelligent cleanup.

## Quick Start

### One-Time Backup
```bash
# Backup a directory with timestamp
./scripts/backup.sh /path/to/source /path/to/backup/dir

# Backup with custom name
./scripts/backup.sh /path/to/source /path/to/backup/dir my-backup
```

### Schedule Daily Backup
```bash
# Set up daily backup at 2 AM
./scripts/setup-cron.sh daily /path/to/source /path/to/backup/dir "0 2 * * *"
```

### Version-Aware Backup
```bash
# Backup only when version changes
./scripts/backup-versioned.sh /path/to/source /path/to/version/file /path/to/backup/dir
```

### Cleanup Old Backups
```bash
# Keep only last 7 days of backups
./scripts/cleanup.sh /path/to/backup/dir 7
```

## Core Capabilities

### 1. Directory Backup
- Creates timestamped tar.gz archives
- Preserves file permissions and structure
- Excludes common temp files (node_modules, .git, etc.)

### 2. Version-Triggered Backup
- Monitors version file or command output
- Backs up only when version changes
- Useful for software updates

### 3. Scheduled Execution
- Integrates with system cron
- Supports custom schedules
- Logs execution results

### 4. Automatic Cleanup
- Deletes backups older than N days
- Keeps minimum number of backups
- Prevents disk space exhaustion

## Scripts

All scripts are in `scripts/` directory:

- `backup.sh` - Single backup execution
- `backup-versioned.sh` - Version-triggered backup
- `setup-cron.sh` - Cron job setup
- `cleanup.sh` - Old backup cleanup
- `list-backups.sh` - List available backups

## Backup Naming Convention

Backups follow the pattern: `{name}_YYYYMMDD_HHMMSS.tar.gz`

Examples:
- `openclabak_20260204_101500.tar.gz`
- `myapp_20260204_000000.tar.gz`

## Workflow

### Setting Up Automated Backups

1. **Decide backup strategy**
   - What to backup (source directory)
   - Where to store (backup directory)
   - How often (schedule)
   - Retention policy (cleanup days)

2. **Run initial backup**
   ```bash
   ./scripts/backup.sh /source /backup
   ```

3. **Set up schedule**
   ```bash
   ./scripts/setup-cron.sh daily /source /backup "0 2 * * *"
   ```

4. **Configure cleanup**
   ```bash
   ./scripts/setup-cron.sh cleanup /backup "" "0 3 * * *" 7
   ```

### Version-Aware Backup Workflow

For software that changes version (like OpenClaw):

1. **Identify version source**
   - Command: `openclaw --version`
   - File: `/path/to/version.txt`

2. **Set up versioned backup**
   ```bash
   ./scripts/backup-versioned.sh /app /app/version.txt /backups/app
   ```

3. **Schedule version check**
   ```bash
   ./scripts/setup-cron.sh versioned /app /backups/app "0 */6 * * *"
   ```

## Common Patterns

### Pattern 1: Daily User Data Backup
```bash
# Backup workspace daily, keep 30 days
./scripts/setup-cron.sh daily /home/user/workspace /backups/workspace "0 2 * * *"
./scripts/setup-cron.sh cleanup /backups/workspace "" "0 3 * * *" 30
```

### Pattern 2: Version-Aware Application Backup
```bash
# Backup when application updates
./scripts/setup-cron.sh versioned /opt/myapp /backups/myapp "0 */6 * * *"
./scripts/setup-cron.sh cleanup /backups/myapp "" "0 4 * * 0" 10
```

### Pattern 3: Multi-Directory Backup
```bash
# Backup multiple directories
./scripts/backup.sh /home/user/.config /backups/config
./scripts/backup.sh /home/user/projects /backups/projects
```

## Cron Schedule Format

Standard cron format: `minute hour day month weekday`

Common schedules:
- Daily at 2 AM: `0 2 * * *`
- Every 6 hours: `0 */6 * * *`
- Weekly on Sunday: `0 0 * * 0`
- Every 30 minutes: `*/30 * * * *`

## Cleanup Policies

- **Time-based**: Keep backups for N days
- **Count-based**: Keep last N backups
- **Combined**: Default keeps 7 days minimum, but at least 3 backups

## Troubleshooting

- **Permission denied**: Ensure scripts are executable (`chmod +x scripts/*.sh`)
- **Cron not running**: Check cron service status (`systemctl status cron`)
- **Disk full**: Run cleanup manually or reduce retention period
- **Backup fails**: Check source directory exists and is readable
