#!/bin/bash
#
# OpenClaw Cost Tracker
# Track and report OpenClaw usage costs
#

# Constants
OPENCLAW_SESSIONS_DIR=~/.openclaw/agents
SYMLINK_SESSIONS_DIR=~/.clawdbot/agents
OUTPUT_FORMAT="text" # text, json, discord
REPORT_TYPE="daily" # daily, weekly, monthly, custom
START_DATE=""
END_DATE=""
DECIMAL_PLACES=2
CURRENCY_SYMBOL="$"
HIDE_ZERO_COST=true  # Hide zero-cost models
SHOW_ERRORS=false    # Show error status model calls

# Help information
show_help() {
    echo "OpenClaw Cost Tracker - Usage Cost Reporting Tool"
    echo
    echo "Usage: $(basename "$0") [options]"
    echo
    echo "Options:"
    echo "  --today                 Display today's cost report"
    echo "  --yesterday             Display yesterday's cost report"
    echo "  --format FORMAT         Output format (text, json, discord)"
    echo "  --show-all              Show all models (including zero-cost models)"
    echo "  --show-errors           Show error status call information"
    echo "  --help                  Display this help information"
    echo
    echo "Examples:"
    echo "  $(basename "$0") --today"
    echo "  $(basename "$0") --yesterday --format discord"
    echo "  $(basename "$0") --yesterday --show-all --show-errors"
    echo
}

# Parameter parsing
while [[ $# -gt 0 ]]; do
    case "$1" in
        --today)
            REPORT_TYPE="daily"
            TODAY=$(date +%Y-%m-%d)
            START_DATE="$TODAY"
            END_DATE="$TODAY"
            shift
            ;;
        --yesterday)
            REPORT_TYPE="daily"
            YESTERDAY=$(date -v-1d +%Y-%m-%d 2>/dev/null || date -d "yesterday" +%Y-%m-%d)
            START_DATE="$YESTERDAY"
            END_DATE="$YESTERDAY"
            shift
            ;;
        --format)
            shift
            OUTPUT_FORMAT="$1"
            shift
            ;;
        --show-all)
            HIDE_ZERO_COST=false
            shift
            ;;
        --show-errors)
            SHOW_ERRORS=true
            shift
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Error: Unknown option $1"
            show_help
            exit 1
            ;;
    esac
done

# Check if jq is installed
if ! command -v jq &> /dev/null; then
    echo "Error: jq command not found. Please install jq: brew install jq (macOS) or apt install jq (Linux)"
    exit 1
fi

# Validate parameters
if [ -z "$START_DATE" ]; then
    echo "Error: No date specified. Use --today or --yesterday"
    show_help
    exit 1
fi

# Determine which sessions directory to use
if [ -d "$OPENCLAW_SESSIONS_DIR" ]; then
    SESSIONS_DIR="$OPENCLAW_SESSIONS_DIR"
elif [ -d "$SYMLINK_SESSIONS_DIR" ]; then
    SESSIONS_DIR="$SYMLINK_SESSIONS_DIR"
else
    echo "Error: OpenClaw sessions directory not found"
    exit 1
fi

# Get yesterday's date
get_yesterday() {
    if [[ "$(uname)" == "Darwin" ]]; then
        # macOS
        date -v-1d +%Y-%m-%d
    else
        # Linux
        date -d "yesterday" +%Y-%m-%d
    fi
}

# Calculate costs for given date
get_cost_for_date() {
    local target_date="$1"
    local total=0
    local temp_file=$(mktemp)
    local error_file=$(mktemp)
    
    # Collect sessions from all agents
    for agent_dir in "$SESSIONS_DIR"/*; do
        if [ -d "$agent_dir/sessions" ]; then
            for session_file in "$agent_dir/sessions"/*.jsonl; do
                if [ -f "$session_file" ]; then
                    # Extract cost records for specific date and save to temp file
                    grep -a "\"timestamp\":\"$target_date" "$session_file" 2>/dev/null | \
                    grep -a "cost\|totalTokens" | \
                    jq -c 'select(.message.usage != null) | {
                        model: .message.model,
                        timestamp: .timestamp,
                        cost: .message.usage.cost.total,
                        totalTokens: .message.usage.totalTokens,
                        inputTokens: .message.usage.input,
                        outputTokens: .message.usage.output,
                        errorMessage: .message.errorMessage
                    }' 2>/dev/null >> "$temp_file" || true
                    
                    # If showing errors, extract error information
                    if [ "$SHOW_ERRORS" = true ]; then
                        grep -a "\"timestamp\":\"$target_date" "$session_file" 2>/dev/null | \
                        grep -a "errorMessage" | \
                        jq -c 'select(.message.errorMessage != null) | {
                            model: .message.model,
                            timestamp: .timestamp,
                            errorMessage: .message.errorMessage,
                            stopReason: .message.stopReason
                        }' 2>/dev/null >> "$error_file" || true
                    fi
                fi
            done
        fi
    done
    
    # If temp file is empty, return empty JSON
    if [ ! -s "$temp_file" ]; then
        rm "$temp_file" "$error_file"
        echo "{\"total\":0,\"models\":[],\"errors\":[]}"
        return
    fi
    
    # Analyze results and group by model
    local filter_expr=''
    if [ "$HIDE_ZERO_COST" = true ]; then
        filter_expr='| map(select(.cost > 0.001))'
    fi
    
    # Read error file
    local errors_json="[]"
    if [ "$SHOW_ERRORS" = true ] && [ -s "$error_file" ]; then
        errors_json=$(cat "$error_file" | jq -s '
        group_by(.model) | map({
            model: .[0].model,
            count: length,
            errors: map(.errorMessage) | unique
        })' 2>/dev/null || echo "[]")
    fi
    
    # Process cost and token data
    cat "$temp_file" | jq -s "
    {
        total: map(.cost) | add,
        totalTokens: map(.totalTokens) | add,
        models: group_by(.model) | map({
            model: .[0].model,
            cost: map(.cost) | add,
            totalTokens: map(.totalTokens) | add,
            inputTokens: map(.inputTokens) | add,
            outputTokens: map(.outputTokens) | add,
            calls: length
        }) | sort_by(-.cost) $filter_expr,
        errors: $errors_json
    }" 2>/dev/null || echo "{\"total\":0,\"totalTokens\":0,\"models\":[],\"errors\":[]}"
    
    rm "$temp_file" "$error_file"
}

# Generate report
generate_report() {
    # Get current date data
    local current_data=$(get_cost_for_date "$START_DATE")
    local total=$(echo "$current_data" | jq -r '.total')
    local totalTokens=$(echo "$current_data" | jq -r '.totalTokens')
    local models=$(echo "$current_data" | jq -r '.models')
    local errors=$(echo "$current_data" | jq -r '.errors')
    
    # If today's report, get yesterday's data for comparison
    local comparison_data=""
    local comparison_total=0
    local percentage_change=0
    local change_indicator=""
    
    if [ "$START_DATE" = "$(date +%Y-%m-%d)" ]; then
        local yesterday=$(get_yesterday)
        comparison_data=$(get_cost_for_date "$yesterday")
        comparison_total=$(echo "$comparison_data" | jq -r '.total')
        
        # Calculate percentage change
        if (( $(echo "$comparison_total > 0" | bc -l 2>/dev/null || echo 0) )); then
            percentage_change=$(echo "scale=1; ($total - $comparison_total) / $comparison_total * 100" | bc -l 2>/dev/null || echo 0)
            
            # Determine increase/decrease indicator
            if (( $(echo "$percentage_change > 0" | bc -l 2>/dev/null || echo 0) )); then
                if [ "$OUTPUT_FORMAT" = "discord" ]; then
                    change_indicator="🔴" # Increase (Chinese style: red=up)
                else
                    change_indicator="▲"
                fi
            elif (( $(echo "$percentage_change < 0" | bc -l 2>/dev/null || echo 0) )); then
                if [ "$OUTPUT_FORMAT" = "discord" ]; then
                    change_indicator="🟢" # Decrease (Chinese style: green=down)
                else
                    change_indicator="▼"
                fi
            else
                if [ "$OUTPUT_FORMAT" = "discord" ]; then
                    change_indicator="⚪" # No change
                else
                    change_indicator="○"
                fi
            fi
        fi
    fi
    
    # Generate report based on output format
    if [ "$OUTPUT_FORMAT" = "json" ]; then
        # JSON output
        echo "{
            \"total\": $total,
            \"totalTokens\": $totalTokens,
            \"models\": $models,
            \"errors\": $errors,
            \"comparisonTotal\": $comparison_total,
            \"percentageChange\": $percentage_change,
            \"date\": \"$START_DATE\"
        }"
    elif [ "$OUTPUT_FORMAT" = "discord" ]; then
        # Discord format output
        local report_title=""
        
        if [ "$START_DATE" = "$(date +%Y-%m-%d)" ]; then
            report_title="💰 OpenClaw Today's Cost Report"
        elif [ "$START_DATE" = "$(get_yesterday)" ]; then
            report_title="💰 OpenClaw Yesterday's Cost Report"
        else
            report_title="💰 OpenClaw Cost Report"
        fi
        
        echo "$report_title ($START_DATE)"
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
        
        # Format total cost
        local formatted_total=$(printf "$CURRENCY_SYMBOL%.${DECIMAL_PLACES}f" "$total")
        
        if [ -n "$comparison_data" ] && [ "$comparison_total" != "0" ]; then
            echo "Total Cost: $formatted_total ($change_indicator ${percentage_change}% vs yesterday) | Total Tokens: $(printf "%'d" $totalTokens)"
        else
            echo "Total Cost: $formatted_total | Total Tokens: $(printf "%'d" $totalTokens)"
        fi
        
        # Model details
        echo -e "\n📊 Model Details:"
        
        # Calculate model count
        local model_count=$(echo "$models" | jq -r '. | length')
        
        if [ "$model_count" -eq 0 ]; then
            echo "• No cost data available"
        else
            echo "$models" | jq -c '.[]' | while read -r model_data; do
                local model_name=$(echo "$model_data" | jq -r '.model')
                local model_cost=$(echo "$model_data" | jq -r '.cost')
                local model_tokens=$(echo "$model_data" | jq -r '.totalTokens')
                local input_tokens=$(echo "$model_data" | jq -r '.inputTokens')
                local output_tokens=$(echo "$model_data" | jq -r '.outputTokens')
                local calls=$(echo "$model_data" | jq -r '.calls')
                
                local formatted_cost=$(printf "$CURRENCY_SYMBOL%.${DECIMAL_PLACES}f" "$model_cost")
                local percentage=0
                
                if (( $(echo "$total > 0" | bc -l 2>/dev/null || echo 0) )); then
                    percentage=$(echo "scale=1; ($model_cost / $total) * 100" | bc -l 2>/dev/null || echo 0)
                fi
                
                echo "• $model_name: $formatted_cost (${percentage}%) | Tokens: $(printf "%'d" $model_tokens) [in:$(printf "%'d" $input_tokens)/out:$(printf "%'d" $output_tokens)] | Calls: $calls"
            done
        fi
        
        # Show error information
        if [ "$SHOW_ERRORS" = true ]; then
            local error_count=$(echo "$errors" | jq -r '. | length')
            
            if [ "$error_count" -gt 0 ]; then
                echo -e "\n⚠️ Model Errors:"
                echo "$errors" | jq -c '.[]' | while read -r error_data; do
                    local model_name=$(echo "$error_data" | jq -r '.model')
                    local count=$(echo "$error_data" | jq -r '.count')
                    local error_msgs=$(echo "$error_data" | jq -r '.errors | join(", ")')
                    
                    echo "• $model_name: $count errors - $error_msgs"
                done
            fi
        fi
    else
        # Default text output
        local report_title=""
        
        if [ "$START_DATE" = "$(date +%Y-%m-%d)" ]; then
            report_title="OpenClaw Today's Cost Report"
        elif [ "$START_DATE" = "$(get_yesterday)" ]; then
            report_title="OpenClaw Yesterday's Cost Report"
        else
            report_title="OpenClaw Cost Report"
        fi
        
        echo "$report_title ($START_DATE)"
        echo "-------------------------------"
        
        # Format total cost
        local formatted_total=$(printf "$CURRENCY_SYMBOL%.${DECIMAL_PLACES}f" "$total")
        
        if [ -n "$comparison_data" ] && [ "$comparison_total" != "0" ]; then
            echo "Total Cost: $formatted_total ($change_indicator ${percentage_change}% vs yesterday)"
            echo "Total Tokens: $(printf "%'d" $totalTokens)"
        else
            echo "Total Cost: $formatted_total"
            echo "Total Tokens: $(printf "%'d" $totalTokens)"
        fi
        
        # Model details
        echo -e "\nModel Details:"
        
        # Calculate model count
        local model_count=$(echo "$models" | jq -r '. | length')
        
        if [ "$model_count" -eq 0 ]; then
            echo "  No cost data available"
        else
            echo "$models" | jq -c '.[]' | while read -r model_data; do
                local model_name=$(echo "$model_data" | jq -r '.model')
                local model_cost=$(echo "$model_data" | jq -r '.cost')
                local model_tokens=$(echo "$model_data" | jq -r '.totalTokens')
                local input_tokens=$(echo "$model_data" | jq -r '.inputTokens')
                local output_tokens=$(echo "$model_data" | jq -r '.outputTokens')
                local calls=$(echo "$model_data" | jq -r '.calls')
                
                local formatted_cost=$(printf "$CURRENCY_SYMBOL%.${DECIMAL_PLACES}f" "$model_cost")
                local percentage=0
                
                if (( $(echo "$total > 0" | bc -l 2>/dev/null || echo 0) )); then
                    percentage=$(echo "scale=1; ($model_cost / $total) * 100" | bc -l 2>/dev/null || echo 0)
                fi
                
                echo "  $model_name:"
                echo "    Cost: $formatted_cost (${percentage}%)"
                echo "    Tokens: $(printf "%'d" $model_tokens) [input:$(printf "%'d" $input_tokens)/output:$(printf "%'d" $output_tokens)]"
                echo "    Calls: $calls"
            done
        fi
        
        # Show error information
        if [ "$SHOW_ERRORS" = true ]; then
            local error_count=$(echo "$errors" | jq -r '. | length')
            
            if [ "$error_count" -gt 0 ]; then
                echo -e "\nModel Errors:"
                echo "$errors" | jq -c '.[]' | while read -r error_data; do
                    local model_name=$(echo "$error_data" | jq -r '.model')
                    local count=$(echo "$error_data" | jq -r '.count')
                    local error_msgs=$(echo "$error_data" | jq -r '.errors | join(", ")')
                    
                    echo "  $model_name: $count errors"
                    echo "    Error messages: $error_msgs"
                done
            fi
        fi
    fi
}

# Main function
main() {
    # Generate report
    generate_report
}

main