#!/usr/bin/env node

/**
 * OpenClaw Cost Tracker CLI
 * 
 * Provides a command line interface for calling the cost_report.sh script
 */

const { execSync } = require('child_process');
const { resolve, join } = require('path');
const { existsSync } = require('fs');

// Get script directory
const baseDir = __dirname;
const scriptPath = join(baseDir, 'scripts', 'cost_report.sh');

// Ensure script exists and is executable
if (!existsSync(scriptPath)) {
  console.error(`Error: Script not found at ${scriptPath}`);
  process.exit(1);
}

// Set script executable permissions
try {
  execSync(`chmod +x "${scriptPath}"`);
} catch (error) {
  console.warn(`Warning: Unable to set script execution permissions: ${error.message}`);
}

// Pass command line arguments to the script
const args = process.argv.slice(2);

try {
  // Set environment variables for the script
  const env = {
    ...process.env,
    LC_ALL: 'en_US.UTF-8', // Ensure English locale for number formatting
  };
  
  // Run script and capture output
  const output = execSync(`"${scriptPath}" ${args.map(arg => `"${arg}"`).join(' ')}`, {
    env,
    stdio: 'pipe',
    encoding: 'utf-8'
  });
  
  // Output the result
  console.log(output.toString());
} catch (error) {
  // Handle script execution errors
  if (error.stdout) {
    console.log(error.stdout.toString());
  }
  if (error.stderr) {
    console.error(error.stderr.toString());
  }
  process.exit(error.status || 1);
}