#!/usr/bin/env python3
"""
Verbose light show for troubleshooting. Prints each action and errors.
"""
import argparse
import asyncio
import colorsys
import time

from kasa import Discover


def parse_args():
    p = argparse.ArgumentParser()
    p.add_argument('--ip', required=True)
    p.add_argument('--duration', type=float, default=6.0)
    p.add_argument('--transition', type=float, default=0.5)
    p.add_argument('--instant', action='store_true')
    p.add_argument('--double-write', action='store_true')
    return p.parse_args()


async def find_device(ip):
    print(f"[{time.time()}] discover_single({ip})")
    dev = await Discover.discover_single(ip)
    print(f"[{time.time()}] discover result: {dev}")
    if dev is None:
        raise RuntimeError('Device not found')
    try:
        print(f"[{time.time()}] calling update()")
        await dev.update()
        print(f"[{time.time()}] update done")
    except Exception as e:
        print(f"[{time.time()}] update failed: {e}; trying connect()")
        try:
            await dev.connect()
            await dev.update()
        except Exception as e:
            print(f"[{time.time()}] connect/update failed: {e}")
            raise
    return dev


async def set_rgb(dev, r, g, b):
    print(f"[{time.time()}] set_rgb -> {(r,g,b)}")
    light_module = None
    for key in getattr(dev, 'modules', {}):
        m = dev.modules[key]
        try:
            name = getattr(m, '__class__').__name__
            if 'Light' in name or 'light' in name.lower():
                light_module = m
                break
        except Exception:
            continue
    try:
        if light_module and hasattr(light_module, 'set_rgb'):
            await light_module.set_rgb(r,g,b)
            print(f"[{time.time()}] set_rgb via module OK")
            return
        if hasattr(dev, 'set_light_state'):
            await dev.set_light_state(rgb=(r,g,b))
            print(f"[{time.time()}] set_light_state OK")
            return
        print(f"[{time.time()}] no rgb method available")
    except Exception as e:
        print(f"[{time.time()}] exception in set_rgb: {e}")


async def run_verbose(ip, duration, transition, instant, double_write):
    print(f"[{time.time()}] start run_verbose ip={ip} dur={duration} trans={transition} instant={instant} double={double_write}")
    dev = await find_device(ip)
    colors = [(0,100,100),(0,0,100),(240,100,100)]
    start = asyncio.get_event_loop().time()
    idx = 0
    while asyncio.get_event_loop().time() - start < duration:
        h,s,v = colors[idx % len(colors)]
        r,g,b = [int(x*255) for x in colorsys.hsv_to_rgb(h/360.0, s/100.0, v/100.0)]
        if instant:
            await set_rgb(dev, r,g,b)
            if double_write:
                await asyncio.sleep(0.05)
                await set_rgb(dev, r,g,b)
        else:
            print(f"[{time.time()}] set_hsv {(h,s,v)}")
            try:
                await dev.set_light_state(hue=h, saturation=s, brightness=v)
                print(f"[{time.time()}] set_light_state hsv OK")
            except Exception as e:
                print(f"[{time.time()}] hsv set failed: {e}")
        await asyncio.sleep(transition)
        idx += 1
    print(f"[{time.time()}] run complete")


if __name__ == '__main__':
    args = parse_args()
    asyncio.run(run_verbose(args.ip, args.duration, args.transition, args.instant, args.double_write))
