#!/usr/bin/env python3
"""
Light show: cycle specified colors for a duration using python-kasa.
Saves initial bulb state and restores it at the end, optionally fading over N seconds.
Usage: uv run --project . python ./scripts/light_show.py --ip 192.168.4.69 --duration 15 --off-flash --fade 3
"""
import argparse
import asyncio
import colorsys
import time
import math

from kasa import Discover

SPEEDUP = 5.0
DEFAULT_TRANSITION = 1.0 / SPEEDUP
INSTANT_SLEEP = 0.3 / SPEEDUP
OFF_FLASH_SLEEP = 0.3 / SPEEDUP
OFF_FLASH_SETTLE = 0.05 / SPEEDUP


def parse_args():
    p = argparse.ArgumentParser()
    p.add_argument('--ip', required=True)
    p.add_argument('--duration', type=float, default=15.0)
    p.add_argument('--transition', type=float, default=DEFAULT_TRANSITION)
    p.add_argument('--instant', action='store_true', help='Force immediate RGB steps (no transition)')
    p.add_argument('--double-write', action='store_true', help='Write each color twice quickly to avoid smoothing')
    p.add_argument('--off-flash', action='store_true', help='Set brightness=0 briefly between blue->red to avoid magenta')
    p.add_argument('--verbose', action='store_true')
    p.add_argument('--white-temp', type=int, default=9000, help='If set (>0), use color temperature (Kelvin) for the white step (default: 9000 to make white "whiter" in shows)')
    p.add_argument('--fade', type=float, default=0.0, help='If >0, fade back to saved state over N seconds')
    p.add_argument('--colors', type=str, default='0,100,100;0,0,100;240,100,100',
                   help='Semicolon-separated HSV triples H,S,V (H:0-360,S:0-100,V:0-100)')
    return p.parse_args()



def log(*a, **k):
    if not k.pop('force', False):
        if not LOG_VERBOSE:
            return
    print(f"[{time.time()}]", *a, **k)

def get_light_module(dev):
    modules = getattr(dev, 'modules', {}) or {}
    for key in ("Light", "light", "light-1"):
        if key in modules:
            return modules[key]
    for m in modules.values():
        try:
            name = getattr(m, '__class__', type(m)).__name__
            if 'Light' in name or 'light' in name.lower():
                return m
        except Exception:
            continue
    return None


def parse_colors(s: str):
    parts = [p.strip() for p in s.split(';') if p.strip()]
    out = []
    for part in parts:
        nums = [int(x) for x in part.split(',')]
        if len(nums) != 3:
            raise ValueError(f'Bad color spec: {part}')
        h = max(0, min(360, nums[0]))
        sat = max(0, min(100, nums[1]))
        val = max(0, min(100, nums[2]))
        out.append((h, sat, val))
    return out


async def find_device(ip):
    log(f"discover_single({ip})")
    dev = await Discover.discover_single(ip)
    log(f"discover result: {dev}")
    if dev is None:
        raise RuntimeError('Device not found')
    try:
        log('calling update()')
        await dev.update()
        log('update done')
    except Exception as e:
        log('update failed:', e, 'trying connect()')
        try:
            await dev.connect()
            await dev.update()
        except Exception as e:
            raise RuntimeError(f"Failed to prepare device: {e}")
    return dev


def hsv_to_rgb_int(h, s, v):
    r, g, b = colorsys.hsv_to_rgb(h/360.0, s/100.0, v/100.0)
    return (int(r*255), int(g*255), int(b*255))


async def get_current_state(dev):
    # Try to read useful properties: on/off, brightness, hsv, color_temp
    state = {'on': None, 'brightness': None, 'hsv': None, 'color_temp': None}
    try:
        # many devices expose .is_on and .brightness or module state
        if hasattr(dev, 'is_on'):
            state['on'] = dev.is_on
        if hasattr(dev, 'brightness'):
            state['brightness'] = dev.brightness
    except Exception:
        pass
    # check Light module
    light = get_light_module(dev)
    if light:
        try:
            if hasattr(light, 'hsv'):
                hsv = getattr(light, 'hsv', None)
                if hsv is not None:
                    state['hsv'] = hsv
            if hasattr(light, 'color_temp'):
                color_temp = getattr(light, 'color_temp', None)
                if color_temp is not None:
                    state['color_temp'] = color_temp
            if hasattr(light, 'brightness'):
                brightness = getattr(light, 'brightness', None)
                if brightness is not None:
                    state['brightness'] = brightness
            # some modules have is_supported
        except Exception:
            pass
    log('saved state:', state, force=True)
    return state


async def set_state_from_saved(dev, state, instant, double_write):
    # Restore on/off and color. Prefer module methods.
    light = get_light_module(dev)
    if state.get('hsv') is not None:
        h, s, v = state['hsv']
        try:
            if light and hasattr(light, 'set_hsv'):
                await light.set_hsv(h, s, v)
                if double_write:
                    await asyncio.sleep(0.05)
                    await light.set_hsv(h, s, v)
            elif hasattr(dev, 'set_light_state'):
                await dev.set_light_state(hue=h, saturation=s, brightness=v)
        except Exception as e:
            log('restore hsv failed:', e, force=True)
    elif state.get('color_temp') is not None:
        try:
            if light and hasattr(light, 'set_color_temp'):
                await light.set_color_temp(state['color_temp'])
            elif hasattr(dev, 'set_light_state'):
                await dev.set_light_state(color_temp=state['color_temp'])
        except Exception as e:
            log('restore color_temp failed:', e, force=True)
    # brightness
    if state.get('brightness') is not None:
        b = state['brightness']
        try:
            if light and hasattr(light, 'set_brightness'):
                await light.set_brightness(b)
            elif hasattr(dev, 'set_light_state'):
                await dev.set_light_state(brightness=b)
        except Exception as e:
            log('restore brightness failed:', e, force=True)
    # on/off
    if state.get('on') is not None:
        try:
            if state['on']:
                if hasattr(dev, 'turn_on'):
                    await dev.turn_on()
            else:
                if hasattr(dev, 'turn_off'):
                    await dev.turn_off()
        except Exception as e:
            log('restore on/off failed:', e, force=True)


async def fade_restore(dev, start_state, duration, steps=20, double_write=False):
    # Fade from current to saved state over duration seconds
    if duration <= 0:
        await set_state_from_saved(dev, start_state, instant=True, double_write=double_write)
        return
    # Read current values
    light = get_light_module(dev)
    cur_hsv = None
    cur_b = None
    if light:
        try:
            cur_hsv = getattr(light, 'hsv', None)
            cur_b = getattr(light, 'brightness', None)
        except Exception:
            pass
    if cur_b is None:
        cur_b = getattr(dev, 'brightness', None) or 0
    target_b = start_state.get('brightness') if start_state.get('brightness') is not None else 100
    # simple linear fade of brightness; color snap to target then fade brightness
    step_time = duration / steps
    for i in range(steps):
        t = (i+1)/steps
        b = int(cur_b + (target_b - cur_b) * t)
        try:
            if light and hasattr(light, 'set_brightness'):
                await light.set_brightness(b)
            elif hasattr(dev, 'set_light_state'):
                await dev.set_light_state(brightness=b)
        except Exception as e:
            log('fade step failed:', e, force=True)
        await asyncio.sleep(step_time)
    # finally set full saved state
    await set_state_from_saved(dev, start_state, instant=True, double_write=double_write)


async def set_via_light_module(dev, h, s, v):
    light = get_light_module(dev)
    if not light:
        return False
    try:
        if hasattr(light, 'set_hsv'):
            await light.set_hsv(h, s, v)
            return True
        if hasattr(light, 'set_state'):
            await light.set_state(hue=h, saturation=s, brightness=v)
            return True
    except Exception as e:
        log('light module set failed:', e)
    return False


async def set_brightness_via_module(dev, b):
    light = get_light_module(dev)
    try:
        if light and hasattr(light, 'set_brightness'):
            await light.set_brightness(b)
            return True
        if hasattr(dev, 'set_light_state'):
            await dev.set_light_state(brightness=b)
            return True
        if hasattr(dev, 'turn_off') and b==0:
            await dev.turn_off()
            return True
        if hasattr(dev, 'turn_on') and b>0:
            await dev.turn_on()
            return True
    except Exception as e:
        log('set_brightness failed:', e)
    return False


async def restore_color_with_brightness(dev, h, s, v, double_write, white_temp):
    light = get_light_module(dev)
    try:
        if white_temp and s == 0 and light and hasattr(light, 'set_color_temp'):
            await light.set_color_temp(white_temp)
            if hasattr(light, 'set_brightness'):
                await light.set_brightness(v)
            if double_write:
                await asyncio.sleep(0.05)
                await light.set_color_temp(white_temp)
                if hasattr(light, 'set_brightness'):
                    await light.set_brightness(v)
            return True
        if light and hasattr(light, 'set_hsv'):
            await light.set_hsv(h, s, v)
            if double_write:
                await asyncio.sleep(0.05)
                await light.set_hsv(h, s, v)
            if hasattr(light, 'set_brightness'):
                await light.set_brightness(v)
            return True
        if hasattr(dev, 'set_light_state'):
            r,g,b = hsv_to_rgb_int(h,s,v)
            await dev.set_light_state(hue=h, saturation=s, brightness=v, rgb=(r,g,b))
            if double_write:
                await asyncio.sleep(0.05)
                await dev.set_light_state(hue=h, saturation=s, brightness=v, rgb=(r,g,b))
            return True
    except Exception as e:
        log('restore failed:', e)
    return False


async def apply_color(dev, h, s, v, instant, double_write, white_temp):
    light = get_light_module(dev)
    if white_temp and s == 0:
        if light and hasattr(light, 'set_color_temp'):
            await light.set_color_temp(white_temp)
            if hasattr(light, 'set_brightness'):
                await light.set_brightness(v)
            if double_write:
                await asyncio.sleep(0.05)
                await light.set_color_temp(white_temp)
                if hasattr(light, 'set_brightness'):
                    await light.set_brightness(v)
            return
    ok = await set_via_light_module(dev, h, s, v)
    if ok:
        if double_write:
            await asyncio.sleep(0.05)
            await set_via_light_module(dev, h, s, v)
        return
    r,g,b = hsv_to_rgb_int(h,s,v)
    try:
        if hasattr(dev, 'set_light_state'):
            await dev.set_light_state(hue=h, saturation=s, brightness=v, rgb=(r,g,b))
            if double_write:
                await asyncio.sleep(0.05)
                await dev.set_light_state(hue=h, saturation=s, brightness=v, rgb=(r,g,b))
            return
    except Exception:
        pass
    if light and hasattr(light, 'set_rgb'):
        await light.set_rgb(r,g,b)
        if double_write:
            await asyncio.sleep(0.05)
            await light.set_rgb(r,g,b)
        return
    log('All methods failed to set color')


async def run_show(ip, duration, transition, colors, instant, double_write, off_flash, white_temp, fade):
    dev = await find_device(ip)
    saved = await get_current_state(dev)

    # Deterministic iteration: compute number of steps from duration and sleep_time
    sleep_time = INSTANT_SLEEP if instant or transition <= 0 else transition
    if not colors:
        raise ValueError("No colors provided")
    steps = max(1, int(math.ceil(duration / sleep_time)))
    n = len(colors)

    # Iterate strictly by index to avoid duplicating the initial color and
    # accidentally skipping a color (previous behavior pre-applied the first
    # color, which shifted the cycle on subsequent loops).
    for i in range(steps):
        h, s, v = colors[i % n]
        prev = colors[(i-1) % n] if i > 0 else None
        # Insert off-flash only when transitioning from blue to an actual red (not white).
        # White is represented with saturation==0, so require target saturation > 0.
        if off_flash and prev and prev[0] == 240 and h == 0 and s > 0:
            log('inserting stronger off-flash between blue->red')
            ok = await set_brightness_via_module(dev, 0)
            if not ok:
                log('failed to set brightness=0')
            await asyncio.sleep(OFF_FLASH_SLEEP)
            await restore_color_with_brightness(dev, h, s, v, double_write, white_temp)
            await asyncio.sleep(OFF_FLASH_SETTLE)
        else:
            log('applying', (h, s, v))
            await apply_color(dev, h, s, v, instant, double_write, white_temp)
        await asyncio.sleep(sleep_time)

    # finished — restore saved state
    log('show finished; restoring saved state', force=True)
    if fade and fade>0:
        await fade_restore(dev, saved, fade, steps=max(6, int(fade*10)), double_write=double_write)
    else:
        await set_state_from_saved(dev, saved, instant=True, double_write=double_write)
    log('restore complete', force=True)


if __name__ == '__main__':
    args = parse_args()
    LOG_VERBOSE = args.verbose
    colors = parse_colors(args.colors)
    asyncio.run(run_show(args.ip, args.duration, args.transition, colors, args.instant, args.double_write, args.off_flash, args.white_temp, args.fade))
