#!/usr/bin/env python3
"""
Updated LAN controller for TP-Link Kasa-compatible smart bulbs using python-kasa (modern API).
This version uses Discover.discover_single() and the device/modules API (no deprecated SmartBulb).

Usage examples:
  # Use uv (recommended; manages deps without manual environment activation):
  uv run --project ./skills/control-ikea-lightbulb python ./skills/control-ikea-lightbulb/scripts/control_kasa_light.py --ip 192.168.4.69 --on
  # Use the provided wrapper script (requires uv):
  ./skills/control-ikea-lightbulb/scripts/run_control_kasa.sh --ip 192.168.4.69 --on --hsv 0 100 80 --brightness 80

Notes:
- The wrapper requires uv to run.
- Run from the same LAN as the bulb. No cloud credentials required.
"""
import argparse
import asyncio


def parse_args():
    p = argparse.ArgumentParser(description="Control a Kasa-compatible smart bulb on the LAN (modern python-kasa API)")
    p.add_argument("--ip", required=True, help="IP address of the bulb on the LAN")
    p.add_argument("--on", action="store_true", help="Turn bulb on")
    p.add_argument("--off", action="store_true", help="Turn bulb off")
    p.add_argument("--brightness", type=int, help="Set brightness (0-100)")
    p.add_argument("--hsv", nargs=3, type=int, metavar=("H","S","V"), help="Set color as HSV integers: H(0-360) S(0-100) V(0-100)")
    p.add_argument("--transition", type=float, default=0, help="Transition duration in seconds (if supported)")
    return p.parse_args()


async def find_device(ip: str):
    try:
        from kasa import Discover
    except Exception:
        raise

    # discover_single returns a Device or None
    dev = await Discover.discover_single(ip)
    if dev is None:
        raise RuntimeError(f"Device at {ip} not found via discovery")
    return dev


async def main():
    args = parse_args()
    try:
        # Import lazily so users without python-kasa get a helpful message
        from kasa import Discover
    except Exception:
        print("Error importing python-kasa. Install with: pip install python-kasa")
        raise

    dev = await find_device(args.ip)

    # Connect/update device state
    try:
        await dev.update()
    except Exception:
        # Some devices require explicit connect
        try:
            await dev.connect()
            await dev.update()
        except Exception as e:
            print("Failed to connect/update device:", e)
            raise

    # Convenience access to light module (module keys may vary in capitalization)
    light_module = None
    for key in ("Light", "light", "light-1"):
        if key in getattr(dev, 'modules', {}):
            light_module = dev.modules[key]
            break
    # Fallback: try to find first module with 'Light' class name
    if light_module is None:
        for m in getattr(dev, 'modules', {}).values():
            # many modules have a 'MODULE' or class name; we try to find a candidate
            try:
                name = getattr(m, '__class__', type(m)).__name__
                if 'Light' in name or 'light' in name.lower():
                    light_module = m
                    break
            except Exception:
                continue

    # If device has high-level methods, use them as fallback
    async def turn_on():
        if hasattr(dev, 'turn_on'):
            await dev.turn_on()
        elif light_module and hasattr(light_module, 'turn_on'):
            await light_module.turn_on()
        else:
            raise RuntimeError("No turn_on method available for device")

    async def turn_off():
        if hasattr(dev, 'turn_off'):
            await dev.turn_off()
        elif light_module and hasattr(light_module, 'turn_off'):
            await light_module.turn_off()
        else:
            raise RuntimeError("No turn_off method available for device")

    async def set_brightness(b):
        if light_module and hasattr(light_module, 'set_brightness'):
            await light_module.set_brightness(b)
        elif 'set_light_state' in dir(dev):
            await dev.set_light_state(brightness=b)
        else:
            # try generic module method
            if light_module and hasattr(light_module, 'set_state'):
                await light_module.set_state(brightness=b)
            else:
                raise RuntimeError("No supported brightness API found on device")

    async def set_hsv(h, s, v):
        if light_module and hasattr(light_module, 'set_hsv'):
            await light_module.set_hsv(h, s, v)
        elif light_module and hasattr(light_module, 'set_rgb'):
            import colorsys
            r, g, b = colorsys.hsv_to_rgb(h/360.0, s/100.0, v/100.0)
            rgb = (int(r*255), int(g*255), int(b*255))
            await light_module.set_rgb(*rgb)
        elif hasattr(dev, 'set_light_state'):
            # try passing dict
            await dev.set_light_state(hue=h, saturation=s, brightness=v)
        else:
            raise RuntimeError("No supported color API found on device")

    # Execute requested actions
    if args.on:
        await turn_on()
        print(f"Turned on {args.ip}")
    if args.off:
        await turn_off()
        print(f"Turned off {args.ip}")
    if args.brightness is not None:
        b = max(0, min(100, args.brightness))
        try:
            await set_brightness(b)
            print(f"Set brightness to {b}%")
        except Exception as e:
            print("Failed to set brightness:", e)
    if args.hsv:
        h, s, v = args.hsv
        h = max(0, min(360, h))
        s = max(0, min(100, s))
        v = max(0, min(100, v))
        try:
            await set_hsv(h, s, v)
            print(f"Set color HSV=({h},{s},{v})")
        except Exception as e:
            print("Failed to set color:", e)


if __name__ == "__main__":
    asyncio.run(main())
