---
name: context-verifier
version: 1.5.1
description: Know the file you're editing is the file you think it is — verify integrity before you act
author: Live Neon <contact@liveneon.dev>
homepage: https://github.com/live-neon/skills/tree/main/agentic/context-verifier
repository: leegitw/context-verifier
license: MIT
tags: [agentic, verification, integrity, validation, checksums, drift-detection, state, file-verification]
layer: foundation
status: active
alias: cv
metadata:
  openclaw:
    requires:
      config:
        - .openclaw/context-verifier.yaml
        - .claude/context-verifier.yaml
      workspace:
        - output/context-packets/
---

# context-verifier (検証)

Unified skill for file hash computation, integrity verification, severity tagging,
and context packet creation. Consolidates 3 granular skills into a single verification system.

**Trigger**: 明示呼出 (explicit invocation)

**Source skills**: context-packet, file-verifier, severity-tagger

## Installation

```bash
openclaw install leegitw/context-verifier
```

**Dependencies**: None (foundational skill)

**Standalone usage**: This skill is fully functional standalone. It provides file integrity
verification that other skills in the suite depend on. Install this first when adopting
the Neon Agentic Suite.

**Data handling**: This skill performs local-only operations. Hash computation uses standard
SHA256 algorithms locally — no file contents are sent to any model, API, or external service.
Results are written to `output/context-packets/` in your workspace. The skill reads config from
`.openclaw/context-verifier.yaml` or `.claude/context-verifier.yaml` only.

**File access scope**: This skill reads user-specified files for hash computation. The metadata
declares config and output paths only — the skill will read ANY file path you provide to
`/cv hash`, `/cv verify`, or `/cv packet`. Use caution with sensitive files.

## What This Solves

AI agents sometimes operate on stale data — editing a file that changed since it was read, or trusting cached content that's now outdated. This skill prevents that by:

1. **Computing hashes** of files before and after operations
2. **Detecting changes** between read and write
3. **Generating context packets** with verifiable checksums for review workflows

**The insight**: Trust but verify. The file you read might not be the file you're about to edit. Check first.

## Usage

```
/cv <sub-command> [arguments]
```

## Sub-Commands

| Command | CJK | Logic | Trigger |
|---------|-----|-------|---------|
| `/cv hash` | 哈希 | file→SHA256(content) | Explicit |
| `/cv verify` | 検証 | file×hash→match✓∨mismatch✗ | Explicit |
| `/cv tag` | 標記 | file→severity∈{critical,important,minor} | Explicit |
| `/cv packet` | 包装 | files[]→{path,hash,severity,timestamp}[] | Explicit |

## Arguments

### /cv hash

| Argument | Required | Description |
|----------|----------|-------------|
| file | Yes | File path to hash |
| --algorithm | No | Hash algorithm: `sha256` only (MD5/SHA-1 removed - cryptographically broken) |

### /cv verify

| Argument | Required | Description |
|----------|----------|-------------|
| file | Yes | File path to verify |
| hash | Yes | Expected hash value |
| --algorithm | No | Hash algorithm: sha256 only |

### /cv tag

| Argument | Required | Description |
|----------|----------|-------------|
| file | Yes | File path to tag |
| severity | No | Severity level: `critical`, `important`, `minor` (auto-detected if omitted) |

### /cv packet

| Argument | Required | Description |
|----------|----------|-------------|
| files | Yes | Comma-separated file paths or glob pattern |
| --name | No | Packet name (default: auto-generated) |
| --include-content | No | Include file content in packet (default: false) - **see Security section** |

> **⚠️ Security Warning**: The `--include-content` flag stores file contents to disk.
> Never use this flag with sensitive files (`.env`, credentials, secrets).
> See the [Security Considerations](#security-considerations) section below.

## Configuration

Configuration is loaded from (in order of precedence):
1. `.openclaw/context-verifier.yaml` (OpenClaw standard)
2. `.claude/context-verifier.yaml` (Claude Code compatibility)
3. Defaults (built-in patterns)

## Security Considerations

**Local-only processing**: All hash computation uses standard SHA256 algorithms executed locally.
No file contents are ever sent to any LLM, API, or external service. The "agent's model" is only
used to interpret your commands — not to process file contents.

**What this skill does NOT do:**
- Send file contents to any model or API (hashing is local)
- Call external APIs or third-party services
- Modify source files (only writes to `output/context-packets/`)

**What this skill accesses:**
- Configuration files in `.openclaw/context-verifier.yaml` and `.claude/context-verifier.yaml`
- **Any user-specified files** for hash computation (read-only) — the skill reads whatever paths you provide
- Its own output directory `output/context-packets/` (write)

**⚠️ IMPORTANT**: Unlike other skills in this suite, context-verifier reads arbitrary files that
you specify. The metadata only declares config/output paths. When you run `/cv hash myfile.go`,
the skill reads `myfile.go` even though it's not in the metadata. This is by design — verification
requires reading the files you want to verify.

This skill handles file metadata and optionally file contents. Follow these guidelines:

### Sensitive File Detection (Not Reading)

The `critical_patterns` (e.g., `*.env`, `*credentials*`, `*secret*`) are used for:
- **Detection**: Identifying files that should trigger warnings
- **Severity tagging**: Marking files as critical for change-blocking behavior

By default, `/cv hash` and `/cv packet` compute hashes **without reading file contents into output**.
The hash is computed but the file content is not stored.

### --include-content Flag

**⚠️ WARNING**: The `--include-content` flag writes actual file contents to disk.

| Risk | Mitigation |
|------|------------|
| Secrets written to disk | Never use `--include-content` with `.env`, credentials, or secret files |
| Sensitive data in git | Add `output/context-packets/` to `.gitignore` (see below) |
| Data retention | Packets are stored indefinitely; manually delete when no longer needed |

**Recommended usage**:
```bash
# Safe: Hash only (default) - no content stored
/cv packet src/*.go --name "pre-refactor"

# Risky: Content included - ensure no sensitive files in glob
/cv packet docs/*.md --name "docs-backup" --include-content

# NEVER do this:
/cv packet .env --include-content  # Stores secrets to disk!
```

### Required .gitignore Entry

Add to your `.gitignore` to prevent accidental commits:

```gitignore
# Context verification packets (may contain sensitive data)
output/context-packets/
```

### Storage and Retention

- **Location**: Packets stored in `output/context-packets/` (workspace-local)
- **Format**: Unencrypted JSON
- **Retention**: No automatic deletion; clean up manually
- **Access**: Standard filesystem permissions (no additional access controls)

For sensitive environments, consider:
1. Restricting `output/` directory permissions
2. Using encrypted filesystems
3. Periodic cleanup of old packets

### Provenance

This skill is developed by Live Neon (https://github.com/live-neon/skills) and published
to ClawHub under the `leegitw` account. Both refer to the same maintainer.

## Core Logic

### Hash Computation

Default algorithm: SHA-256

```
hash(file) = SHA256(file.content)
```

### Severity Classification

Files are auto-classified based on configurable patterns:

| Severity | Default Patterns | Behavior on Change |
|----------|------------------|-------------------|
| critical | `*.env`, `*credentials*`, `*secret*`, project config | Block operation |
| important | `*.go`, `*.ts`, `*.md` (in docs/) | Warn user |
| minor | `*.log`, `*.tmp`, `output/*` | Info only |

Critical file patterns are configurable via `.openclaw/context-verifier.yaml`:

```yaml
# .openclaw/context-verifier.yaml
critical_patterns:
  - "*.env"
  - "*credentials*"
  - "*secret*"
  - "CLAUDE.md"     # Claude Code projects
  - "AGENTS.md"     # OpenClaw projects
  - "pyproject.toml" # Python projects
  - "Cargo.toml"    # Rust projects
```

### Context Packet Structure

```json
{
  "id": "PKT-20260215-001",
  "created": "2026-02-15T10:30:00Z",
  "files": [
    {
      "path": "src/main.go",
      "hash": "abc123...",
      "severity": "important",
      "size": 1234,
      "modified": "2026-02-15T10:00:00Z"
    }
  ],
  "metadata": {
    "purpose": "pre-refactor snapshot",
    "creator": "context-verifier"
  }
}
```

## Output

### /cv hash output

```
[HASH] src/main.go
Algorithm: SHA-256
Hash: a1b2c3d4e5f6...
Size: 1,234 bytes
Modified: 2026-02-15 10:00:00
```

### /cv verify output (match)

```
[VERIFY] src/main.go
Status: ✓ MATCH
Expected: a1b2c3d4e5f6...
Actual:   a1b2c3d4e5f6...
```

### /cv verify output (mismatch)

```
[VERIFY] src/main.go
Status: ✗ MISMATCH
Expected: a1b2c3d4e5f6...
Actual:   x9y8z7w6v5u4...

WARNING: File has changed since last read.
Action: Re-read file before making changes.
```

### /cv tag output

```
[TAG] src/main.go
Severity: important
Reason: Go source file
Behavior: Warn on unexpected change
```

### /cv packet output

```
[PACKET CREATED]
ID: PKT-20260215-001
Files: 4
Total size: 10,234 bytes

Files included:
- src/main.go (important) - a1b2c3...
- src/handler.go (important) - d4e5f6...
- docs/README.md (important) - j0k1l2...
- config/settings.yaml (important) - m3n4o5...

Stored: output/context-packets/PKT-20260215-001.json
```

> **Note**: Avoid including sensitive files (`.env`, credentials) in packets.
> Use specific globs like `src/*.go` rather than `*` to exclude secrets.

## Integration

- **Layer**: Foundation (no dependencies)
- **Depends on**: None (foundational verification system)
- **Used by**: failure-memory (for file change detection), constraint-engine (for pre-action checks)

## Failure Modes

| Condition | Behavior |
|-----------|----------|
| File not found | Error: "File not found: {path}" |
| Permission denied | Error: "Cannot read file: {path}" |
| Invalid hash format | Error: "Invalid hash format. Expected: {algorithm}" |
| Glob matches no files | Warning: "No files match pattern: {glob}" |

## Next Steps

After invoking this skill:

| Condition | Action |
|-----------|--------|
| Hash mismatch | Alert user, suggest re-read of file |
| Critical file changed | Block operation, require verification |
| Packet created | Store in `output/context-packets/` for audit |

## Workspace Files

This skill reads/writes:

```
output/
└── context-packets/
    └── PKT-YYYYMMDD-XXX.json
```

## Examples

### Verify file before editing

```
/cv hash src/main.go
# Save hash: a1b2c3d4e5f6...

# ... later, before editing ...

/cv verify src/main.go a1b2c3d4e5f6
# ✓ MATCH - safe to edit
```

### Create context packet for refactoring

```
/cv packet src/*.go,internal/**/*.go --name "pre-refactor"
# Creates packet with all Go files

# ... after refactoring ...

# Can compare against packet to see what changed
```

### Tag sensitive files

```
/cv tag .env
# Severity: critical

/cv tag src/handler.go
# Severity: important
```

### Verify database migration before deployment

```
/cv packet db/migrations/*.sql --name "pre-deploy-migrations"
# Creates packet with all migration files

# After staging deployment...
/cv verify db/migrations/001_users.sql abc123...
# ✓ MATCH - migration file unchanged, safe to deploy to production
```

### Create API schema verification packet

```
/cv packet api/schemas/*.json,api/openapi.yaml --name "api-schema-v2"
# Creates packet with all API schema files for version control
```

## Acceptance Criteria

- [ ] `/cv hash` computes SHA-256 hash of file
- [ ] `/cv verify` compares file hash against expected value
- [ ] `/cv verify` clearly indicates match/mismatch
- [ ] `/cv tag` auto-classifies file severity based on patterns
- [ ] `/cv tag` allows manual severity override
- [ ] `/cv packet` creates JSON packet with file metadata
- [ ] `/cv packet` supports glob patterns
- [ ] Critical file changes trigger block behavior
- [ ] Workspace files stored in documented location

---

*Consolidated from 3 skills as part of agentic skills consolidation (2026-02-15).*
