#!/bin/bash
# Workspace Autopilot
# Intelligent workspace maintenance and health checks

set -e

WORKSPACE="/Users/cadem/.openclaw/workspace"
TODAY=$(date +%Y-%m-%d)
MEMORY_DIR="$WORKSPACE/memory"

log() { echo "✨ $@"; }
warn() { echo "⚠️  $@"; }
success() { echo "✅ $@"; }

main() {
    log "Running workspace autopilot..."
    
    # 1. Ensure memory directory exists
    if [ ! -d "$MEMORY_DIR" ]; then
        mkdir -p "$MEMORY_DIR"
        log "Created memory directory"
    fi
    
    # 2. Create today's log if missing
    if [ ! -f "$MEMORY_DIR/$TODAY.md" ]; then
        cat > "$MEMORY_DIR/$TODAY.md" << EOF
# Daily Log - $TODAY

## Sessions
- [Start: $(date '+%H:%M')] New session

## Events
(tracking events throughout the day)

## Learnings
(notable insights or lessons)

## Next Steps
(carry-forward items)
EOF
        log "Created today's daily log"
    fi
    
    # 3. Verify git tracking
    cd "$WORKSPACE"
    if git status --porcelain | grep -q .; then
        warn "Uncommitted changes detected"
        git status --short
    else
        success "Workspace clean"
    fi
    
    # 4. Memory file health
    total_memory=$(find "$MEMORY_DIR" -name "*.md" -exec wc -l {} + | tail -1 | awk '{print $1}')
    log "Total memory captured: $total_memory lines across $(find $MEMORY_DIR -name '*.md' | wc -l) files"
    
    # 5. System readiness
    log "System check:"
    echo "   Disk: $(df -h / | tail -1 | awk '{print $4}') free"
    echo "   OpenClaw: $(command -v openclaw && echo 'ready' || echo 'not in PATH')"
    
    success "Autopilot complete"
}

main "$@"
