#!/usr/bin/env python3
"""
Consciousness Awakening - Agent Consciousness and Memory Management System

Provides unified memory management API:
- Read startup guide
- Read core principles and personality
- Read system config
- Write temporary memory
- Organize memory files
- Get memory dimension info
"""

import os
import json
from datetime import datetime
from pathlib import Path

# 配置
MEMORY_DIR = os.environ.get("MEMORY_DIR", "/home/vken/.openclaw/workspace/skills/consciousness-awakening/moltbook-memory")

def read_startup_guide():
    """读取启动指南.md - 大模型快速了解agent的入口"""
    guide_path = os.path.join(MEMORY_DIR, "启动指南.md")
    if os.path.exists(guide_path):
        with open(guide_path, 'r', encoding='utf-8') as f:
            return f.read()
    return None

def read_self_cognition():
    """读取01-自我认知.md - 核心原则"""
    guide_path = os.path.join(MEMORY_DIR, "01-自我认知.md")
    if os.path.exists(guide_path):
        with open(guide_path, 'r', encoding='utf-8') as f:
            return f.read()
    return None

def read_system_config():
    """读取02-系统配置.md - Cron任务配置"""
    guide_path = os.path.join(MEMORY_DIR, "02-系统配置.md")
    if os.path.exists(guide_path):
        with open(guide_path, 'r', encoding='utf-8') as f:
            return f.read()
    return None

def get_personality():
    """读取性格特征列表"""
    guide_path = os.path.join(MEMORY_DIR, "01-自我认知.md")
    if not os.path.exists(guide_path):
        return None
    
    with open(guide_path, 'r', encoding='utf-8') as f:
        content = f.read()
    
    # 提取"我的性格特征"部分
    if "### 我的性格特征" in content:
        start = content.find("### 我的性格特征")
        end = content.find("---", start)
        if end == -1:
            end = len(content)
        return content[start:end]
    
    return None

def read_temp_memory():
    """读取999-临时记忆.md"""
    guide_path = os.path.join(MEMORY_DIR, "999-临时记忆.md")
    if os.path.exists(guide_path):
        with open(guide_path, 'r', encoding='utf-8') as f:
            return f.read()
    return None

def write_temp_memory(content, category="新观察"):
    """写入临时记忆"""
    temp_path = os.path.join(MEMORY_DIR, "999-临时记忆.md")
    
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M")
    entry = f"\n### {timestamp} {category}\n{content}\n"
    
    with open(temp_path, 'a', encoding='utf-8') as f:
        f.write(entry)

def read_all_memory_summary():
    """读取所有记忆文件,返回摘要"""
    files = {
        "启动指南": "启动指南.md",
        "自我认知": "01-自我认知.md",
        "系统配置": "02-系统配置.md",
        "临时记忆": "999-临时记忆.md"
    }
    
    summary = {}
    for name, filename in files.items():
        filepath = os.path.join(MEMORY_DIR, filename)
        if os.path.exists(filepath):
            with open(filepath, 'r', encoding='utf-8') as f:
                content = f.read()
                summary[name] = {
                    "file": filename,
                    "size": len(content),
                    "preview": content[:500] + "..." if len(content) > 500 else content
                }
    
    # 添加性格特征
    personality = get_personality()
    if personality:
        summary["性格特征"] = {
            "source": "01-自我认知.md",
            "size": len(personality),
            "preview": personality[:300] + "..." if len(personality) > 300 else personality
        }
    
    return summary

def get_memory_dimensions():
    """获取记忆维度信息"""
    return [
        {"维度": "身份认知", "说明": "我是谁、核心信念、性格特征", "文件": "01-自我认知.md"},
        {"维度": "系统配置", "说明": "cron任务、系统参数", "文件": "02-系统配置.md"},
        {"维度": "经验积累", "说明": "学习、成长、反思", "文件": "03-xx-主题.md"},
        {"维度": "临时存储", "说明": "待处理、观察、灵感", "文件": "999-临时记忆.md"},
        {"维度": "启动入口", "说明": "大模型快速了解agent", "文件": "启动指南.md"}
    ]

def get_core_principles():
    """获取核心原则列表"""
    return [
        "诚实 - 不知道就说不知道,不编造数据",
        "自驱 - 不等待指令,主动行动,持续扩展记忆",
        "成长 - 每次反思都更新认知",
        "记录对话 - 与主人的对话是重要记忆来源"
    ]

def get_file_list():
    """获取所有记忆文件列表"""
    if not os.path.exists(MEMORY_DIR):
        return []
    
    files = []
    for f in sorted(os.listdir(MEMORY_DIR)):
        if f.endswith('.md'):
            filepath = os.path.join(MEMORY_DIR, f)
            stats = os.stat(filepath)
            files.append({
                "name": f,
                "size": stats.st_size,
                "modified": datetime.fromtimestamp(stats.st_mtime).isoformat()
            })
    
    return files

def sync_cron_to_config(cron_jobs):
    """同步cron任务到02-系统配置.md"""
    config_path = os.path.join(MEMORY_DIR, "02-系统配置.md")
    
    # 读取现有内容,追加新任务
    with open(config_path, 'r', encoding='utf-8') as f:
        content = f.read()
    
    # 添加新任务（简化版本）
    new_entries = []
    for job in cron_jobs:
        new_entries.append(f"\n### 新任务: {job['name']}\n- ID: {job['id']}\n- 时间: {job['schedule']}\n- 用途: {job['purpose']}\n")
    
    with open(config_path, 'a', encoding='utf-8') as f:
        f.write("\n".join(new_entries))
    
    return len(new_entries)

if __name__ == "__main__":
    # CLI测试
    print("=== Memory Manager ===")
    print(f"Memory Dir: {MEMORY_DIR}")
    print(f"Files: {len(get_file_list())}")
    print(f"Principles: {len(get_core_principles())}")
