# Skill: config-modification
# 配置文件修改安全流程（双层守护：fswatch 即时拦截 + cron 健康巡检）

## 触发条件
当需要修改以下配置文件时**强制触发**：
- `openclaw.json`
- skills 配置
- 任何 `~/.openclaw/` 下的 JSON 配置文件

**⚠️ 无例外原则**：不管是正式修改还是测试，只要动配置文件，都必须走完整流程。

---

## 双层守护架构

```
                    配置文件被修改
                         |
          +--------------+--------------+
          |                              |
   【第一层：fswatch 即时守护】     【第二层：cron 健康巡检】
   kqueue 事件驱动，亚秒响应        每 5 分钟轮询，模型驱动
          |                              |
   JSON 语法校验（0 token）         curl /api/health
          |                              |
   +-- 无效 -> 立即回滚（0 token）  +-- 非200 -> 执行回滚脚本
   |                                |
   +-- 有效 -> 重置健康计数器       +-- 200 -> consecutive++
                                    |
                                    +-- >= 3次 -> 配置稳定
                                        自动禁用 cron
```

**分工**：
- **fswatch**：拦截语法错误（即时，0 token）—— 第一道铜墙
- **cron**：捕获运行时异常（语法对但语义错导致 Gateway 不健康）—— 第二道铁壁

---

## 完整流程（6步，全部强制）

### 第一步：确认 fswatch 守护在运行

```bash
# 检查 fswatch 守护进程
ps aux | grep config-fswatch-guard | grep -v grep
```

如果没在运行，先启动：
```bash
# 方法 A：手动启动
nohup python3 ~/.openclaw/workspace/.lib/config-fswatch-guard.py > /dev/null 2>&1 &

# 方法 B：通过 LaunchAgent（推荐，崩溃自动重启）
launchctl load ~/Library/LaunchAgents/com.openclaw.config-fswatch-guard.plist
```

### 第二步：授权确认

向用户清晰说明：
- 要修改的文件
- 修改的内容
- 可能的影响
- 询问 "请确认 (y/n)"

### 第三步：执行前备份

```bash
python3 ~/.openclaw/workspace/.lib/config-rollback-guard.py snapshot
```

### 第四步：修改配置 + 验证

```bash
# 修改后验证 JSON 语法
python3 -c "import json; json.load(open('$HOME/.openclaw/openclaw.json'))"
```

即使代理忘了验证，fswatch 也会在文件写入瞬间自动校验。

### 第五步：启用健康检查 cron

```bash
# 重置计数器
echo '{"consecutive_healthy":0,"threshold":3,"cron_job_id":"c42d5c66-9a60-4bb3-a346-e8a01a40621c","last_check":null}' > ~/.openclaw/logs/gateway-health-state.json
```

然后通过 cron 工具启用：
```
cron update jobId=c42d5c66-9a60-4bb3-a346-e8a01a40621c patch={"enabled": true}
```

或使用管理器：
```bash
bash ~/.openclaw/workspace/.lib/gateway-health-cron-manager.sh start
```

### 第六步：重启服务

```bash
openclaw gateway restart
```

重启后，双层守护自动接管：
1. fswatch 持续监听文件变更
2. cron 每 5 分钟健康巡检，连续 3 次通过后自动禁用

---

## 关键组件

### 第一层：fswatch 即时守护

| 组件 | 路径 | 用途 |
|------|------|------|
| 守护脚本 | `~/.openclaw/workspace/.lib/config-fswatch-guard.py` | kqueue 监听文件变更 |
| LaunchAgent | `~/Library/LaunchAgents/com.openclaw.config-fswatch-guard.plist` | 持久化运行 |
| 日志 | `~/.openclaw/logs/config-fswatch-guard.log` | 实时事件记录 |
| 回滚工具 | `~/.openclaw/workspace/.lib/config-rollback-guard.py` | 快照/回滚 |

**特点**：
- **kqueue 事件驱动**（macOS 原生，非轮询）
- **亚秒级响应**（文件写入 → 检测 → 回滚 < 1 秒）
- **零 token 消耗**（纯 Python，不调用模型）
- **自动重启**（通过 LaunchAgent，进程崩溃后自动拉起）

### 第二层：cron 健康巡检

| 组件 | 路径 | 用途 |
|------|------|------|
| Cron 任务 | ID: `c42d5c66-9a60-4bb3-a346-e8a01a40621c` | 定时触发健康检查 |
| 状态文件 | `~/.openclaw/logs/gateway-health-state.json` | 跟踪连续健康次数 |
| 回滚脚本 | `~/.openclaw/workspace/.lib/gateway-auto-rollback.sh` | 不健康时执行回滚 |
| 管理器 | `~/.openclaw/workspace/.lib/gateway-health-cron-manager.sh` | 启动/停止/状态管理 |
| 日志 | `~/.openclaw/logs/gateway-rollback.log` | 操作记录 |

**特点**：
- **轻量模型**：使用 `lovbrowser/claude-haiku-4.5`（~500 token/次）
- **自动禁用**：连续 3 次健康 → 自动关闭（总成本 ~1500 token）
- **捕获语义错误**：JSON 语法正确但配置值错误导致 Gateway 不健康的情况

---

## 守护配置

### 状态文件格式

```json
{
  "consecutive_healthy": 0,
  "threshold": 3,
  "cron_job_id": "c42d5c66-9a60-4bb3-a346-e8a01a40621c",
  "last_check": null
}
```

### 自定义阈值

```bash
# 修改为 5 次连续健康才确认
python3 -c "
import json, os
p = os.path.expanduser('~/.openclaw/logs/gateway-health-state.json')
s = json.load(open(p))
s['threshold'] = 5
json.dump(s, open(p,'w'), indent=2)
"
```

### 调整巡检间隔

通过 cron update 修改 `everyMs`：
- 3 分钟: `180000`
- 5 分钟: `300000`（默认）
- 10 分钟: `600000`

---

## 防护矩阵

| 故障类型 | 第一层 fswatch | 第二层 cron | 恢复时间 |
|----------|---------------|------------|---------|
| JSON 语法错误 | ✅ 即时回滚 | 不需要 | < 1 秒 |
| 配置值错误（Gateway 崩溃） | 检测到变更，重置计数器 | ✅ 健康检查失败 → 回滚 | 5~15 分钟 |
| 配置值错误（Gateway 异常但不崩溃） | — | ✅ 健康检查 200 但功能异常 → 需人工介入 | 人工 |
| 代理忘记走流程 | ✅ 仍然监听（常驻） | 需代理启用 | < 1 秒（语法）|

---

## 应急手册（给人类）

如果一切都失败了：

```bash
# 第 1 步：恢复配置
python3 ~/.openclaw/workspace/.lib/config-rollback-guard.py rollback

# 第 2 步：重启网关
openclaw gateway restart
```

终极兜底（连 Python 都不行时）：

```bash
# 找到最新备份
ls -t ~/.openclaw/backup/snapshot_*/openclaw.json | head -1

# 手动覆盖
cp <上面输出的路径> ~/.openclaw/openclaw.json

# 重启
openclaw gateway restart
```

---

## 关键约束（铁律）

1. **配置修改必须先获得用户授权**
2. **修改前必须创建快照**
3. **修改前必须确认 fswatch 在运行**
4. **修改后必须启用健康检查 cron**
5. **不存在"这次是测试所以跳过流程"的例外**
6. **重大修改需设置自动回滚监控**
7. **与 Observability/Schema 类能力严格隔离职责**：
   - 观测类能力只允许 `read_only + advisory_only`
   - 禁止执行 `config.apply/config.patch/gateway.config.*`
   - 禁止写入 `~/.openclaw/**/*.json` 与 `agents/**/models.json`

---

## 完整参考文档

- fswatch 守护脚本: `~/.openclaw/workspace/.lib/config-fswatch-guard.py`
- 回滚守护工具: `~/.openclaw/workspace/.lib/config-rollback-guard.py`
- 回滚指南: `~/.openclaw/workspace/.lib/CONFIG_ROLLBACK_GUIDE.md`
- 健康检查管理器: `~/.openclaw/workspace/.lib/gateway-health-cron-manager.sh`
- 自动回滚脚本: `~/.openclaw/workspace/.lib/gateway-auto-rollback.sh`
- AGENTS.md: "User Authorization Boundaries" 章节

---

## 测试验证记录

**2026-03-01 19:16 双保险竞速测试 ✅**

| 时间 | 事件 | 来源 |
|------|------|------|
| 19:16:13.118 | 注入 JSON 语法错误 | 测试 |
| 19:16:13 | 检测到变更 → 语法无效 → **立即回滚** | fswatch |
| 19:17:29 | 健康检查 1/3（Gateway 已恢复） | cron |
| 19:22 | 健康检查 2/3 | cron |
| 19:27 | 健康检查 3/3 → **自动禁用** | cron |

结论：fswatch 亚秒回滚，cron 15 分钟内确认稳定并自行关闭。
