# 复杂任务三步法 (Complex Task Three-Step Methodology)

> **通用方法论** — 适用于任何被评估为复杂的任务，不限于 Coding Team。
> 覆盖开发、研究、内容创作、系统运维、数据分析等所有领域。

---

## 架构总览：S0 → S3 立体评估与执行

```
用户消息
    ↓
┌─────────────────────────────────────┐
│ S0: 零成本预筛选（每条消息，0 token）  │
│   规则匹配 → 简单? → 直接执行         │
│                  → 疑似复杂? ↓        │
├─────────────────────────────────────┤
│ S1: 轻量评估（~300 token）            │
│   五维打分 → 总分 ≤ 8? → 直接执行     │
│             → 9-15? → 轻规划执行      │
│             → > 15? ↓                │
├─────────────────────────────────────┤
│ S2: 深度规划 & 审计                   │
│   Plan Mode (Opus) → Audit (Sonnet)  │
│   → 最多 2 轮修改 → 执行蓝图锁定      │
├─────────────────────────────────────┤
│ S3: 分阶段执行 & 质量控制              │
│   Phase 并行（DAG） → QA 审计循环     │
│   → 成果锁定 → 缺陷修改分级           │
└─────────────────────────────────────┘
```

---

## S0: 零成本预筛选

**每条用户消息都经过此层，纯规则匹配，不调用模型，零 token 开销。**

### 直接放行（白名单）— 跳过评估，直接执行

| 类型 | 示例 |
|------|------|
| 单轮问答 | "几点了"、"天气怎样"、"翻译这句话" |
| 延续指令 | "继续"、"接着说"、"下一步"、"然后呢" |
| 简单指令 | "帮我搜索X"、"打开Y"、"发消息给Z" |
| 闲聊/确认 | "好的"、"明白"、"谢谢"、"嗯" |

### 触发 S1 评估的信号（命中任一即进入 S1）

| 信号类型 | 检测规则 |
|----------|----------|
| **长度信号** | 用户消息 > 200 字，或包含多段落/列表 |
| **意图信号** | 出现动词：开发/构建/设计/部署/迁移/重构/分析/调研/实现/搭建 |
| **范围信号** | 出现词汇：整个/全部/系统/架构/从零开始/端到端/完整/全面 |
| **多步信号** | 出现模式："先…然后…最后…"、"第一步…第二步…"、多个动词并列 |
| **不确定信号** | 代理读完后判断不出明确的单步执行路径 |
| **显式触发** | 用户明确说"复杂任务"、"三步法"、"需要规划" |

### 预估流量分布

| 消息类型 | 占比 | 处理 | 额外成本 |
|----------|------|------|----------|
| 简单问答/闲聊 | ~60% | S0 直接放行 | 0 token |
| 明确单步指令 | ~20% | S0 直接放行 | 0 token |
| 疑似复杂 | ~15% | → S1 评估 | ~300 token |
| 真正复杂 | ~5% | → S1 → S2 → S3 | 300 + S2 成本 |

**平均每条消息额外开销：约 50-80 token。**

---

## S1: 轻量复杂度评估

**仅对通过 S0 筛选的消息执行。五维快速打分，~200-500 token。**

### 评估维度

| 维度 | 1分 | 3分 | 5分 |
|------|-----|-----|-----|
| **步骤数** | 1-2步可完成 | 3-5步 | 6步以上 |
| **知识域** | 单一领域 | 2-3个领域交叉 | 4+领域，需专家知识 |
| **不确定性** | 路径清晰 | 部分需要搜索 | 大量未知，需调研 |
| **失败代价** | 重做成本低 | 中等回退成本 | 不可逆或高代价 |
| **工具链** | 单工具 | 2-3个工具协调 | 复杂工具链/多系统 |

### 决策阈值

| 总分 | 复杂度等级 | 执行方式 |
|------|-----------|----------|
| **≤ 8** | 简单 | 直接执行，无需规划 |
| **9 - 15** | 中等 | 轻规划：心里列步骤，边做边调整 |
| **> 15** | 复杂 | 完整三步法：S2 规划审计 → S3 分阶段执行 |

### 动态升级兜底

即使 S0 漏判或 S1 低估，执行过程中出现以下情况时**动态升级**：

- 已尝试 2 次失败
- 发现实际步骤远多于预估
- 遇到未预期的依赖或阻塞
- 需要的知识域超出预期

→ **中途触发 S1 重新评估**，决定是否升级到完整三步法。允许运行时纠偏。

---

## S2: 深度规划 & 审计

**仅 S1 评分 > 15 的复杂任务进入此阶段。**

### 2.1 Plan Mode

```
输入：任务描述 + S1 评估结果
模型：高能力模型（如 Opus）
输出：
  ├─ 任务分解（DAG 结构，支持并行）
  ├─ 每步的预期产物
  ├─ 依赖关系图
  ├─ 风险点标注
  └─ 资源/工具需求
```

### 2.2 Audit Mode

```
输入：Plan Mode 的输出
模型：审计模型（如 Sonnet）
检查：
  ├─ 步骤完整性（有无遗漏）
  ├─ 依赖合理性（有无循环依赖）
  ├─ 风险覆盖度（有无未标注风险）
  ├─ 资源可行性（工具/权限是否可用）
  └─ 时间合理性（预估是否靠谱）
输出：
  ├─ APPROVED — 直接进入 S3
  ├─ APPROVED_WITH_SUGGESTIONS — 进入 S3，附带改进建议
  └─ NEEDS_REVISION — 返回 Plan Mode 修改（最多 2 轮）
```

### 2.3 步骤规划：DAG 并行结构

**步骤规划不强制串行。** 支持有向无环图（DAG）结构：

```
Step 1: 分析需求
Step 2a: 搜索 API 文档  ┐
Step 2b: 检查本地缓存    ├─ 并行执行
Step 2c: 查询数据库      ┘
Step 3: 综合结果（依赖 2a, 2b, 2c 全部完成）
Step 4: 生成报告
```

数据结构：

```json
{
  "steps": [
    {"id": 1, "action": "分析需求", "depends_on": []},
    {"id": "2a", "action": "搜索API文档", "depends_on": [1]},
    {"id": "2b", "action": "检查本地缓存", "depends_on": [1]},
    {"id": "2c", "action": "查询数据库", "depends_on": [1]},
    {"id": 3, "action": "综合结果", "depends_on": ["2a", "2b", "2c"]},
    {"id": 4, "action": "生成报告", "depends_on": [3]}
  ]
}
```

**执行规则：** 所有 `depends_on` 都已完成的步骤，同时发起执行。

### 2.4 执行蓝图锁定

Plan + Audit 通过后，输出**执行蓝图**：
- 锁定步骤、依赖、产物定义
- 整个 S3 围绕此蓝图执行
- 偏离计划必须记录原因

### 2.5 蓝图快照机制（新增，强制）

**S2 阶段一旦生成 DAG 执行蓝图，必须立即生成“带项目名”的蓝图快照。**

#### 目的
- 为中断恢复、断点续跑、历史审计提供稳定基线
- 保证后续维护是“增量版本”而非覆盖原件

#### 强制规则
1. **首次快照**：蓝图生成后立即落盘
2. **命名要求**：必须包含项目名称 + 版本号 + 时间戳
3. **不可覆盖**：后续维护不得修改原快照
4. **增量演进**：任何调整都生成新快照（版本递增 + 新时间戳）

#### 命名规范（示例）

```text
blueprints/<project_name>/
  ├─ blueprint-v1-2026-03-03T20-25-00+08-00.json
  ├─ blueprint-v2-2026-03-03T21-10-32+08-00.json
  └─ blueprint-v3-2026-03-04T09-08-11+08-00.json
```

#### 最小元数据（每个快照）

```json
{
  "project_name": "<项目名>",
  "version": "v2",
  "created_at": "2026-03-03T21:10:32+08:00",
  "based_on": "blueprint-v1-2026-03-03T20-25-00+08-00.json",
  "change_summary": "新增 Phase 3 的依赖约束",
  "blueprint": { "steps": [] }
}
```

---

## S3: 分阶段执行 & 质量控制

**按执行蓝图分 Phase 执行，每个 Phase 有独立的 QA 审计循环。**

### 3.1 Phase 执行

```
Phase 1: [步骤组]
  ├─ 同 Phase 内步骤可并行（DAG）
  ├─ 每步完成 → QA 审计
  ├─ QA 通过 → 成果锁定
  └─ QA 不通过 → 缺陷修改循环

Phase 2: [步骤组]（使用 Phase 1 的锁定成果）
  ├─ ...
  └─ ...

所有 Phase 完成 → ✅ 任务完成（含完整审计记录）
```

### 3.2 三道防线

| 防线 | 角色 | 职责 |
|------|------|------|
| **Audit** | 审计模型 | 计划阶段的风险识别 |
| **QA** | QA 审计 | 执行阶段的质量把关 |
| **Defect Rule** | 缺陷规则 | 贯穿全程的问题修复 |

### 3.3 缺陷修改分级

| 严重度 | 处理方式 |
|--------|----------|
| **Critical** | 自动批准修改 |
| **High** | 自动批准 + 通知 Sir |
| **Medium** | Sir 确认后修改 |
| **Low** | QA 自行决定 |

所有修改都记录：版本、变更日志、影响分析。

### 3.4 成果锁定机制

- 每步通过 QA 后，成果被"锁定"
- 后续 Phase 使用前置 Phase 的锁定成果
- 修改已锁定成果需遵循缺陷修改分级

### 3.5 模型分工（参考）

| 角色 | 推荐模型 | 职责 |
|------|----------|------|
| Plan Mode | Opus | 深度规划，全局思维 |
| Audit Mode | Sonnet | 批判分析，风险识别 |
| 执行 Agent | 按需 | 具体实施，遵循蓝图 |
| QA | Sonnet | 质量把关，找问题 |
| Sir | 人类 | 最终决策，资源平衡 |

---

## 完整伪代码

```python
async def handle_user_message(message):
    """
    S0-S3 立体复杂任务评估与执行
    """

    # ==================== S0: 零成本预筛选 ====================
    if is_simple_message(message):
        # 白名单命中：单轮问答、延续、简单指令、闲聊
        return await direct_execution(message)

    if not has_complexity_signal(message):
        # 无复杂信号：长度、意图、范围、多步、不确定
        return await direct_execution(message)

    # ==================== S1: 轻量评估 ====================
    score = await evaluate_complexity(
        message=message,
        dimensions=["步骤数", "知识域", "不确定性", "失败代价", "工具链"],
    )

    if score.total <= 8:
        return await direct_execution(message)

    if score.total <= 15:
        return await light_plan_execution(message, score)

    # ==================== S2: 深度规划 & 审计 ====================
    plan = await plan_mode(
        model="opus",
        task=message,
        complexity=score,
        structure="dag",  # 支持并行步骤
    )

    audit = await audit_mode(model="sonnet", plan=plan)

    for revision in range(2):
        if audit.verdict in ["APPROVED", "APPROVED_WITH_SUGGESTIONS"]:
            break
        plan = await revise_plan(plan, audit)
        audit = await audit_mode(model="sonnet", plan=plan)

    if audit.verdict == "REJECTED":
        return await escalate_to_human("Plan 修改超限")

    blueprint = finalize_blueprint(plan, audit)

    # S2 新增：蓝图快照（强制）
    # 规则：首次立即快照；后续更新只增量生成新版本，禁止覆盖旧快照
    snapshot_path = create_blueprint_snapshot(
        project_name=derive_project_name(message),
        blueprint=blueprint,
        based_on=None,
        change_summary="S2初版DAG蓝图"
    )

    # ==================== S3: 分阶段执行 ====================
    results = {}

    for phase in blueprint.phases:
        # 并行执行同 Phase 内的独立步骤（DAG）
        phase_results = await execute_parallel_steps(
            phase=phase,
            blueprint=blueprint,
            previous_results=results,
        )

        # QA 审计每个步骤
        for step_id, result in phase_results.items():
            qa_result = await qa_audit(result, blueprint.steps[step_id])

            if qa_result.passed:
                results[step_id] = lock_artifact(result)  # 成果锁定
            else:
                # 缺陷修改循环
                result = await defect_fix_loop(
                    result, qa_result,
                    severity_rules={
                        "critical": "auto_approve",
                        "high": "auto_approve_notify_sir",
                        "medium": "sir_confirm",
                        "low": "qa_decide",
                    }
                )
                results[step_id] = lock_artifact(result)

    return TaskComplete(results=results, audit_trail=collect_audit_trail())


async def dynamic_upgrade_check(execution_context):
    """
    动态升级兜底：执行过程中检测是否需要升级到完整三步法
    """
    if (execution_context.failure_count >= 2
        or execution_context.actual_steps > execution_context.estimated_steps * 2
        or execution_context.unexpected_blockers > 0):

        new_score = await evaluate_complexity(execution_context.original_message)
        if new_score.total > 15:
            # 中途升级到完整三步法
            return await upgrade_to_full_three_step(execution_context)
```

---

## 递归嵌套：子代理也执行 S0-S3

### 核心规则

主代理通过 S2 规划后，将步骤分配给子代理执行。**子代理收到分配的任务后，也必须对自己的任务独立运行 S0-S3 评估**——因为一个在主代理视角下是"单步"的任务，到了子代理手里可能仍然是复杂的。

```
主代理 (Layer 0)
  ├─ S0-S3 评估 → 分配步骤给子代理
  │
  ├─ 子代理 A (Layer 1)
  │   ├─ S0: 预筛选自己的任务
  │   ├─ S1: 评估 → 简单? → 直接执行
  │   │              → 复杂? → S2 规划 → S3 执行
  │   │                         │
  │   │                         ├─ 子子代理 (Layer 2)
  │   │                         │   ├─ S0-S1 评估
  │   │                         │   └─ 最多再分一层 (Layer 3) ← 硬上限
  │   │                         └─ ...
  │   └─ 返回结果给主代理
  │
  └─ 子代理 B (Layer 1)
      └─ ...
```

### 嵌套深度硬上限：3 层

| 层级 | 角色 | 说明 |
|------|------|------|
| **Layer 0** | 主代理 | 接收用户任务，执行顶层 S0-S3 |
| **Layer 1** | 子代理 | 接收主代理分配的步骤，独立 S0-S3 |
| **Layer 2** | 子子代理 | 接收 Layer 1 分配的子步骤，独立 S0-S3 |
| **Layer 3** | 叶子代理 | 最深层，**禁止再向下 spawn**，必须自行完成 |

**Layer 3 的子代理在 S1 评估时，即使总分 > 15，也不得进入 S2 规划分配，而是以 "轻规划" 模式自行执行。**

### 嵌套深度传递

调度子代理时，**必须传递当前嵌套深度**：

```python
# 主代理调度子代理
sessions_spawn(
    task=f"""
    [COMPLEXITY_DEPTH=1]
    {step_description}
    
    你被分配了一个任务。请按照 complex-task-methodology 技能独立评估此任务的复杂度。
    当前嵌套深度: 1（最大允许: 3）
    如果你的 S1 评估 > 15 且深度 < 3，可以继续向下分配子代理。
    如果深度 = 3，必须自行完成，不得再 spawn。
    """,
    ...
)
```

### 防死循环机制

| 机制 | 说明 |
|------|------|
| **深度硬上限** | Layer 3 禁止再 spawn，强制自行完成 |
| **深度必须递增** | 每次 spawn 时 depth += 1，不可伪造或重置 |
| **超时保护** | 每层有独立超时，防止无限等待 |
| **任务缩减验证** | 子代理收到的任务范围必须严格小于父代理的任务范围 |

### 实际场景示例

```
用户: "从零搭建一个带用户认证的电商系统"

Layer 0 (主代理):
  S1 评分: 22 → 进入 S2
  S2 规划:
    Phase 1: 需求分析 [直接执行]
    Phase 2: 架构设计 → spawn 架构代理
    Phase 3: 前端开发 → spawn 前端代理  ┐ 并行
             后端开发 → spawn 后端代理  ┘
    Phase 4: 集成测试 → spawn QA 代理

Layer 1 (后端代理):
  收到: "实现用户认证 + 商品管理 + 订单系统"
  S1 评分: 18 → 进入 S2
  S2 规划:
    Step 1: 用户认证模块 → spawn 子代理 [DEPTH=2]
    Step 2: 商品管理模块 → spawn 子代理 [DEPTH=2]  ┐ 并行
    Step 3: 订单系统模块 → spawn 子代理 [DEPTH=2]  ┘

Layer 2 (订单系统子代理):
  收到: "实现订单 CRUD + 支付对接 + 库存扣减"
  S1 评分: 12 → 轻规划，自行执行（不再向下分配）

Layer 3: [本例未触发，但如果 Layer 2 评分 > 15 可以再分一层]
```

---

## 配置

### 可调参数

| 参数 | 默认值 | 说明 |
|------|--------|------|
| `s0_message_length_threshold` | 200 字 | 长度信号触发阈值 |
| `s1_simple_threshold` | 8 | S1 直接执行上限 |
| `s1_medium_threshold` | 15 | S1 轻规划上限，超过进入 S2 |
| `s2_max_revision_rounds` | 2 | Plan-Audit 最大修改轮数 |
| `dynamic_upgrade_failure_threshold` | 2 | 动态升级触发失败次数 |
| `plan_model` | opus | S2 规划模型 |
| `audit_model` | sonnet | S2/S3 审计模型 |

### 触发词配置

```json
{
  "intent_verbs": ["开发", "构建", "设计", "部署", "迁移", "重构", "分析", "调研", "实现", "搭建", "创建", "优化"],
  "scope_words": ["整个", "全部", "系统", "架构", "从零开始", "端到端", "完整", "全面", "整套"],
  "explicit_triggers": ["复杂任务", "三步法", "需要规划", "完整流程"]
}
```

---

## 与现有系统的关系

| 现有机制 | 关系 |
|----------|------|
| 语义路由 (S0 关键词) | S0 预筛选复用同一模式匹配思路 |
| AGENTS.md 任务执行铁律 | 三步法是其**上层框架**，铁律中的"分解→搜索→尝试"对应 S1-S3 的简化版 |
| Coding Team 流程 | 三步法的一个**特化实例**（开发领域），Coding Team 角色映射到 S3 的模型分工 |
| Heartbeat 主动工作 | Heartbeat 中的任务默认走 S0 筛选 |

---

*Created: 2026-03-01 | Version: 1.0.0*
*Origin: Sir + DeepEye collaborative design*
