#!/usr/bin/env python3
"""
Download model weight URLs into a ComfyUI installation's models folder.
Uses pget (https://github.com/replicate/pget) when available for parallel downloads;
installs pget to ~/.local/bin if missing. Falls back to Python urllib if pget
cannot be used. Reads URLs from arguments or stdin (one per line; optional
"url subfolder" per line).
"""
import argparse
import os
import platform
import re
import shutil
import subprocess
import sys
import tempfile
import urllib.request
from urllib.parse import unquote, urlparse

PGET_RELEASE = "https://github.com/replicate/pget/releases/latest/download"


# ComfyUI models subfolders (under ComfyUI/models/)
SUBFOLDERS = {
    "checkpoints",
    "clip",
    "clip_vision",
    "controlnet",
    "diffusion_models",
    "embeddings",
    "loras",
    "text_encoders",
    "unet",
    "vae",
    "vae_approx",
    "upscale_models",
    "hypernetworks",
    "photomaker",
    "style_models",
}

# Heuristics: filename fragment -> subfolder (lowercase match)
SUBFOLDER_HINTS = [
    (r"\b(vae|ae\.safetensors)\b", "vae"),
    (r"\b(clip|text_encoder|qwen|t5)\b", "text_encoders"),
    (r"\b(clip_vision)\b", "clip_vision"),
    (r"\b(lora|loras)\b", "loras"),
    (r"\b(controlnet|control_net)\b", "controlnet"),
    (r"\b(upscale|esrgan|realesrgan)\b", "upscale_models"),
    (r"\b(embedding|embeddings|ti_)\b", "embeddings"),
    (r"\b(unet)\b", "unet"),
    (r"\b(diffusion)\b", "diffusion_models"),
]


def infer_subfolder(url_or_filename: str) -> str:
    s = (url_or_filename or "").lower()
    for pattern, subfolder in SUBFOLDER_HINTS:
        if re.search(pattern, s):
            return subfolder
    return "checkpoints"


def get_pget_binary() -> str | None:
    """Return path to pget binary, or None if not found and install failed."""
    pget = shutil.which("pget")
    if pget:
        return pget
    # Install to ~/.local/bin
    local_bin = os.path.expanduser("~/.local/bin")
    pget_path = os.path.join(local_bin, "pget")
    if os.path.isfile(pget_path) and os.access(pget_path, os.X_OK):
        return pget_path
    os.makedirs(local_bin, exist_ok=True)
    sysname = platform.system()
    machine = platform.machine()
    if machine == "aarch64":
        machine = "arm64"
    elif machine == "x86_64":
        machine = "x86_64"
    # Replicate releases: pget_Linux_x86_64, pget_Darwin_arm64, etc.
    asset = f"pget_{sysname}_{machine}"
    url = f"{PGET_RELEASE}/{asset}"
    try:
        req = urllib.request.Request(url, headers={"User-Agent": "ComfyUI-Skill/1.0"})
        with urllib.request.urlopen(req, timeout=60) as resp:
            with open(pget_path, "wb") as f:
                f.write(resp.read())
        os.chmod(pget_path, 0o755)
        return pget_path
    except Exception as e:
        print(f"Could not install pget ({e}); falling back to built-in download.", file=sys.stderr)
        return None


def resolve_url_dest(
    raw: str, base: str, overwrite: bool, default_subfolder: str | None
) -> tuple[str | None, str | None, bool]:
    """Return (url, dest_path, is_skip). url None = skip/ignore."""
    url = raw.strip()
    if not url or url.startswith("#"):
        return (None, None, True)
    subfolder = default_subfolder if default_subfolder in SUBFOLDERS else "checkpoints"
    if " " in url:
        url, sub = url.strip().split(None, 1)
        sub = sub.strip().lower()
        if sub in SUBFOLDERS:
            subfolder = sub
    elif not default_subfolder or default_subfolder not in SUBFOLDERS:
        subfolder = infer_subfolder(url)
    base = os.path.expanduser(base)
    model_dir = os.path.join(base, "models", subfolder)
    os.makedirs(model_dir, exist_ok=True)
    path = urlparse(url).path
    name = path.rstrip("/").split("/")[-1]
    name = unquote(name) if name else "downloaded.safetensors"
    out_path = os.path.join(model_dir, name)
    if os.path.isfile(out_path) and not overwrite:
        return (None, out_path, True)
    return (url, out_path, False)


def download_with_pget(manifest_path: str, pget_bin: str, overwrite: bool) -> bool:
    cmd = [pget_bin, "multifile", manifest_path]
    if overwrite:
        cmd.append("-f")
    r = subprocess.run(cmd)
    return r.returncode == 0


def download_one_fallback(
    url: str, dest_path: str, overwrite: bool
) -> tuple[str, str]:
    """Download one file with urllib. Returns (status, path_or_message)."""
    if os.path.isfile(dest_path) and not overwrite:
        return ("skipped", dest_path)
    req = urllib.request.Request(url, headers={"User-Agent": "ComfyUI-Skill/1.0"})
    try:
        with urllib.request.urlopen(req, timeout=600) as resp:
            with open(dest_path, "wb") as f:
                while True:
                    chunk = resp.read(1 << 20)
                    if not chunk:
                        break
                    f.write(chunk)
        return ("ok", dest_path)
    except Exception as e:
        return ("error", str(e))


def main():
    ap = argparse.ArgumentParser(
        description="Download model weight URLs into ComfyUI models folder (uses pget when available)."
    )
    ap.add_argument(
        "urls",
        nargs="*",
        help="URLs to download (or read one per line from stdin)",
    )
    ap.add_argument(
        "--base",
        default=os.path.expanduser("~/ComfyUI"),
        help="ComfyUI install root (default: ~/ComfyUI)",
    )
    ap.add_argument(
        "--subfolder",
        default=None,
        help="Default subfolder under models/ (default: infer from URL or checkpoints)",
    )
    ap.add_argument(
        "--overwrite",
        action="store_true",
        help="Overwrite existing files",
    )
    ap.add_argument(
        "--no-pget",
        action="store_true",
        help="Do not use or install pget; use built-in download only",
    )
    args = ap.parse_args()

    base = os.path.expanduser(args.base)
    if not os.path.isdir(base):
        print(f"Error: ComfyUI base not found: {base}", file=sys.stderr)
        sys.exit(1)

    urls = list(args.urls)
    if not urls:
        for line in sys.stdin:
            urls.append(line.strip())

    default_sub = args.subfolder
    entries = []
    skipped_paths = []
    for raw in urls:
        url, dest, is_skip = resolve_url_dest(raw, base, args.overwrite, default_sub)
        if url is None and is_skip and dest:
            skipped_paths.append(dest)
            continue
        if url is None:
            continue
        entries.append((url, dest))

    if not entries:
        for p in skipped_paths:
            print(f"skipped {p}")
        print("Done: 0 downloaded,", len(skipped_paths), "skipped.", file=sys.stderr)
        return

    use_pget = not args.no_pget
    pget_bin = get_pget_binary() if use_pget else None

    if pget_bin and entries:
        with tempfile.NamedTemporaryFile(mode="w", suffix=".txt", delete=False) as f:
            for url, dest in entries:
                f.write(url + " " + dest + "\n")
            manifest_path = f.name
        try:
            if download_with_pget(manifest_path, pget_bin, args.overwrite):
                for _url, dest in entries:
                    print(dest)
                print(f"Done: {len(entries)} downloaded (pget).", file=sys.stderr)
            else:
                use_pget = False
        except Exception as e:
            print(f"pget failed ({e}); falling back to built-in download.", file=sys.stderr)
            use_pget = False
        finally:
            os.unlink(manifest_path)

    if not use_pget or not (pget_bin and entries):
        ok, skipped, err = 0, 0, 0
        for url, dest_path in entries:
            status, path_or_msg = download_one_fallback(url, dest_path, args.overwrite)
            if status == "ok":
                print(dest_path)
                ok += 1
            elif status == "skipped":
                print(f"skipped {path_or_msg}")
                skipped += 1
            else:
                print(f"error {path_or_msg}", file=sys.stderr)
                err += 1
        for p in skipped_paths:
            print(f"skipped {p}")
            skipped += 1
        if err:
            sys.exit(1)
        print(f"Done: {ok} downloaded, {skipped} skipped.", file=sys.stderr)


if __name__ == "__main__":
    main()
