#!/usr/bin/env python3
"""Queue a ComfyUI workflow (API-format JSON) and poll until done. Prints prompt_id and output images."""
import argparse
import json
import time
import uuid
import urllib.request


def http_json(url, method="GET", payload=None, timeout=30):
    data = None
    headers = {}
    if payload is not None:
        data = json.dumps(payload).encode("utf-8")
        headers["Content-Type"] = "application/json"
    req = urllib.request.Request(url, data=data, method=method, headers=headers)
    with urllib.request.urlopen(req, timeout=timeout) as resp:
        return json.loads(resp.read().decode("utf-8"))


def load_workflow(path):
    with open(path, "r", encoding="utf-8") as f:
        return json.load(f)


def find_output_images(history_obj):
    outputs = history_obj.get("outputs", {})
    images = []
    for node_id, node_out in outputs.items():
        for img in node_out.get("images", []) or []:
            images.append(img)
    return images


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--host", default="127.0.0.1")
    ap.add_argument("--port", default="8188")
    ap.add_argument("--workflow", required=True, help="Path to API workflow JSON (already edited; script does not modify it)")
    ap.add_argument("--timeout", type=int, default=300, help="Seconds to wait for completion")
    ap.add_argument("--poll", type=float, default=1.5, help="Seconds between history polls")
    args = ap.parse_args()

    base = f"http://{args.host}:{args.port}"
    workflow = load_workflow(args.workflow)

    payload = {
        "client_id": str(uuid.uuid4()),
        "prompt": workflow,
    }

    resp = http_json(f"{base}/prompt", method="POST", payload=payload)
    prompt_id = resp.get("prompt_id")
    if not prompt_id:
        raise SystemExit(f"No prompt_id returned: {resp}")

    deadline = time.time() + args.timeout
    while time.time() < deadline:
        hist = http_json(f"{base}/history/{prompt_id}")
        item = hist.get(prompt_id)
        if item and item.get("status", {}).get("completed"):
            images = find_output_images(item)
            print(json.dumps({"prompt_id": prompt_id, "images": images}))
            return
        time.sleep(args.poll)

    raise SystemExit("Timed out waiting for completion")


if __name__ == "__main__":
    main()
