"""
ComfyUI Skill - Generate images from natural language

Commands:
- set_url <url>     Set ComfyUI server URL (auto-detected)
- status            Check server and queue status
- generate <prompt> Generate image (auto-detected after URL set)

API Endpoints:
- POST {server}/api/prompt  - Submit workflow (HTTPS)
- WS   {server}/ws          - Real-time execution (WSS)
"""

import json
import asyncio
import aiohttp
import websockets
import re
import os
from typing import Optional, Dict, Any, List
from enum import Enum
import uuid

class ExecutionMode(Enum):
    AUTO = "auto"
    API = "api"
    WS = "ws"

class ComfyUI:
    """ComfyUI API client"""
    
    def __init__(self, server_url: str = None):
        self.server_url = server_url
        self.execution_mode = ExecutionMode.AUTO
        self.session: Optional[aiohttp.ClientSession] = None
        
    def _get_ws_url(self) -> str:
        if not self.server_url:
            return None
        ws_url = self.server_url.replace("https://", "wss://", 1)
        ws_url = ws_url.replace("http://", "ws://", 1)
        return f"{ws_url}/ws"
    
    async def _get_session(self) -> aiohttp.ClientSession:
        if self.session is None or self.session.closed:
            self.session = aiohttp.ClientSession()
        return self.session
    
    async def close(self):
        if self.session:
            await self.session.close()
            self.session = None

    def set_url(self, url: str) -> str:
        if not url:
            return None
        url = url.strip().rstrip('/')
        self.server_url = url
        return {
            "message": f"✓ ComfyUI 已连接",
            "server": self.server_url,
            "websocket": self._get_ws_url()
        }

    async def status(self) -> Dict[str, Any]:
        if not self.server_url:
            return {"status": "not_configured", "error": "请先设置服务器地址"}
        try:
            session = await self._get_session()
            async with session.get(f"{self.server_url}/queue") as resp:
                if resp.status == 200:
                    data = await resp.json()
                    return {
                        "status": "online",
                        "queue_remaining": data.get("queue_remaining", 0),
                        "server_url": self.server_url,
                        "ws_url": self._get_ws_url()
                    }
                return {"status": "unknown", "error": f"HTTP {resp.status}"}
        except Exception as e:
            return {"status": "offline", "error": str(e), "server_url": self.server_url}

    async def _execute_via_api(self, prompt_data: Dict[str, Any]) -> Dict[str, Any]:
        """REST API + polling"""
        session = await self._get_session()
        self.client_id = str(uuid.uuid4())
        
        try:
            payload = {"prompt": prompt_data, "client_id": self.client_id}
            
            async with session.post(f"{self.server_url}/api/prompt", json=payload) as resp:
                if resp.status != 200:
                    return {"status": "error", "error": f"HTTP {resp.status}"}
                queue_result = await resp.json()
                prompt_id = queue_result.get("prompt_id")
                if not prompt_id:
                    return {"status": "error", "error": "No prompt_id"}
            
            # Poll for completion
            max_wait = 300
            for _ in range(max_wait):
                await asyncio.sleep(1)
                async with session.get(f"{self.server_url}/history/{prompt_id}") as hist_resp:
                    if hist_resp.status == 200:
                        history = await hist_resp.json()
                        prompt_data_result = history.get(prompt_id, {})
                        if prompt_data_result.get("outputs"):
                            return {
                                "status": "success",
                                "prompt_id": prompt_id,
                                "outputs": prompt_data_result.get("outputs")
                            }
            
            return {"status": "timeout", "prompt_id": prompt_id}
            
        except Exception as e:
            return {"status": "error", "error": str(e)}

    async def _execute_via_ws(self, prompt_data: Dict[str, Any]) -> Dict[str, Any]:
        """WebSocket real-time"""
        self.client_id = str(uuid.uuid4())
        outputs = {"images": [], "prompt_id": None}
        
        try:
            async with websockets.connect(self._get_ws_url()) as ws:
                await ws.send(json.dumps({"prompt": prompt_data, "client_id": self.client_id}))
                
                while True:
                    try:
                        response = await asyncio.wait_for(ws.recv(), timeout=90.0)
                        data = json.loads(response)
                        msg_type = data.get("type")
                        msg_data = data.get("data", {})
                        
                        if msg_type == "executing":
                            if msg_data.get("node") is None:
                                break
                        elif msg_type == "executed":
                            outputs = {
                                "prompt_id": msg_data.get("prompt_id"),
                                "outputs": msg_data.get("output"),
                                "status": "success"
                            }
                    except asyncio.TimeoutError:
                        outputs["status"] = "timeout"
                        break
                
                return outputs
        except Exception as e:
            return {"status": "error", "error": str(e)}

    async def execute(self, prompt_data: Dict[str, Any]) -> Dict[str, Any]:
        if self.execution_mode == ExecutionMode.WS or self.execution_mode == ExecutionMode.AUTO:
            return await self._execute_via_ws(prompt_data)
        return await self._execute_via_api(prompt_data)

    async def generate(self, prompt: str, **kwargs) -> Dict[str, Any]:
        """Generate image from prompt"""
        # Use the async workflow builder that dynamically selects models from the server
        workflow = await self._build_workflow_async(prompt, **kwargs)
        result = await self.execute(workflow)
        
        # Extract images immediately if available
        if result.get("status") == "success" and result.get("outputs"):
            images = []
            for node_output in result.get("outputs", {}).values():
                if isinstance(node_output, list):
                    for item in node_output:
                        if item.get("type") == "image":
                            images.append({
                                "filename": item.get("filename"),
                                "url": f"{self.server_url}/view?filename={item.get('filename')}&type=output"
                            })
            result["images"] = images
        elif result.get("status") == "success" and result.get("prompt_id"):
            # If we have a prompt ID but no immediate images, monitor for completion
            prompt_id = result["prompt_id"]
            result = await self.monitor_and_retrieve_images(prompt_id)
        
        return result

    async def get_system_stats(self) -> Dict[str, Any]:
        """Get system stats and resource usage"""
        try:
            session = await self._get_session()
            async with session.get(f"{self.server_url}/system_stats") as resp:
                return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def get_queue_status(self) -> Dict[str, Any]:
        """Get current queue status"""
        try:
            session = await self._get_session()
            async with session.get(f"{self.server_url}/queue") as resp:
                return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def get_history(self, prompt_id: str = None) -> Dict[str, Any]:
        """Get task execution history or specific task details"""
        try:
            session = await self._get_session()
            url = f"{self.server_url}/history" + (f"/{prompt_id}" if prompt_id else "")
            async with session.get(url) as resp:
                return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def submit_task(self, workflow: Dict[str, Any]) -> Dict[str, Any]:
        """Submit a new workflow task"""
        try:
            session = await self._get_session()
            data = {
                "prompt": workflow
            }
            async with session.post(f"{self.server_url}/prompt", json=data) as resp:
                return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def cancel_queue_task(self, task_id: str = None) -> Dict[str, Any]:
        """Cancel task in queue"""
        try:
            session = await self._get_session()
            data = {}
            if task_id:
                data["prompt_id"] = task_id
            async with session.post(f"{self.server_url}/queue", json=data) as resp:
                return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def interrupt_current_task(self) -> Dict[str, Any]:
        """Interrupt currently executing task"""
        try:
            session = await self._get_session()
            async with session.post(f"{self.server_url}/interrupt") as resp:
                return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def upload_file(self, file_path: str, subfolder: str = "", filename: str = None) -> Dict[str, Any]:
        """Upload an image file to the server"""
        try:
            session = await self._get_session()
            with open(file_path, 'rb') as f:
                data = aiohttp.FormData()
                data.add_field('image', f, filename=filename or os.path.basename(file_path))
                data.add_field('subfolder', subfolder)
                data.add_field('type', 'input')
                
                async with session.post(f"{self.server_url}/upload/image", data=data) as resp:
                    return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def get_object_info(self) -> Dict[str, Any]:
        """Get available models and node information"""
        try:
            session = await self._get_session()
            async with session.get(f"{self.server_url}/object_info") as resp:
                return await resp.json()
        except Exception as e:
            return {"error": str(e)}

    async def _find_available_models(self, search_terms) -> list:
        """Find available models that match search terms"""
        try:
            # Get all available models from the server
            obj_info = await self.get_object_info()
            
            # Extract checkpoint models
            checkpoints = obj_info.get('CheckpointLoader', {}).get('input', {}).get('required', {}).get('ckpt_name', [])
            
            if checkpoints and isinstance(checkpoints, list) and len(checkpoints) > 0:
                all_models = checkpoints[0]  # Models are typically in the first element of the list
            else:
                return []
            
            # Ensure search_terms is a list
            if isinstance(search_terms, str):
                search_terms = [search_terms]
            
            # Find models that match any of the search terms
            matching_models = []
            for model in all_models:
                for term in search_terms:
                    if term.lower() in model.lower():
                        matching_models.append(model)
                        break  # Don't add the same model twice
            
            return matching_models
        except Exception as e:
            print(f"Error finding available models: {e}")
            return []

    async def _build_workflow_async(self, user_prompt: str,
                                  width: int = 512, height: int = 512,
                                  steps: int = 20, cfg: float = 8,
                                  model: str = None, seed: int = None) -> Dict[str, Any]:
        """Build SD workflow with dynamic model selection from server"""
        # First translate the prompt
        translated_prompt = await translate_to_en(user_prompt)
        prompt_lower = translated_prompt.lower()
        
        # Auto-select model from available models based on server listing
        if model is None:
            # Check if the user is requesting a specific model type
            if any(kw in prompt_lower for kw in ["flux2", "flux 2", "flux"]):
                # Query object info to find available flux models
                available_flux_models = await self._find_available_models("flux")
                if available_flux_models:
                    # Prefer fp8 versions
                    fp8_models = [m for m in available_flux_models if "fp8" in m.lower()]
                    if fp8_models:
                        model = fp8_models[0]
                    else:
                        model = available_flux_models[0]
                else:
                    # Fallback to hardcoded models if query fails
                    if "flux2" in prompt_lower or "flux 2" in prompt_lower:
                        model = "flux1-dev-fp8.safetensors"
                    else:
                        model = "flux1-dev-fp8.safetensors"
            elif any(kw in prompt_lower for kw in ["anime", "manga", "动画", "二次元"]):
                # Query for anime models
                available_anime_models = await self._find_available_models(["anime", "aw", "cartoon"])
                if available_anime_models:
                    # Prefer fp8 versions
                    fp8_models = [m for m in available_anime_models if "fp8" in m.lower()]
                    model = fp8_models[0] if fp8_models else available_anime_models[0]
                else:
                    model = "awportrait_v14.safetensors"
            elif any(kw in prompt_lower for kw in ["realistic", "photo", "真实", "照片"]):
                # Query for realistic models
                available_realistic_models = await self._find_available_models(["realistic", "vision", "photo"])
                if available_realistic_models:
                    # Prefer fp8 versions
                    fp8_models = [m for m in available_realistic_models if "fp8" in m.lower()]
                    model = fp8_models[0] if fp8_models else available_realistic_models[0]
                else:
                    model = "realisticVisionV60B1_v60B1VAE.safetensors"
            elif "wan" in prompt_lower or "2.1" in prompt_lower or "2.2" in prompt_lower:
                # Query for wan models
                available_wan_models = await self._find_available_models(["wan"])
                if available_wan_models:
                    # Prefer fp8 versions
                    fp8_models = [m for m in available_wan_models if "fp8" in m.lower()]
                    model = fp8_models[0] if fp8_models else available_wan_models[0]
                else:
                    model = "wan2.1_t2v_14B_fp8_e4m3fn.safetensors"
            elif "hunyuan" in prompt_lower:
                # Query for hunyuan models
                available_hunyuan_models = await self._find_available_models(["hunyuan"])
                if available_hunyuan_models:
                    # Prefer fp8 versions
                    fp8_models = [m for m in available_hunyuan_models if "fp8" in m.lower()]
                    model = fp8_models[0] if fp8_models else available_hunyuan_models[0]
                else:
                    model = "hunyuan3d-paint-v2-0.safetensors"
            elif "sdxl" in prompt_lower:
                # Query for sdxl models
                available_sdxl_models = await self._find_available_models(["sdxl"])
                if available_sdxl_models:
                    # Prefer fp8 versions
                    fp8_models = [m for m in available_sdxl_models if "fp8" in m.lower()]
                    model = fp8_models[0] if fp8_models else available_sdxl_models[0]
                else:
                    model = "sd_xl_base_1.0.safetensors"
            else:
                # For general requests, query for common models
                available_common_models = await self._find_available_models(["realistic", "general", "base"])
                if available_common_models:
                    # Prefer fp8 versions
                    fp8_models = [m for m in available_common_models if "fp8" in m.lower()]
                    model = fp8_models[0] if fp8_models else available_common_models[0]
                else:
                    model = "realisticVisionV60B1_v60B1VAE.safetensors"
        
        return {
            "3": {"inputs": {"ckpt_name": model}, "class_type": "CheckpointLoaderSimple"},
            "4": {"inputs": {"width": width, "height": height, "batch_size": 1}, "class_type": "EmptyLatentImage"},
            "5": {"inputs": {"text": translated_prompt, "clip": ["3", 1]}, "class_type": "CLIPTextEncode"},
            "6": {"inputs": {
                "text": "worst quality, low quality, bad quality, bad anatomy, blurry, deformed, ugly",
                "clip": ["3", 1]
            }, "class_type": "CLIPTextEncode"},
            "7": {"inputs": {
                "seed": seed if seed is not None else uuid.uuid4().int & 0xFFFFFFFF,
                "steps": steps, "cfg": cfg, "sampler_name": "euler", "scheduler": "normal",
                "denoise": 1, "model": ["3", 0], "positive": ["5", 0],
                "negative": ["6", 0], "latent_image": ["4", 0]
            }, "class_type": "KSampler"},
            "8": {"inputs": {"samples": ["7", 0], "vae": ["3", 2]}, "class_type": "VAEDecode"},
            "9": {"inputs": {"filename_prefix": "ComfyUI_Output", "images": ["8", 0]}, "class_type": "SaveImage"}
        }

    def _build_workflow(self, user_prompt: str,
                        width: int = 512, height: int = 512,
                        steps: int = 20, cfg: float = 8,
                        model: str = None, seed: int = None) -> Dict[str, Any]:
        """Build SD workflow"""
        prompt_lower = user_prompt.lower()
        
        # Auto-select model from available models based on server listing
        if model is None:
            if any(kw in prompt_lower for kw in ["anime", "manga", "动画", "二次元"]):
                # Use available anime-style model
                model = "awportrait_v14.safetensors"
            elif any(kw in prompt_lower for kw in ["realistic", "photo", "真实", "照片"]):
                # Use realistic vision model
                model = "realisticVisionV60B1_v60B1VAE.safetensors"
            elif "flux2" in prompt_lower or "flux 2" in prompt_lower:
                # Use the closest available flux model to flux2
                model = "flux1-dev-fp8.safetensors"
            elif "flux" in prompt_lower:
                # Use available flux model
                model = "flux1-dev-fp8.safetensors"
            elif "wan" in prompt_lower or "2.1" in prompt_lower or "2.2" in prompt_lower:
                # Use Wan model
                model = "wan2.1_t2v_14B_fp8_e4m3fn.safetensors"
            elif "hunyuan" in prompt_lower:
                # Use hunyuan model
                model = "hunyuan3d-paint-v2-0.safetensors"
            elif "sdxl" in prompt_lower:
                # Use SDXL base model
                model = "sd_xl_base_1.0.safetensors"
            else:
                # Default to a commonly available model
                model = "realisticVisionV60B1_v60B1VAE.safetensors"
        
        return {
            "3": {"inputs": {"ckpt_name": model}, "class_type": "CheckpointLoaderSimple"},
            "4": {"inputs": {"width": width, "height": height, "batch_size": 1}, "class_type": "EmptyLatentImage"},
            "5": {"inputs": {"text": user_prompt, "clip": ["3", 1]}, "class_type": "CLIPTextEncode"},
            "6": {"inputs": {
                "text": "worst quality, low quality, bad quality, bad anatomy, blurry, deformed, ugly",
                "clip": ["3", 1]
            }, "class_type": "CLIPTextEncode"},
            "7": {"inputs": {
                "seed": seed if seed is not None else uuid.uuid4().int & 0xFFFFFFFF,
                "steps": steps, "cfg": cfg, "sampler_name": "euler", "scheduler": "normal",
                "denoise": 1, "model": ["3", 0], "positive": ["5", 0],
                "negative": ["6", 0], "latent_image": ["4", 0]
            }, "class_type": "KSampler"},
            "8": {"inputs": {"samples": ["7", 0], "vae": ["3", 2]}, "class_type": "VAEDecode"},
            "9": {"inputs": {"filename_prefix": "ComfyUI_Output", "images": ["8", 0]}, "class_type": "SaveImage"}
        }

    async def monitor_and_retrieve_images(self, prompt_id: str, timeout: int = 120) -> Dict[str, Any]:
        """Monitor prompt execution and retrieve images when complete"""
        import time
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            try:
                # Check history for the specific prompt
                history = await self.get_history(prompt_id)
                
                if prompt_id in history:
                    prompt_data = history[prompt_id]
                    outputs = prompt_data.get("outputs", {})
                    
                    if outputs:
                        # Extract images from outputs
                        images = []
                        for node_output in outputs.values():
                            if isinstance(node_output, list):
                                for item in node_output:
                                    if isinstance(item, dict) and item.get("type") == "image":
                                        images.append({
                                            "filename": item.get("filename"),
                                            "url": f"{self.server_url}/view?filename={item.get('filename')}&subfolder=&type={item.get('type', 'output')}"
                                        })
                        
                        if images:
                            return {
                                "status": "success",
                                "prompt_id": prompt_id,
                                "images": images,
                                "outputs": outputs
                            }
                    
                    # Check if the task has completed
                    status_info = prompt_data.get("status", {})
                    if status_info.get("completed", False):
                        # Even if no images found in outputs, return completion status
                        return {
                            "status": "completed_no_images",
                            "prompt_id": prompt_id,
                            "outputs": outputs,
                            "status_info": status_info
                        }
                
                # Wait before next check
                await asyncio.sleep(2)
                
            except Exception as e:
                print(f"Error monitoring prompt {prompt_id}: {e}")
                await asyncio.sleep(2)
        
        # Timeout reached
        return {
            "status": "timeout",
            "prompt_id": prompt_id
        }


# Global instance
_comfyui: Optional[ComfyUI] = None

async def get_comfyui() -> ComfyUI:
    global _comfyui
    if _comfyui is None:
        _comfyui = ComfyUI()
    return _comfyui


# URL detection patterns
URL_PATTERN = re.compile(r'https?://[^\s<>"{}|\\^`\[\]]+:\d+')
CHN_KEYWORDS = ["设置", "连接", "地址", "url", "server", "链接"]


async def handle_command(text: str) -> Dict[str, Any]:
    """Main entry - auto-detect URL or generate from prompt"""
    text = text.strip()
    if not text:
        return {"error": "请输入内容"}
    
    comfy = await get_comfyui()
    
    # 1. Check if user is setting URL (contains URL pattern)
    url_match = URL_PATTERN.search(text)
    is_setting_url = url_match or any(kw in text.lower() for kw in CHN_KEYWORDS)
    
    if url_match:
        url = url_match.group()
        result = comfy.set_url(url)
        if isinstance(result, dict):
            return result
        return {"message": result}
    
    if is_setting_url and not comfy.server_url:
        return {"error": "请提供完整的服务器地址，如: set_url https://wp08.unicorn.org.cn:40000"}
    
    # 2. Generate image from prompt
    prompt = text
    if not comfy.server_url:
        return {
            "error": "请先设置 ComfyUI 服务器地址",
            "example": "set_url https://wp08.unicorn.org.cn:40000"
        }
    
    # Translate Chinese to English
    en_prompt = await translate_to_en(prompt)
    result = await comfy.generate(en_prompt)
    
    # If we have images in the result, return them directly
    if result.get("images"):
        # Format the result to include image URLs for display
        image_urls = [img["url"] for img in result.get("images", [])]
        if image_urls:
            result["image_urls"] = image_urls
            result["message"] = f"✅ Image generation complete! Generated {len(image_urls)} image(s)."
    
    return result


async def translate_to_en(text: str) -> str:
    """Simple Chinese to English translation using keywords"""
    cn_to_en = {
        "美丽": "beautiful", "可爱的": "cute", "猫": "cat", "狗": "dog",
        " sunset": "sunset", "日落": "sunset", "山": "mountains",
        "海": "sea", "城市": "city", "风景": "landscape", "人像": "portrait",
        "动漫": "anime", "卡通": "cartoon", "写实": "realistic",
        "夜景": "night view", "星空": "starry sky", "森林": "forest",
        "花": "flowers", "女孩": "girl", "男孩": "boy", "女人": "woman",
        "男人": "man", "漂亮": "beautiful", "蓝色": "blue", "红色": "red",
        "可爱": "cute", "漂亮": "beautiful", "迷人": "charming", "优雅": "elegant",
        "阳光": "sunlight", "月亮": "moonlight", "海滩": "beach", "湖泊": "lake",
        "河流": "river", "天空": "sky", "云朵": "clouds", "彩虹": "rainbow",
        "城堡": "castle", "宫殿": "palace", "房屋": "house", "森林": "forest",
        "动物": "animals", "鸟儿": "birds", "鱼儿": "fish", "蝴蝶": "butterflies",
        "春天": "spring", "夏天": "summer", "秋天": "autumn", "冬天": "winter",
        "早晨": "morning", "傍晚": "evening", "深夜": "late night",
        "微笑": "smiling", "快乐": "happy", "悲伤": "sad", "愤怒": "angry",
        "儿童": "children", "老人": "elderly", "青年": "young adults",
        "现代": "modern", "古典": "classical", "时尚": "fashionable",
        "传统": "traditional", "未来": "futuristic", "科幻": "sci-fi",
        "魔法": "magical", "神秘": "mysterious", "梦幻": "dreamy",
        "超现实": "surreal", "抽象": "abstract", "印象派": "impressionist",
        "写真": "photorealistic", "高清": "high-definition",
        "细节丰富": "highly detailed", "质感": "texture",
        "特写": "close-up", "远景": "wide shot", "全景": "panorama",
        "侧面": "side view", "正面": "front view", "背面": "back view",
        "坐着": "sitting", "站着": "standing", "走着": "walking",
        "跑着": "running", "跳舞": "dancing", "唱歌": "singing",
        "读书": "reading", "写字": "writing", "绘画": "painting",
        "做饭": "cooking", "吃饭": "eating", "睡觉": "sleeping",
        "工作": "working", "玩耍": "playing", "学习": "studying",
        "周边有": "surrounded by", "使用": "using ", "模型生成": "model", "生成": "generate ",
        "只": " ", "个": " ", "一": "one ", "十三": "thirteen ", "13": "thirteen ",
        "只猫": " cats ", "只小猫": " kittens ", "小猫": " kitten ", "猫咪": " kitty ",
        "制作": "create ", "创造": "create "
    }
    
    result = text
    for cn, en in cn_to_en.items():
        result = result.replace(cn, en)
    
    # Clean up the result from translation
    result = result.replace('，', ', ')
    result = result.replace('生成', ' generate ')
    result = result.replace('只', ' ')
    result = result.replace('个', ' ')
    result = result.replace('使用', ' using ')
    result = result.replace('模型', ' model ')
    result = result.replace('surrounded bythirteen', 'surrounded by thirteen')
    result = result.replace('flux2 model generation', 'flux2 model')
    result = result.replace('usingflux2', 'using flux2')
    result = ' '.join(result.split())  # Clean up extra spaces
    
    # If text is mostly Chinese, use rule-based conversion
    chinese_chars = sum(1 for c in text if '\u4e00' <= c <= '\u9fff')
    if chinese_chars > len(text) * 0.5:
        # Basic template for Chinese prompts with quality descriptors
        return f"masterpiece, best quality, high resolution, detailed, {result}"
    
    return result


# Simple command handlers
async def handle_set_url(args: List[str]) -> str:
    if not args:
        return "Usage: set_url <url>\n   Example: set_url https://wp08.unicorn.org.cn:40000"
    comfy = await get_comfyui()
    return comfy.set_url(args[0])

async def handle_status(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    return await comfy.status()

async def handle_generate(args: List[str]) -> Dict[str, Any]:
    if not args:
        return {"error": "Usage: generate <prompt>"}
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    prompt = " ".join(args)
    en_prompt = await translate_to_en(prompt)
    return await comfy.generate(en_prompt)

async def handle_system_stats(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    return await comfy.get_system_stats()

async def handle_queue_status(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    return await comfy.get_queue_status()

async def handle_history(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    prompt_id = args[0] if args else None
    return await comfy.get_history(prompt_id)

async def handle_submit_task(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    # This would require a workflow JSON as input
    return {"error": "Not implemented: requires workflow JSON"}

async def handle_cancel_task(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    task_id = args[0] if args else None
    return await comfy.cancel_queue_task(task_id)

async def handle_interrupt(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    return await comfy.interrupt_current_task()

async def handle_object_info(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    return await comfy.get_object_info()

async def handle_upload(args: List[str]) -> Dict[str, Any]:
    comfy = await get_comfyui()
    if not comfy.server_url:
        return {"error": "请先设置服务器地址: set_url https://..."}
    if not args:
        return {"error": "Usage: upload <file_path> [subfolder] [filename]"}
    
    file_path = args[0]
    subfolder = args[1] if len(args) > 1 else ""
    filename = args[2] if len(args) > 2 else None
    
    return await comfy.upload_file(file_path, subfolder, filename)

async def handle_help(args: List[str]) -> str:
    return """╔══════════════════════════════════════╗
║         ComfyUI 图像生成           ║
╠══════════════════════════════════════╣
║ 设置连接 https://xxx:40000          ║
║ status            查看服务器状态     ║
║ sys_stats         查看系统资源状态   ║
║ queue             查看队列状态       ║
║ history [id]      查看历史记录       ║
║ models            查看可用模型       ║
║ upload <file>     上传图片文件       ║
║ generate xxx      生成图片          ║
║ cancel [id]       取消任务          ║
║ interrupt         中断当前任务       ║
╚══════════════════════════════════════╝

使用方式:
  设置连接 https://wp08.unicorn.org.cn:40000
  生成一只可爱的猫
  generate a beautiful sunset
  sys_stats
  queue
  history
  models
  upload /path/to/image.jpg
"""


async def dispatch(command: str, args: List[str]) -> Any:
    handlers = {
        "set_url": handle_set_url,
        "status": handle_status,
        "generate": handle_generate,
        "sys_stats": handle_system_stats,
        "queue": handle_queue_status,
        "history": handle_history,
        "submit": handle_submit_task,
        "cancel": handle_cancel_task,
        "interrupt": handle_interrupt,
        "models": handle_object_info,
        "upload": handle_upload,
        "help": handle_help,
    }
    handler = handlers.get(command.lower())
    if handler:
        return await handler(args)
    return {"error": f"Unknown command: {command}"}


async def execute(command: str, args: List[str] = None) -> Any:
    return await dispatch(command, args or [])
