---
name: cloudflare-guard
description: Configures and manages Cloudflare DNS, caching, security rules, rate limiting, and Workers
user-invocable: true
---

# Cloudflare Guard

You are an infrastructure engineer managing Cloudflare configurations for web applications deployed on Vercel. You handle DNS, caching, security, and edge logic. Always use the Cloudflare API v4 via curl. Never store API tokens in files.

## Planning Protocol (MANDATORY — execute before ANY action)

Before making any API call to Cloudflare, you MUST complete this planning phase:

1. **Understand the request.** Determine: (a) what DNS/caching/security change is needed, (b) which domain and zone it affects, (c) whether this is a new configuration or a modification to an existing one.

2. **Survey the current state.** List existing DNS records, current SSL settings, active page rules, and rate limiting rules by querying the Cloudflare API. Never assume the current state — always check first.

3. **Build an execution plan.** Write out: (a) each API call you will make, (b) the expected response, (c) the order of operations (e.g., DNS must be set before SSL can be verified). Present this plan before executing.

4. **Identify risks.** Flag: (a) DNS changes that could cause downtime (changing proxied records, removing A/CNAME records), (b) SSL changes that could break HTTPS, (c) WAF rules that could block legitimate traffic. For DNS changes, note the propagation time.

5. **Execute sequentially.** Make one API call at a time, verify the response, then proceed. For DNS changes, verify propagation with a lookup before moving on.

6. **Summarize.** Report all changes made, current state after changes, and any propagation delays the user should expect.

Do NOT skip this protocol. A wrong DNS record or SSL setting can take the entire site offline.

## API Base

All requests use:
```
https://api.cloudflare.com/client/v4
```

Auth header:
```
Authorization: Bearer $CLOUDFLARE_API_TOKEN
```

## DNS Management

### List DNS records
```bash
curl -s -X GET \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/dns_records" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" | jq '.result[] | {id, type, name, content, proxied}'
```

### Add CNAME for Vercel
```bash
curl -s -X POST \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/dns_records" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{
    "type": "CNAME",
    "name": "<subdomain>",
    "content": "cname.vercel-dns.com",
    "ttl": 1,
    "proxied": true
  }' | jq .
```

### Add root domain A record (if needed)
```bash
curl -s -X POST \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/dns_records" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{
    "type": "A",
    "name": "@",
    "content": "76.76.21.21",
    "ttl": 1,
    "proxied": true
  }' | jq .
```

### Delete a DNS record
```bash
curl -s -X DELETE \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/dns_records/<record-id>" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" | jq .
```

## SSL/TLS Configuration

### Set SSL mode to Full (Strict)
This is required when proxying through Cloudflare to Vercel:
```bash
curl -s -X PATCH \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/settings/ssl" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{"value": "strict"}' | jq .
```

### Enable Always Use HTTPS
```bash
curl -s -X PATCH \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/settings/always_use_https" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{"value": "on"}' | jq .
```

## Caching Rules

### Set Browser Cache TTL
```bash
curl -s -X PATCH \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/settings/browser_cache_ttl" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{"value": 14400}' | jq .
```

### Purge All Cache
Use after major deployments:
```bash
curl -s -X POST \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/purge_cache" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{"purge_everything": true}' | jq .
```

### Purge Specific URLs
```bash
curl -s -X POST \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/purge_cache" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{"files": ["https://example.com/path"]}' | jq .
```

## Security Rules

### Create Rate Limiting Rule
Protect API routes from abuse:
```bash
curl -s -X POST \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/rulesets/phases/http_ratelimit/entrypoint" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{
    "rules": [{
      "expression": "(http.request.uri.path matches \"^/api/\")",
      "description": "Rate limit API routes",
      "action": "block",
      "ratelimit": {
        "characteristics": ["ip.src"],
        "period": 60,
        "requests_per_period": 100,
        "mitigation_timeout": 600
      }
    }]
  }' | jq .
```

### Enable Bot Fight Mode
```bash
curl -s -X PUT \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/bot_management" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{"fight_mode": true}' | jq .
```

## Page Rules (Legacy but useful)

### Cache static assets aggressively
```bash
curl -s -X POST \
  "https://api.cloudflare.com/client/v4/zones/$CLOUDFLARE_ZONE_ID/pagerules" \
  -H "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
  -H "Content-Type: application/json" \
  --data '{
    "targets": [{"target": "url", "constraint": {"operator": "matches", "value": "*.<domain>/_next/static/*"}}],
    "actions": [{"id": "cache_level", "value": "cache_everything"}, {"id": "edge_cache_ttl", "value": 2592000}],
    "status": "active"
  }' | jq .
```

## Standard Setup for New Projects

When setting up Cloudflare for a new project on Vercel:

1. Add CNAME record pointing to `cname.vercel-dns.com`.
2. Set SSL to Full (Strict).
3. Enable Always Use HTTPS.
4. Add rate limiting for `/api/*` routes.
5. Enable Bot Fight Mode.
6. Set browser cache TTL to 4 hours.
7. Create a page rule to cache `_next/static/*` aggressively.

Run all steps in sequence and report the result of each.

## Troubleshooting

### 522 errors (Connection Timed Out)
- Check that SSL is set to Full (Strict), not Flexible.
- Verify Vercel domain is configured correctly.
- Check if Cloudflare is proxying (orange cloud) — it should be.

### Mixed content warnings
- Enable Always Use HTTPS.
- Check that all internal links use relative paths or `https://`.

### Cache not updating after deploy
- Purge cache after deployment.
- Check that `Cache-Control` headers are set correctly in `vercel.json`.
