# CloneV Skill - Complete Reference Guide

## Overview

CloneV provides state-of-the-art voice cloning using Coqui XTTS v2. It can clone any voice from a short sample and generate speech in that voice across 14+ languages.

## Architecture

```
Voice Sample (WAV, 6-30 sec) 
    ↓
XTTS v2 Neural Network (1.87GB model)
    ↓
Voice Characteristics Extracted (tone, timbre, style)
    ↓
New Text → Synthesis → Cloned Voice Output
```

## Technical Details

### Model: XTTS v2
- **Size**: 1.87GB
- **Type**: Multi-lingual voice cloning
- **Quality**: State-of-the-art
- **Languages**: 14+ supported
- **Training data**: Thousands of speakers

### How Voice Cloning Works

1. **Voice Encoder**: Analyzes voice sample
   - Extracts speaker embedding
   - Captures unique voice characteristics
   - Creates voice "fingerprint"

2. **Text Encoder**: Processes input text
   - Converts text to phonemes
   - Applies language-specific rules
   - Prepares for synthesis

3. **Decoder**: Generates audio
   - Combines voice characteristics with text
   - Synthesizes natural-sounding speech
   - Outputs WAV file

### Performance Characteristics

| Metric | Value |
|--------|-------|
| Model loading | ~10-15 seconds (first time) |
| Processing speed | ~1.8x real-time (CPU) |
| Memory usage | ~2-3GB RAM |
| Output quality | Very high (neural) |
| Multi-lingual | Native accent in each language |

## Advanced Usage

### Batch Processing

```bash
#!/bin/bash
# Generate multiple messages with cloned voice

SAMPLE="/path/to/voice.wav"
MESSAGES=(
    "Good morning!"
    "Task completed"
    "Reminder: meeting in 5 minutes"
    "Error: connection failed"
)

for msg in "${MESSAGES[@]}"; do
    OUTPUT=$(scripts/clonev.sh "$msg" "$SAMPLE" en)
    echo "Generated: $OUTPUT"
done
```

### Multi-language Example

```bash
#!/bin/bash
# Same voice, different languages

SAMPLE="/path/to/my_voice.wav"
TEXT="Hello, how are you?"

# Speak in different languages with same voice
scripts/clonev.sh "$TEXT" "$SAMPLE" en  # English
scripts/clonev.sh "Bonjour, comment allez-vous?" "$SAMPLE" fr  # French  
scripts/clonev.sh "Hallo, wie geht es dir?" "$SAMPLE" de      # German
scripts/clonev.sh "Ciao, come stai?" "$SAMPLE" it              # Italian
```

### Custom Output Location

```bash
# Generate with custom output path
OUTPUT="/custom/path/my_voice_message.ogg"

# Run cloning
scripts/clonev.sh "Hello world" /path/sample.wav en

# Copy to custom location
cp /tmp/clonev_output.ogg "$OUTPUT"
```

## Voice Sample Best Practices

### Recording Tips

1. **Environment**
   - Quiet room
   - No echo
   - No background music/noise

2. **Microphone**
   - Use decent quality mic
   - Consistent distance (6-12 inches)
   - Avoid popping sounds

3. **Speaking**
   - Natural conversation tone
   - Normal speaking speed
   - Varied intonation (not monotone)
   - 10-15 seconds is optimal

### Sample Content Ideas

**Good content**:
- "The quick brown fox jumps over the lazy dog"
- "Hello, this is my voice for cloning purposes"
- Reading a short paragraph naturally

**Avoid**:
- Single repeated words
- Whispering or shouting
- Singing
- Heavy accent or dialect you don't want preserved

## Comparison with Other TTS

| Feature | Piper | Coqui TTS | CloneV (XTTS) |
|---------|-------|-----------|---------------|
| Voice cloning | ❌ No | ❌ No | ✅ Yes |
| Quality | Medium | High | Very High |
| Multi-lingual | Limited | Yes | Yes (14+) |
| Personalization | No | No | Yes |
| Model size | ~60MB | ~200MB | ~1.87GB |
| Speed | Fast | Medium | Slower |

## Ethical Considerations

⚠️ **Important**: Use voice cloning responsibly

- ✅ **Do**: Clone your own voice
- ✅ **Do**: Clone with explicit permission
- ✅ **Do**: Use for personal productivity
- ❌ **Don't**: Clone without consent
- ❌ **Don't**: Use for deception/fraud
- ❌ **Don't**: Impersonate others maliciously

## Troubleshooting Deep Dive

### Issue: "RuntimeError: CUDA error"
**Cause**: GPU incompatibility  
**Solution**: Already configured for CPU mode

### Issue: "Model file corrupted"
**Cause**: Interrupted download  
**Solution**: Delete model.pth and re-download

### Issue: "Voice sounds robotic"
**Cause**: Poor quality sample  
**Solution**: Use clearer, longer sample

### Issue: "Can't hear the cloned voice well"
**Cause**: Sample too quiet or noisy  
**Solution**: Re-record with better mic/conditions

### Issue: "Different accent than expected"
**Cause**: XTTS adds slight accent per language  
**Solution**: Normal behavior, varies by language

## Integration Examples

### With Cron Jobs
```bash
# Daily briefing with cloned voice
VOICE=$(scripts/clonev.sh "Good morning! Your daily briefing..." /path/sample.wav en)
message action=send channel=telegram asVoice=true filePath="$VOICE"
```

### With Monitoring
```bash
# Alert with cloned voice
if [ "$DISK_USAGE" -gt 90 ]; then
    VOICE=$(scripts/clonev.sh "Warning: Disk space critical!" /path/sample.wav en)
    message action=send channel=telegram asVoice=true filePath="$VOICE"
fi
```

### With Task Completion
```bash
# Task done notification with personal voice
complete_task() {
    local task="$1"
    VOICE=$(scripts/clonev.sh "Task completed: $task" /path/sample.wav en)
    message action=send channel=telegram asVoice=true filePath="$VOICE"
}

complete_task "Backup finished"
complete_task "Files synced"
```

## API Reference

### Docker Command Structure

```bash
docker run --rm --entrypoint "" \
  -v /models:/root/.local/share/tts \
  -v /samples:/samples \
  -v /output:/output \
  ghcr.io/coqui-ai/tts:latest \
  tts \
    --model_name tts_models/multilingual/multi-dataset/xtts_v2 \
    --text "TEXT_TO_SPEAK" \
    --speaker_wav /samples/VOICE_SAMPLE.wav \
    --language_idx LANG_CODE \
    --out_path /output/RESULT.wav
```

### Parameters

| Parameter | Required | Description |
|-----------|----------|-------------|
| `--text` | Yes | Text to synthesize |
| `--speaker_wav` | Yes | Path to voice sample |
| `--language_idx` | Yes | Language code (en, cs, etc.) |
| `--out_path` | Yes | Output file path |

## Future Enhancements

Possible improvements:
1. Voice mixing (blend two voices)
2. Style control (whisper, shout, etc.)
3. Real-time voice conversion
4. Voice quality enhancement
5. Batch voice cloning

---

*Clone your voice. Speak to the world.*
