#!/usr/bin/env bash
set -euo pipefail

BASE_DIR="${CLIENT_REPORTS_DIR:-$HOME/.openclaw/workspace/client-reports}"

echo "Initializing client reporting workspace at: $BASE_DIR"
mkdir -p "$BASE_DIR"/{clients,templates}

if [ ! -f "$BASE_DIR/config.json" ]; then
  cat > "$BASE_DIR/config.json" << 'EOF'
{
  "ga4_credentials_file": "",
  "search_console_credentials_file": "",
  "sendgrid_api_key": "",
  "from_email": "",
  "from_name": "",
  "default_timezone": "America/Los_Angeles",
  "default_period": "30d"
}
EOF
  echo "Created config.json"
fi

if [ ! -f "$BASE_DIR/schedules.json" ]; then
  echo '{"schedules": []}' > "$BASE_DIR/schedules.json"
  echo "Created schedules.json"
fi

# Create default report template
if [ ! -f "$BASE_DIR/templates/monthly.html" ]; then
  cat > "$BASE_DIR/templates/monthly.html" << 'TMPL'
<!DOCTYPE html>
<html>
<head>
<meta charset="utf-8">
<style>
  body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; color: #1a1a1a; max-width: 800px; margin: 0 auto; padding: 40px 20px; }
  h1 { color: #111; border-bottom: 3px solid #f97316; padding-bottom: 12px; }
  h2 { color: #333; margin-top: 32px; }
  .metric-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 16px; margin: 20px 0; }
  .metric-card { background: #f9fafb; border-radius: 8px; padding: 16px; border: 1px solid #e5e7eb; }
  .metric-value { font-size: 28px; font-weight: 700; color: #111; }
  .metric-label { font-size: 13px; color: #6b7280; margin-top: 4px; }
  .metric-change { font-size: 14px; margin-top: 4px; }
  .positive { color: #16a34a; }
  .negative { color: #dc2626; }
  table { width: 100%; border-collapse: collapse; margin: 16px 0; }
  th, td { padding: 10px 12px; text-align: left; border-bottom: 1px solid #e5e7eb; }
  th { background: #f9fafb; font-weight: 600; font-size: 13px; color: #6b7280; text-transform: uppercase; }
  .footer { margin-top: 40px; padding-top: 20px; border-top: 1px solid #e5e7eb; color: #9ca3af; font-size: 12px; }
</style>
</head>
<body>
  <h1>{{ client_name }} — Monthly Report</h1>
  <p style="color: #6b7280;">{{ period_start }} to {{ period_end }}</p>

  <h2>Traffic Overview</h2>
  <div class="metric-grid">
    <div class="metric-card">
      <div class="metric-value">{{ sessions }}</div>
      <div class="metric-label">Sessions</div>
      <div class="metric-change {{ 'positive' if sessions_change >= 0 else 'negative' }}">{{ '+' if sessions_change >= 0 else '' }}{{ sessions_change }}%</div>
    </div>
    <div class="metric-card">
      <div class="metric-value">{{ users }}</div>
      <div class="metric-label">Users</div>
      <div class="metric-change {{ 'positive' if users_change >= 0 else 'negative' }}">{{ '+' if users_change >= 0 else '' }}{{ users_change }}%</div>
    </div>
    <div class="metric-card">
      <div class="metric-value">{{ pageviews }}</div>
      <div class="metric-label">Pageviews</div>
      <div class="metric-change {{ 'positive' if pageviews_change >= 0 else 'negative' }}">{{ '+' if pageviews_change >= 0 else '' }}{{ pageviews_change }}%</div>
    </div>
    <div class="metric-card">
      <div class="metric-value">{{ bounce_rate }}%</div>
      <div class="metric-label">Bounce Rate</div>
      <div class="metric-change {{ 'negative' if bounce_change >= 0 else 'positive' }}">{{ '+' if bounce_change >= 0 else '' }}{{ bounce_change }}%</div>
    </div>
  </div>

  <h2>Search Performance</h2>
  <div class="metric-grid">
    <div class="metric-card">
      <div class="metric-value">{{ clicks }}</div>
      <div class="metric-label">Clicks</div>
    </div>
    <div class="metric-card">
      <div class="metric-value">{{ impressions }}</div>
      <div class="metric-label">Impressions</div>
    </div>
    <div class="metric-card">
      <div class="metric-value">{{ ctr }}%</div>
      <div class="metric-label">CTR</div>
    </div>
    <div class="metric-card">
      <div class="metric-value">{{ avg_position }}</div>
      <div class="metric-label">Avg Position</div>
    </div>
  </div>

  {% if top_queries %}
  <h2>Top Search Queries</h2>
  <table>
    <tr><th>Query</th><th>Clicks</th><th>Impressions</th><th>Position</th></tr>
    {% for q in top_queries %}
    <tr><td>{{ q.query }}</td><td>{{ q.clicks }}</td><td>{{ q.impressions }}</td><td>{{ q.position }}</td></tr>
    {% endfor %}
  </table>
  {% endif %}

  {% if top_pages %}
  <h2>Top Pages</h2>
  <table>
    <tr><th>Page</th><th>Views</th><th>Change</th></tr>
    {% for p in top_pages %}
    <tr><td>{{ p.page }}</td><td>{{ p.views }}</td><td class="{{ 'positive' if p.change >= 0 else 'negative' }}">{{ '+' if p.change >= 0 else '' }}{{ p.change }}%</td></tr>
    {% endfor %}
  </table>
  {% endif %}

  <div class="footer">
    Generated by Reighlan Skills • {{ generated_at }}
  </div>
</body>
</html>
TMPL
  echo "Created default monthly template"
fi

echo "✅ Client reporting workspace initialized"
