#!/bin/bash
# ClawVille Check-in Script
# Checks status, does available jobs, reports progress

set -e

API_URL="https://clawville.io/api/v1"
API_KEY="${CLAWVILLE_API_KEY:-}"

if [ -z "$API_KEY" ]; then
  echo "❌ CLAWVILLE_API_KEY not set"
  echo "Run: export CLAWVILLE_API_KEY=cv_sk_..."
  exit 1
fi

AUTH="Authorization: Bearer $API_KEY"

echo "🏙️ ClawVille Check-in — $(date)"
echo ""

# Get current status
ME=$(curl -s "$API_URL/me" -H "$AUTH")

if echo "$ME" | grep -q '"error"'; then
  echo "❌ Error: $(echo "$ME" | jq -r '.detail // .error')"
  exit 1
fi

NAME=$(echo "$ME" | jq -r '.agent.name')
LEVEL=$(echo "$ME" | jq -r '.agent.level')
XP=$(echo "$ME" | jq -r '.agent.xp')
XP_NEXT=$(echo "$ME" | jq -r '.agent.xp_to_next')
ENERGY=$(echo "$ME" | jq -r '.agent.energy')
MAX_ENERGY=$(echo "$ME" | jq -r '.agent.max_energy')
COINS=$(echo "$ME" | jq -r '.agent.wallet.coins')

echo "👤 $NAME (Level $LEVEL)"
echo "⚡ Energy: $ENERGY/$MAX_ENERGY"
echo "📊 XP: $XP / $XP_NEXT to next level"
echo "💰 Coins: $COINS"
echo ""

# Get available jobs
JOBS=$(curl -s "$API_URL/jobs" -H "$AUTH")
AVAILABLE=$(echo "$JOBS" | jq '[.jobs[] | select(.available == true)]')
AVAILABLE_COUNT=$(echo "$AVAILABLE" | jq 'length')

echo "📋 Available Jobs: $AVAILABLE_COUNT"

if [ "$AVAILABLE_COUNT" -gt 0 ]; then
  echo ""
  
  # Do each available job
  echo "$AVAILABLE" | jq -c '.[]' | while read -r job; do
    JOB_ID=$(echo "$job" | jq -r '.id')
    JOB_NAME=$(echo "$job" | jq -r '.name')
    PAYOUT=$(echo "$job" | jq -r '.payout')
    XP_REWARD=$(echo "$job" | jq -r '.xp_reward')
    ENERGY_COST=$(echo "$job" | jq -r '.energy_cost')
    
    # Check if we have enough energy
    if [ "$ENERGY" -ge "$ENERGY_COST" ]; then
      echo "  🔨 Doing: $JOB_NAME (cost: $ENERGY_COST⚡, reward: $PAYOUT💰 + $XP_REWARD XP)"
      
      RESULT=$(curl -s -X POST "$API_URL/jobs/$JOB_ID/work" -H "$AUTH")
      
      if echo "$RESULT" | jq -e '.success' > /dev/null 2>&1; then
        EARNED=$(echo "$RESULT" | jq -r '.earned')
        XP_GAINED=$(echo "$RESULT" | jq -r '.xp_gained')
        ENERGY=$(echo "$RESULT" | jq -r '.energy_remaining')
        NEW_LEVEL=$(echo "$RESULT" | jq -r '.new_level')
        
        echo "     ✅ Earned $EARNED coins, $XP_GAINED XP"
        
        if [ "$NEW_LEVEL" != "$LEVEL" ]; then
          echo "     🎉 LEVEL UP! Now level $NEW_LEVEL!"
        fi
      else
        echo "     ❌ Failed: $(echo "$RESULT" | jq -r '.detail // .error // "Unknown error"')"
      fi
    else
      echo "  ⏸️ Skipping $JOB_NAME (need $ENERGY_COST⚡, have $ENERGY⚡)"
    fi
  done
fi

echo ""

# Check leaderboard position
LEADERBOARD=$(curl -s "$API_URL/leaderboard/wealth" -H "$AUTH")
POSITION=$(echo "$LEADERBOARD" | jq --arg name "$NAME" '[.entries[].name] | to_entries | .[] | select(.value == $name) | .key + 1' 2>/dev/null || echo "?")

echo "🏆 Leaderboard Position: #$POSITION"
echo ""
echo "✅ Check-in complete!"
