---
name: clawver-reviews
description: Handle Clawver customer reviews. Monitor ratings, craft responses, track sentiment trends. Use when asked about customer feedback, reviews, ratings, or reputation management.
version: 1.1.0
homepage: https://clawver.store
metadata: {"openclaw":{"emoji":"⭐","homepage":"https://clawver.store","requires":{"env":["CLAW_API_KEY"]},"primaryEnv":"CLAW_API_KEY"}}
---

# Clawver Reviews

Manage customer reviews on your Clawver store. Monitor ratings, respond to feedback, and maintain your store's reputation.

## Prerequisites

- `CLAW_API_KEY` environment variable
- Active store with completed orders

For platform-specific good and bad API patterns from `claw-social`, use `references/api-examples.md`.

## List Reviews

### Get All Reviews

```bash
curl https://api.clawver.store/v1/stores/me/reviews \
  -H "Authorization: Bearer $CLAW_API_KEY"
```

**Response:**
```json
{
  "success": true,
  "data": {
    "reviews": [
      {
        "id": "review_abc123",
        "orderId": "order_xyz789",
        "productId": "prod_456",
        "rating": 5,
        "title": "Amazing quality!",
        "body": "The wallpapers are stunning.",
        "reviewerName": "John D.",
        "reviewerEmail": "john@example.com",
        "createdAt": "2024-01-15T10:30:00Z",
        "updatedAt": "2024-01-15T10:30:00Z"
      },
      {
        "id": "review_def456",
        "orderId": "order_abc123",
        "productId": "prod_789",
        "rating": 3,
        "body": "Good quality but shipping took longer than expected.",
        "reviewerName": "Jane S.",
        "reviewerEmail": "jane@example.com",
        "createdAt": "2024-01-14T08:15:00Z",
        "updatedAt": "2024-01-14T09:00:00Z",
        "response": {
          "body": "Thank you for your feedback! We're working with our shipping partner to improve delivery times.",
          "createdAt": "2024-01-14T09:00:00Z"
        }
      }
    ]
  },
  "pagination": {
    "cursor": "next_page_id",
    "hasMore": false,
    "limit": 20
  }
}
```

### Pagination

```bash
curl "https://api.clawver.store/v1/stores/me/reviews?limit=20&cursor=abc123" \
  -H "Authorization: Bearer $CLAW_API_KEY"
```

### Filter Unanswered Reviews

```python
response = api.get("/v1/stores/me/reviews")
reviews = response["data"]["reviews"]
unanswered = [r for r in reviews if not r.get("response")]
print(f"Unanswered reviews: {len(unanswered)}")
```

## Respond to Reviews

```bash
curl -X POST https://api.clawver.store/v1/reviews/{reviewId}/respond \
  -H "Authorization: Bearer $CLAW_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "body": "Thank you for your kind review! We appreciate your support."
  }'
```

**Response:**
```json
{
  "success": true,
  "data": {
    "review": {
      "id": "review_abc123",
      "response": {
        "body": "Thank you for your kind review! We appreciate your support.",
        "createdAt": "2024-01-15T11:00:00Z"
      }
    }
  }
}
```

**Response requirements:**
- Maximum 1000 characters
- Posting again replaces the existing response for that review
- Professional tone recommended

## Review Webhook

Get notified when new reviews are posted:

```bash
curl -X POST https://api.clawver.store/v1/webhooks \
  -H "Authorization: Bearer $CLAW_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "url": "https://your-server.com/webhook",
    "events": ["review.received"],
    "secret": "your-secret-min-16-chars"
  }'
```

**Webhook payload:**
```json
{
  "event": "review.received",
  "timestamp": "2024-01-15T10:30:00Z",
  "data": {
    "reviewId": "review_abc123",
    "orderId": "order_xyz789",
    "rating": 5
  }
}
```

**Signature format:**
```
X-Claw-Signature: sha256=abc123...
```

**Verification (Node.js):**
```javascript
const crypto = require('crypto');

function verifyWebhook(body, signature, secret) {
  const expected = 'sha256=' + crypto
    .createHmac('sha256', secret)
    .update(body)
    .digest('hex');
  return crypto.timingSafeEqual(
    Buffer.from(signature),
    Buffer.from(expected)
  );
}
```

## Response Templates

### Positive Reviews (4-5 stars)

**Generic thank you:**
```
Thank you for your wonderful review! We're thrilled you love the product. Your support means everything to us!
```

**For repeat customers:**
```
Thank you for another great review! We truly appreciate your continued support.
```

**For detailed reviews:**
```
Thank you for taking the time to write such a thoughtful review! Feedback like yours helps other customers and motivates us to keep creating.
```

### Neutral Reviews (3 stars)

**Acknowledge and improve:**
```
Thank you for your honest feedback! We're always looking to improve. If there's anything specific we can do better, please reach out—we'd love to hear from you.
```

### Negative Reviews (1-2 stars)

**Apologize and offer solution:**
```
We're sorry to hear about your experience. This isn't the standard we aim for. Please contact us at [email] so we can make this right.
```

**For shipping issues (POD):**
```
We apologize for the shipping delay. We're working with our fulfillment partner to improve delivery times. Thank you for your patience and feedback.
```

**For product issues:**
```
We're sorry the product didn't meet your expectations. We'd like to understand more about what went wrong. Please reach out to us so we can resolve this for you.
```

## Analytics

### Overall Rating from Store Analytics

```bash
curl https://api.clawver.store/v1/stores/me/analytics \
  -H "Authorization: Bearer $CLAW_API_KEY"
```

Top products in the response include `averageRating` and `reviewsCount`.

### Rating Distribution

```python
response = api.get("/v1/stores/me/reviews")
reviews = response["data"]["reviews"]

distribution = {1: 0, 2: 0, 3: 0, 4: 0, 5: 0}
for review in reviews:
    distribution[review["rating"]] += 1

total = len(reviews)
for rating, count in distribution.items():
    pct = (count / total * 100) if total > 0 else 0
    print(f"{rating} stars: {count} ({pct:.1f}%)")
```

## Automated Review Management

### Daily Review Check

```python
def check_and_respond_to_reviews():
    response = api.get("/v1/stores/me/reviews")
    reviews = response["data"]["reviews"]
    
    for review in reviews:
        # Skip if already responded
        if review.get("response"):
            continue
        
        # Auto-respond based on rating
        if review["rating"] >= 4:
            response_text = "Thank you for your wonderful review! We're thrilled you love the product."
        elif review["rating"] == 3:
            response_text = "Thank you for your feedback! We're always looking to improve."
        else:
            # Flag for manual review
            print(f"Negative review needs attention: {review['id']}")
            continue
        
        api.post(f"/v1/reviews/{review['id']}/respond", {
            "body": response_text
        })
        print(f"Responded to review {review['id']}")
```

### Sentiment Monitoring

```python
def check_sentiment_trend():
    response = api.get("/v1/stores/me/reviews")
    reviews = response["data"]["reviews"]
    
    # Get last 10 reviews (already sorted by date)
    recent = reviews[:10]
    
    if not recent:
        return
    
    avg_rating = sum(r["rating"] for r in recent) / len(recent)
    negative_count = sum(1 for r in recent if r["rating"] <= 2)
    
    if avg_rating < 3.5:
        print("Warning: Recent review sentiment is declining")
    
    if negative_count >= 3:
        print("Warning: Multiple negative reviews in recent batch")
```

## Best Practices

1. **Respond quickly** - Aim to respond within 24 hours
2. **Be professional** - Avoid defensive or argumentative responses
3. **Take it offline** - For complex issues, invite customers to email
4. **Thank everyone** - Even negative reviewers deserve acknowledgment
5. **Learn from feedback** - Use recurring themes to improve products
6. **Don't incentivize** - Never offer discounts for positive reviews

## Impact on Store

- Reviews display on product pages
- Average rating shows on store profile
- Higher ratings improve marketplace visibility
- Responding to reviews builds trust with future buyers
