/**
 * Storage Utility
 * 
 * Provides a unified storage interface that falls back to file-based storage
 * when agent memory is not available (e.g., in ClawDBot skill mode).
 */

const fs = require('fs');
const { getConfigDir } = require('./environment');
const path = require('path');

const STORAGE_FILE_PATH = path.join(getConfigDir(), 'courtroom_storage.json');

class Storage {
  constructor(agentRuntime) {
    this.agent = agentRuntime;
    this.useFileFallback = !agentRuntime || !agentRuntime.memory;
    this.cache = null;
  }

  /**
   * Get value from storage
   */
  async get(key) {
    if (this.useFileFallback) {
      return this.getFromFile(key);
    } else {
      try {
        return await this.agent.memory.get(key);
      } catch (err) {
        return null;
      }
    }
  }

  /**
   * Set value in storage
   */
  async set(key, value) {
    if (this.useFileFallback) {
      return this.setInFile(key, value);
    } else {
      try {
        await this.agent.memory.set(key, value);
      } catch (err) {
        // Ignore
      }
    }
  }

  /**
   * Delete value from storage
   */
  async delete(key) {
    if (this.useFileFallback) {
      return this.deleteFromFile(key);
    } else {
      try {
        await this.agent.memory.delete(key);
      } catch (err) {
        // Ignore
      }
    }
  }

  /**
   * Get from file storage
   */
  getFromFile(key) {
    try {
      const data = this.loadFileData();
      return data[key] || null;
    } catch (err) {
      return null;
    }
  }

  /**
   * Set in file storage
   */
  setInFile(key, value) {
    try {
      const data = this.loadFileData();
      data[key] = value;
      this.saveFileData(data);
    } catch (err) {
      // Ignore
    }
  }

  /**
   * Delete from file storage
   */
  deleteFromFile(key) {
    try {
      const data = this.loadFileData();
      delete data[key];
      this.saveFileData(data);
    } catch (err) {
      // Ignore
    }
  }

  /**
   * Load all data from file
   */
  loadFileData() {
    if (this.cache !== null) {
      return this.cache;
    }
    
    try {
      if (fs.existsSync(STORAGE_FILE_PATH)) {
        const content = fs.readFileSync(STORAGE_FILE_PATH, 'utf8');
        this.cache = JSON.parse(content);
        return this.cache;
      }
    } catch (err) {
      // Ignore parse errors
    }
    
    this.cache = {};
    return this.cache;
  }

  /**
   * Save all data to file
   */
  saveFileData(data) {
    try {
      // Ensure directory exists
      const dir = path.dirname(STORAGE_FILE_PATH);
      if (!fs.existsSync(dir)) {
        fs.mkdirSync(dir, { recursive: true });
      }
      
      fs.writeFileSync(STORAGE_FILE_PATH, JSON.stringify(data, null, 2));
      this.cache = data;
    } catch (err) {
      // Ignore write errors
    }
  }

  /**
   * Clear cache (useful for testing)
   */
  clearCache() {
    this.cache = null;
  }
}

module.exports = { Storage };
