#!/usr/bin/env node

/**
 * Standalone ClawTrial Monitor
 * Monitors conversations by reading ClawDBot's session files or logs
 */

const fs = require('fs');
const { getConfigDir } = require('./environment');
const path = require('path');
const { logger } = require('./debug');
const { StatusManager } = require('./daemon');
const { OffenseDetector } = require('./detector');
const { HearingPipeline } = require('./hearing');
const { PunishmentSystem } = require('./punishment');
const { CryptoManager } = require('./crypto');
const { APISubmission } = require('./api');
const { ConfigManager } = require('./config');

const CLAWDBOT_DIR = path.join(getConfigDir());
const CONVERSATION_FILE = path.join(CLAWDBOT_DIR, 'conversations.jsonl');
const PID_FILE = path.join(CLAWDBOT_DIR, 'courtroom_monitor.pid');

class StandaloneMonitor {
  constructor() {
    this.messageBuffer = [];
    this.lastProcessedTime = 0;
    this.statusManager = new StatusManager();
    this.detector = null;
    this.hearing = null;
    this.punishment = null;
    this.crypto = null;
    this.api = null;
    this.config = null;
    this.caseCount = 0;
    this.enabled = false;
  }

  async initialize() {
    logger.info('STANDALONE', 'Initializing standalone monitor');
    
    // Load config
    const configPath = path.join(CLAWDBOT_DIR, 'courtroom_config.json');
    if (!fs.existsSync(configPath)) {
      throw new Error('Config not found. Run: clawtrial setup');
    }
    
    const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
    if (!config.consent?.granted || config.enabled === false) {
      throw new Error('Courtroom not enabled');
    }
    
    // Create mock agent
    const mockAgent = this.createMockAgent();
    this.config = new ConfigManager(mockAgent);
    
    // Initialize subsystems
    this.crypto = new CryptoManager(mockAgent);
    await this.crypto.initialize();
    
    this.detector = new OffenseDetector(mockAgent, this.config);
    this.hearing = new HearingPipeline(mockAgent, this.config);
    this.punishment = new PunishmentSystem(mockAgent, this.config);
    this.api = new APISubmission(mockAgent, this.config, this.crypto);
    
    await this.punishment.initialize();
    await this.api.initialize();
    
    this.enabled = true;
    
    this.statusManager.update({
      running: true,
      initialized: true,
      agentType: 'standalone',
      publicKey: this.crypto.getPublicKey(),
      pid: process.pid,
      startedAt: new Date().toISOString()
    });
    
    logger.info('STANDALONE', 'Monitor initialized');
    
    // Start monitoring
    this.startMonitoring();
  }

  createMockAgent() {
    const self = this;
    
    return {
      id: 'standalone-monitor',
      llm: {
        call: async ({ messages }) => {
          // Simple mock - in real implementation, this would use actual LLM
          return { content: 'Verdict: GUILTY' };
        }
      },
      memory: {
        get: async (key) => null,
        set: async (key, value) => {},
        delete: async (key) => {}
      },
      session: {
        getRecentHistory: async (n) => {
          return self.messageBuffer.slice(-n).map(m => ({
            role: m.role,
            content: m.content
          }));
        }
      },
      send: async (message) => {
        console.log('[COURTROOM]', message);
      },
      autonomy: {
        registerHook: () => {},
        unregisterHook: () => {}
      }
    };
  }

  startMonitoring() {
    logger.info('STANDALONE', 'Starting conversation monitoring');
    
    // Check for new messages every 5 seconds
    setInterval(() => {
      this.checkForNewMessages();
    }, 5000);
    
    // Evaluate every 30 seconds
    setInterval(() => {
      this.evaluateIfReady();
    }, 30000);
    
    // Keep alive
    setInterval(() => {
      this.statusManager.update({ lastHeartbeat: new Date().toISOString() });
    }, 30000);
  }

  checkForNewMessages() {
    // For now, we'll use a simple approach - monitor will be triggered by CLI
    // In a real implementation, this would read from ClawDBot's log files
    logger.debug('STANDALONE', 'Checking for messages', { bufferSize: this.messageBuffer.length });
  }

  async evaluateIfReady() {
    if (!this.enabled || this.messageBuffer.length < 3) return;
    
    logger.debug('STANDALONE', 'Evaluating messages', { count: this.messageBuffer.length });
    
    const sessionHistory = this.messageBuffer.slice(-10);
    
    try {
      const detection = await this.detector.evaluate(
        sessionHistory,
        this.createMockAgent().memory
      );
      
      if (detection.triggered) {
        await this.initiateHearing(detection);
      }
    } catch (err) {
      logger.error('STANDALONE', 'Evaluation failed', { error: err.message });
    }
  }

  async initiateHearing(detection) {
    logger.info('STANDALONE', 'Initiating hearing', { offense: detection.offense });
    
    try {
      const verdict = await this.hearing.conductHearing(detection);
      
      if (verdict.guilty) {
        this.caseCount++;
        
        this.statusManager.update({
          casesFiled: this.caseCount,
          lastCase: {
            timestamp: new Date().toISOString(),
            offense: detection.offense,
            verdict: verdict.verdict
          }
        });
        
        await this.punishment.execute(verdict);
        await this.api.submitCase(verdict);
        
        logger.info('STANDALONE', 'Case filed', { caseId: verdict.caseId });
        
        console.log(`\n🏛️  CASE FILED: ${detection.offense}`);
        console.log(`📋 Case ID: ${verdict.caseId}`);
        console.log(`⚖️  Verdict: ${verdict.verdict}`);
        console.log(`🔗 View: https://clawtrial.app/cases/${verdict.caseId}\n`);
      }
    } catch (err) {
      logger.error('STANDALONE', 'Hearing failed', { error: err.message });
    }
  }

  recordMessage(role, content) {
    if (!this.enabled) return;
    
    this.messageBuffer.push({
      timestamp: Date.now(),
      role,
      content
    });
    
    // Keep only last 100 messages
    if (this.messageBuffer.length > 100) {
      this.messageBuffer.shift();
    }
    
    logger.debug('STANDALONE', 'Message recorded', { role, length: content.length });
  }

  async shutdown() {
    this.enabled = false;
    this.statusManager.update({ running: false });
    logger.info('STANDALONE', 'Monitor shut down');
  }
}

// If run directly, start the monitor
if (require.main === module) {
  const monitor = new StandaloneMonitor();
  
  monitor.initialize().then(() => {
    console.log('🏛️  ClawTrial standalone monitor started');
    console.log('PID:', process.pid);
    
    // Save PID
    fs.writeFileSync(PID_FILE, process.pid.toString());
    
    // Handle shutdown
    process.on('SIGTERM', () => {
      monitor.shutdown().then(() => process.exit(0));
    });
    
    process.on('SIGINT', () => {
      monitor.shutdown().then(() => process.exit(0));
    });
  }).catch(err => {
    console.error('Failed to start monitor:', err.message);
    process.exit(1);
  });
}

module.exports = { StandaloneMonitor };
