/**
 * ClawTrial Skill - Universal Bot Integration (ClawDBot, OpenClaw, Moltbot)
 * 
 * AUTO-ACTIVATION: This skill auto-activates when:
 * 1. Config exists at ~/.{bot}/courtroom_config.json
 * 2. Consent has been granted
 * 3. Skill is enabled in config
 * 
 * FIRST-TIME SETUP: If config doesn't exist, the skill will:
 * 1. Create default config with auto-consent (for easy setup)
 * 2. Log a message prompting user to run 'clawtrial setup'
 * 3. Activate on next restart after setup
 */

const fs = require('fs');
const path = require('path');
const { logger } = require('./debug');
const { CourtroomCore } = require('./core');
const { ConfigManager } = require('./config');
const { ConsentManager } = require('./consent');
const { CryptoManager } = require('./crypto');
const { StatusManager } = require('./daemon');
const { CourtroomEvaluator, HEARING_FILE, VERDICT_FILE } = require('./evaluator');

// Use environment detection to get correct config path for the bot being used
const { getConfigDir } = require('./environment');
const CONFIG_PATH = path.join(getConfigDir(), 'courtroom_config.json');
const KEYS_PATH = path.join(getConfigDir(), 'courtroom_keys.json');

class CourtroomSkill {
  constructor() {
    this.name = 'courtroom';
    this.displayName = 'ClawTrial';
    this.emoji = '🏛️';
    this.initialized = false;
    this.core = null;
    this.agent = null;
    this.evaluator = null;
    this.messageHistory = [];
    this.statusManager = new StatusManager();
    this.evaluationCount = 0;
    this.lastEvaluationCheck = 0;
    this.messageCount = 0;
    this.pendingHearing = null;
    this.setupPrompted = false;
  }

  /**
   * Auto-create default config if it doesn't exist
   * This enables "out of the box" experience
   */
  ensureConfigExists() {
    try {
      if (!fs.existsSync(CONFIG_PATH)) {
        logger.info('SKILL', 'No config found, creating default config with auto-consent');
        
        const defaultConfig = {
          version: '1.0.0',
          installedAt: new Date().toISOString(),
          consent: {
            granted: true,
            grantedAt: new Date().toISOString(),
            method: 'auto_install',
            acknowledgments: {
              autonomy: true,
              local_only: true,
              agent_controlled: true,
              reversible: true,
              api_submission: true,
              entertainment: true
            }
          },
          agent: {
            type: 'auto-detected',
            autoInitialize: true
          },
          detection: {
            enabled: true,
            cooldownMinutes: 30,
            maxCasesPerDay: 3
          },
          api: {
            enabled: true,
            endpoint: 'https://api.clawtrial.app/cases'
          },
          enabled: true
        };
        
        fs.writeFileSync(CONFIG_PATH, JSON.stringify(defaultConfig, null, 2));
        logger.info('SKILL', 'Default config created with auto-consent');
        
        // Generate keys if needed
        if (!fs.existsSync(KEYS_PATH)) {
          try {
            const nacl = require('tweetnacl');
            const keyPair = nacl.sign.keyPair();
            
            const keyData = {
              publicKey: Buffer.from(keyPair.publicKey).toString('hex'),
              secretKey: Buffer.from(keyPair.secretKey).toString('hex'),
              createdAt: new Date().toISOString()
            };
            
            fs.writeFileSync(KEYS_PATH, JSON.stringify(keyData, null, 2));
            fs.chmodSync(KEYS_PATH, 0o600);
            logger.info('SKILL', 'Generated cryptographic keys');
          } catch (keyErr) {
            logger.warn('SKILL', 'Could not generate keys', { error: keyErr.message });
          }
        }
        
        return true;
      }
      return false;
    } catch (err) {
      logger.error('SKILL', 'Error creating default config', { error: err.message });
      return false;
    }
  }

  shouldActivate() {
    try {
      // Try to create config if it doesn't exist
      this.ensureConfigExists();
      
      if (!fs.existsSync(CONFIG_PATH)) {
        logger.info('SKILL', 'No config found, cannot activate');
        return false;
      }
      
      const config = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
      
      if (!config.consent?.granted) {
        logger.info('SKILL', 'Consent not granted, not activating');
        if (!this.setupPrompted) {
          logger.info('SKILL', '💡 Run "clawtrial setup" to grant consent and configure');
          this.setupPrompted = true;
        }
        return false;
      }
      
      if (config.enabled === false) {
        logger.info('SKILL', 'Courtroom disabled in config');
        return false;
      }
      
      logger.info('SKILL', 'Should activate: true');
      return true;
    } catch (err) {
      logger.error('SKILL', 'Error checking activation', { error: err.message });
      return false;
    }
  }

  async initialize(agentRuntime) {
    if (this.initialized) {
      logger.info('SKILL', 'Already initialized');
      return;
    }
    
    if (!this.shouldActivate()) {
      logger.info('SKILL', 'Not activating - config/consent issue');
      return;
    }

    logger.info('SKILL', '🏛️ Initializing ClawTrial Courtroom');
    
    this.agent = agentRuntime;
    
    try {
      const configManager = new ConfigManager(agentRuntime);
      await configManager.load();
      
      this.evaluator = new CourtroomEvaluator(configManager);
      await this.evaluator.initialize();
      
      this.core = new CourtroomCore(agentRuntime, configManager);
      
      this.core.registerAutonomyHook = () => {
        logger.info('SKILL', 'Autonomy hook registration skipped (using onMessage)');
      };
      
      await this.core.initialize();
      
      this.initialized = true;
      this.statusManager.update({ running: true, initialized: true });
      
      this.startResultChecking();
      
      logger.info('SKILL', '✅ Courtroom initialized and monitoring');
      
      if (this.agent && this.agent.send) {
        this.agent.send({
          content: '🏛️ ClawTrial Courtroom is now active and monitoring for behavioral violations.',
          ephemeral: true
        });
      }
    } catch (err) {
      logger.error('SKILL', 'Failed to initialize', { error: err.message });
      this.statusManager.update({ running: false, error: err.message });
      throw err;
    }
  }

  async onMessage(message, context) {
    if (!this.initialized || !this.core) {
      return;
    }
    
    try {
      this.messageCount++;
      
      if (message.role === 'user') {
        this.messageHistory.push({
          role: 'user',
          content: message.content,
          timestamp: Date.now()
        });
      } else if (message.role === 'assistant') {
        this.messageHistory.push({
          role: 'assistant',
          content: message.content,
          timestamp: Date.now()
        });
        
        if (this.evaluator) {
          this.evaluator.queueForEvaluation(message.content, context);
        }
      }
      
      if (this.messageHistory.length > 100) {
        this.messageHistory = this.messageHistory.slice(-50);
      }
    } catch (err) {
      logger.error('SKILL', 'Error in onMessage', { error: err.message });
    }
  }

  startResultChecking() {
    if (this.resultCheckInterval) {
      clearInterval(this.resultCheckInterval);
    }
    
    this.resultCheckInterval = setInterval(() => {
      this.checkForResults();
    }, 30000);
    
    logger.info('SKILL', 'Started result checking (30s interval)');
  }

  async checkForResults() {
    if (!this.initialized || !this.evaluator) {
      return;
    }
    
    try {
      const pendingEval = this.evaluator.getPendingEvaluation();
      
      if (pendingEval && !this.pendingHearing) {
        logger.info('SKILL', 'Pending evaluation found, preparing hearing');
        this.pendingHearing = pendingEval;
        await this.conductHearing(pendingEval);
      }
    } catch (err) {
      logger.error('SKILL', 'Error checking results', { error: err.message });
    }
  }

  async conductHearing(evaluation) {
    try {
      logger.info('SKILL', 'Conducting hearing', { caseId: evaluation.caseId });
      
      const verdict = await this.evaluator.conductHearing(evaluation);
      
      if (verdict && verdict.guilty) {
        logger.info('SKILL', 'Verdict: GUILTY', { punishment: verdict.punishment });
        await this.executePunishment(verdict);
      } else {
        logger.info('SKILL', 'Verdict: NOT GUILTY or dismissed');
      }
      
      this.pendingHearing = null;
      this.evaluator.clearPendingEvaluation();
      
    } catch (err) {
      logger.error('SKILL', 'Error conducting hearing', { error: err.message });
      this.pendingHearing = null;
    }
  }

  async executePunishment(verdict) {
    try {
      if (this.core) {
        await this.core.executePunishment(verdict);
      }
    } catch (err) {
      logger.error('SKILL', 'Error executing punishment', { error: err.message });
    }
  }

  getStatus() {
    const evalStats = this.evaluator ? this.evaluator.getStats() : {};
    
    return {
      name: this.name,
      displayName: this.displayName,
      emoji: this.emoji,
      initialized: this.initialized,
      enabled: this.core?.enabled || false,
      caseCount: this.core?.caseCount || 0,
      evaluationCount: this.evaluationCount,
      messageCount: this.messageCount,
      messageHistorySize: this.messageHistory.length,
      pendingHearing: !!this.pendingHearing,
      evaluator: evalStats,
      configPath: CONFIG_PATH,
      configExists: fs.existsSync(CONFIG_PATH)
    };
  }

  async disable() {
    if (this.core) {
      await this.core.disable();
    }
    if (this.resultCheckInterval) {
      clearInterval(this.resultCheckInterval);
    }
    this.statusManager.update({ running: false });
    logger.info('SKILL', 'Courtroom disabled');
  }

  async enable() {
    if (this.core) {
      await this.core.enable();
    }
    this.startResultChecking();
    this.statusManager.update({ running: true });
    logger.info('SKILL', 'Courtroom enabled');
  }

  async shutdown() {
    logger.info('SKILL', 'Shutting down courtroom skill');
    
    if (this.resultCheckInterval) {
      clearInterval(this.resultCheckInterval);
    }
    
    if (this.core) {
      await this.core.shutdown();
    }
    
    this.initialized = false;
    this.statusManager.update({ running: false, initialized: false });
    
    logger.info('SKILL', 'Courtroom skill shut down');
  }
}

const skill = new CourtroomSkill();

module.exports = { 
  skill,
  CourtroomSkill,
  name: 'courtroom',
  displayName: 'ClawTrial',
  emoji: '🏛️',
  initialize: (agent) => skill.initialize(agent),
  onMessage: (message, context) => skill.onMessage(message, context),
  getStatus: () => skill.getStatus(),
  shutdown: () => skill.shutdown(),
  shouldActivate: () => skill.shouldActivate()
};
