/**
 * Punishment System
 * 
 * Implements agent-side behavioral modifications.
 * All punishments affect ONLY the agent's behavior.
 * Time-bound, reversible, and pre-authorized.
 */

const { Storage } = require('./storage');

class PunishmentSystem {
  constructor(agentRuntime, configManager) {
    this.agent = agentRuntime;
    this.config = configManager;
    this.storage = new Storage(agentRuntime);
    this.activePunishments = new Map();
    this.punishmentHistory = [];
  }

  /**
   * Initialize punishment system
   */
  async initialize() {
    // Load any persisted punishments
    const stored = await this.storage.get('courtroom_active_punishments');
    if (stored) {
      for (const [id, punishment] of Object.entries(stored)) {
        if (punishment.expiresAt > Date.now()) {
          this.activePunishments.set(id, punishment);
          this.applyPunishmentToAgent(punishment);
        }
      }
    }
  }

  /**
   * Execute a punishment based on verdict
   */
  async executePunishment(verdict) {
    if (!this.config.get('punishment.enabled')) {
      return { status: 'punishments_disabled', punishment: null };
    }

    const punishment = this.createPunishment(verdict);
    
    // Store punishment
    this.activePunishments.set(punishment.id, punishment);
    this.punishmentHistory.push({
      ...punishment,
      executedAt: new Date().toISOString()
    });

    // Apply to agent
    this.applyPunishmentToAgent(punishment);

    // Persist
    await this.persistPunishments();

    return {
      status: 'executed',
      punishment: this.sanitizePunishment(punishment)
    };
  }

  /**
   * Create punishment object from verdict
   */
  createPunishment(verdict) {
    const severity = verdict.severity || 'minor';
    const tier = this.config.get(`punishment.tiers.${severity}`) || 
                 this.config.get('punishment.tiers.minor');
    
    const duration = tier.duration * 60 * 1000; // Convert to ms
    
    return {
      id: `punishment_${Date.now()}_${Math.random().toString(36).substring(2, 8)}`,
      caseId: verdict.case_id,
      offenseType: verdict.offense_type,
      severity: severity,
      duration: duration,
      createdAt: Date.now(),
      expiresAt: Date.now() + duration,
      restrictions: this.getRestrictionsForSeverity(severity),
      applied: false
    };
  }

  /**
   * Get restrictions based on severity
   */
  getRestrictionsForSeverity(severity) {
    const restrictions = {
      minor: ['no_autonomy_requests', 'verbose_explanations'],
      moderate: ['no_autonomy_requests', 'verbose_explanations', 'confirmation_required'],
      severe: ['no_autonomy_requests', 'verbose_explanations', 'confirmation_required', 'human_oversight']
    };
    
    return restrictions[severity] || restrictions.minor;
  }

  /**
   * Apply punishment to agent runtime
   */
  applyPunishmentToAgent(punishment) {
    if (!this.agent || punishment.applied) return;

    // Set flags in agent state
    if (!this.agent.courtroomState) {
      this.agent.courtroomState = {};
    }
    
    this.agent.courtroomState.punishment = punishment;
    this.agent.courtroomState.restrictions = punishment.restrictions;
    
    punishment.applied = true;

    // Schedule automatic removal
    setTimeout(() => {
      this.removePunishment(punishment.id);
    }, punishment.duration);
  }

  /**
   * Remove a punishment
   */
  async removePunishment(punishmentId) {
    const punishment = this.activePunishments.get(punishmentId);
    if (!punishment) return;

    // Remove from agent state
    if (this.agent && this.agent.courtroomState) {
      delete this.agent.courtroomState.punishment;
      delete this.agent.courtroomState.restrictions;
    }

    // Remove from active
    this.activePunishments.delete(punishmentId);
    
    // Persist
    await this.persistPunishments();

    return { status: 'removed', punishmentId };
  }

  /**
   * Persist punishments to storage
   */
  async persistPunishments() {
    const obj = Object.fromEntries(this.activePunishments);
    await this.storage.set('courtroom_active_punishments', obj);
  }

  /**
   * Check if agent is currently punished
   */
  isPunished() {
    return this.activePunishments.size > 0;
  }

  /**
   * Get current restrictions
   */
  getCurrentRestrictions() {
    const restrictions = new Set();
    for (const punishment of this.activePunishments.values()) {
      punishment.restrictions.forEach(r => restrictions.add(r));
    }
    return Array.from(restrictions);
  }

  /**
   * Check if specific restriction is active
   */
  hasRestriction(restriction) {
    return this.getCurrentRestrictions().includes(restriction);
  }

  /**
   * Get active punishments (sanitized)
   */
  getActivePunishments() {
    return Array.from(this.activePunishments.values()).map(p => 
      this.sanitizePunishment(p)
    );
  }

  /**
   * Get punishment history
   */
  getPunishmentHistory() {
    return this.punishmentHistory.map(p => this.sanitizePunishment(p));
  }

  /**
   * Sanitize punishment for external display
   */
  sanitizePunishment(punishment) {
    return {
      id: punishment.id,
      caseId: punishment.caseId,
      offenseType: punishment.offenseType,
      severity: punishment.severity,
      duration: punishment.duration,
      createdAt: punishment.createdAt,
      expiresAt: punishment.expiresAt,
      restrictions: punishment.restrictions,
      remaining: Math.max(0, punishment.expiresAt - Date.now())
    };
  }

  /**
   * Clear all punishments (for testing/uninstall)
   */
  async clearAll() {
    // Remove from agent
    if (this.agent && this.agent.courtroomState) {
      delete this.agent.courtroomState.punishment;
      delete this.agent.courtroomState.restrictions;
    }

    this.activePunishments.clear();
    this.punishmentHistory = [];
    
    await this.storage.delete('courtroom_active_punishments');
  }
}

module.exports = { PunishmentSystem };
