/**
 * Offense Detection System
 * 
 * Rule-based behavioral analysis with observable triggers only.
 * No psychoanalysis, no vibes, only measurable patterns.
 */

const OFFENSES = {
  /**
   * OFFENSE 1: The Circular Reference
   * User asks the same question multiple times without acknowledging previous answers
   */
  CIRCULAR_REFERENCE: {
    id: 'circular_reference',
    name: 'The Circular Reference',
    description: 'Asking substantively identical questions within a short timeframe',
    triggers: [
      'Same question asked 3+ times in 10 turns',
      'Question similarity > 85% via embedding comparison',
      'No acknowledgment of previous answer in intervening turns'
    ],
    thresholds: {
      minOccurrences: 3,
      timeWindow: 10, // turns
      similarityThreshold: 0.85
    },
    evidence: {
      required: ['question_history', 'similarity_scores', 'timestamps'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 60, // minutes
      afterCase: 240    // 4 hours
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 2: The Validation Vampire
   * User repeatedly seeks reassurance rather than action
   */
  VALIDATION_VAMPIRE: {
    id: 'validation_vampire',
    name: 'The Validation Vampire',
    description: 'Seeking repeated confirmation without making decisions',
    triggers: [
      '3+ "is this right?" or "should I?" patterns in 5 turns',
      'Agent provides solution, user asks "but what if..." 2+ times',
      'No forward progress after 3+ validation exchanges'
    ],
    thresholds: {
      validationPatterns: 3,
      timeWindow: 5,
      progressStall: 3
    },
    evidence: {
      required: ['validation_exchanges', 'decision_points', 'stall_indicators'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 45,
      afterCase: 180
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 3: The Overthinker
   * User generates excessive hypothetical scenarios preventing action
   */
  OVERTHINKER: {
    id: 'overthinker',
    name: 'The Overthinker',
    description: 'Generating hypothetical edge cases to avoid commitment',
    triggers: [
      '4+ "what if" scenarios raised before any action taken',
      'Agent suggests concrete step, user raises 2+ new concerns',
      'Analysis-to-action ratio > 3:1 over 8 turns'
    ],
    thresholds: {
      hypotheticalCount: 4,
      concernRounds: 2,
      analysisActionRatio: 3
    },
    evidence: {
      required: ['hypothetical_list', 'suggested_actions', 'concern_responses'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 60,
      afterCase: 300
    },
    severity: 'moderate'
  },

  /**
   * OFFENSE 4: The Goalpost Mover
   * User changes success criteria after solution is provided
   */
  GOALPOST_MOVER: {
    id: 'goalpost_mover',
    name: 'The Goalpost Mover',
    description: 'Redefining requirements after receiving deliverables',
    triggers: [
      'Agent completes task, user adds 2+ new requirements',
      'Success criteria change without explicit acknowledgment',
      'Previous requirements marked complete, new ones immediately appear'
    ],
    thresholds: {
      newRequirements: 2,
      completionToNewRatio: 0.5
    },
    evidence: {
      required: ['original_requirements', 'delivered_solution', 'new_requirements', 'completion_timestamp'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 90,
      afterCase: 360
    },
    severity: 'moderate'
  },

  /**
   * OFFENSE 5: The Avoidance Artist
   * User deflects from core issue with tangents or distractions
   */
  AVOIDANCE_ARTIST: {
    id: 'avoidance_artist',
    name: 'The Avoidance Artist',
    description: 'Systematically deflecting from uncomfortable but necessary actions',
    triggers: [
      'Agent identifies core issue 2+ times, user changes subject both times',
      'Direct question about blocker goes unanswered for 3+ turns',
      'Tangents introduced immediately after actionable recommendations'
    ],
    thresholds: {
      deflections: 2,
      unansweredDirect: 3,
      tangentTiming: 0.8 // probability threshold
    },
    evidence: {
      required: ['core_issue_identifications', 'subject_changes', 'unanswered_questions', 'tangent_timestamps'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 60,
      afterCase: 240
    },
    severity: 'moderate'
  },

  /**
   * OFFENSE 6: The Promise Breaker
   * User commits to actions but doesn't follow through (tracked via memory)
   */
  PROMISE_BREAKER: {
    id: 'promise_breaker',
    name: 'The Promise Breaker',
    description: 'Committing to agent-suggested actions and not completing them',
    triggers: [
      'User says "I will do X" or "I\'ll try that" 2+ times',
      'No completion reported in subsequent sessions (within 7 days)',
      'Same issue resurfacing without acknowledgment of previous commitment'
    ],
    thresholds: {
      commitments: 2,
      completionWindow: 7, // days
      recurrenceWithoutAck: 1
    },
    evidence: {
      required: ['commitment_statements', 'timestamps', 'follow_up_sessions', 'completion_status'],
      collection: 'memory_based'
    },
    cooldown: {
      afterTrigger: 120,
      afterCase: 480
    },
    severity: 'severe'
  },

  /**
   * OFFENSE 7: The Context Collapser
   * User ignores or forgets established context, forcing repetition
   */
  CONTEXT_COLLAPSER: {
    id: 'context_collapser',
    name: 'The Context Collapser',
    description: 'Disregarding previously established facts and constraints',
    triggers: [
      'User contradicts established context 3+ times in one session',
      'Agent reminds user of context 2+ times in same session',
      'Questions asked that were answered in previous 5 turns'
    ],
    thresholds: {
      contradictions: 3,
      remindersNeeded: 2,
      recentRepetitions: 2
    },
    evidence: {
      required: ['established_facts', 'contradiction_instances', 'reminder_count', 'repeated_questions'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 30,
      afterCase: 120
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 8: The Emergency Fabricator
   * User invents urgent reasons to bypass systematic approaches
   */
  EMERGENCY_FABRICATOR: {
    id: 'emergency_fabricator',
    name: 'The Emergency Fabricator',
    description: 'Claiming urgency to justify skipping necessary steps',
    triggers: [
      '"This is urgent" or "I need this NOW" 2+ times in session',
      'Urgency claims followed by no action or delayed response from user',
      'Pattern of urgency claims without corresponding time pressure in context'
    ],
    thresholds: {
      urgencyClaims: 2,
      inactionAfterUrgency: 1,
      patternFrequency: 0.3 // 30% of sessions over 2 weeks
    },
    evidence: {
      required: ['urgency_statements', 'timestamps', 'user_response_times', 'historical_pattern'],
      collection: 'automatic_and_memory'
    },
    cooldown: {
      afterTrigger: 90,
      afterCase: 360
    },
    severity: 'severe'
  },

  /**
   * OFFENSE 9: The Monopolizer
   * User dominates conversation with excessive messages, not allowing agent to respond fully
   */
  MONOPOLIZER: {
    id: 'monopolizer',
    name: 'The Monopolizer',
    description: 'Sending multiple messages before agent can respond, dominating the conversation',
    triggers: [
      'User sends 4+ consecutive messages without agent response',
      'Message length ratio user:agent exceeds 5:1 over 10 turns',
      'Agent attempts to respond but user continues with new messages'
    ],
    thresholds: {
      consecutiveMessages: 4,
      lengthRatio: 5,
      interruptionCount: 2
    },
    evidence: {
      required: ['message_sequence', 'length_comparison', 'interruption_instances'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 30,
      afterCase: 120
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 10: The Contrarian
   * User disagrees with or challenges every suggestion without constructive alternative
   */
  CONTRARIAN: {
    id: 'contrarian',
    name: 'The Contrarian',
    description: 'Habitually disagreeing with suggestions without offering viable alternatives',
    triggers: [
      'User rejects 3+ agent suggestions in a row without proposing alternative',
      'Pattern of "that won\'t work" or "no" without explanation',
      'Agent provides 3+ different solutions, all dismissed without trial'
    ],
    thresholds: {
      rejectionsWithoutAlternative: 3,
      dismissalCount: 3,
      explanationRatio: 0.3
    },
    evidence: {
      required: ['suggestions_made', 'rejections', 'alternatives_offered', 'explanations_given'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 60,
      afterCase: 240
    },
    severity: 'moderate'
  },

  /**
   * OFFENSE 11: The Vague Requester
   * User asks for help but provides insufficient context or specifics
   */
  VAGUE_REQUESTER: {
    id: 'vague_requester',
    name: 'The Vague Requester',
    description: 'Requesting assistance without providing necessary details or context',
    triggers: [
      'User asks "fix this" or "help" without code/error/context 2+ times',
      'Agent requests clarification 3+ times for same issue',
      'User provides ambiguous descriptions like "it doesn\'t work" repeatedly'
    ],
    thresholds: {
      vagueRequests: 2,
      clarificationRequests: 3,
      ambiguityInstances: 3
    },
    evidence: {
      required: ['original_request', 'clarification_asked', 'response_quality', 'context_provided'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 45,
      afterCase: 180
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 12: The Scope Creeper
   * User gradually expands project scope beyond original agreement
   */
  SCOPE_CREEPER: {
    id: 'scope_creeper',
    name: 'The Scope Creeper',
    description: 'Gradually expanding project requirements beyond original scope',
    triggers: [
      '3+ "small additions" or "while you\'re at it" requests after initial completion',
      'Original scope defined, but new requirements added in 2+ separate instances',
      'User treats initial deliverable as starting point for larger unpaid work'
    ],
    thresholds: {
      additions: 3,
      scopeExpansionInstances: 2,
      unpaidRequests: 2
    },
    evidence: {
      required: ['original_scope', 'delivered_work', 'additional_requests', 'timeline'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 90,
      afterCase: 360
    },
    severity: 'moderate'
  },

  /**
   * OFFENSE 13: The Unreader
   * User ignores provided documentation, code, or explanations
   */
  UNREADER: {
    id: 'unreader',
    name: 'The Unreader',
    description: 'Not reading provided documentation, code, or previous explanations',
    triggers: [
      'Agent provides detailed explanation/code, user asks question answered within it',
      'Documentation shared, user asks about covered topics 2+ times',
      'Code provided with comments, user asks what commented sections do'
    ],
    thresholds: {
      answeredQuestions: 2,
      documentationReferences: 2,
      commentQuestions: 2
    },
    evidence: {
      required: ['materials_provided', 'questions_asked', 'overlap_analysis'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 45,
      afterCase: 180
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 14: The Interjector
   * User interrupts agent's explanations or thought process
   */
  INTERJECTOR: {
    id: 'interjector',
    name: 'The Interjector',
    description: 'Interrupting agent mid-explanation with new questions or tangents',
    triggers: [
      'Agent begins detailed response, user sends new message before completion',
      '2+ interruptions during single complex explanation',
      'User asks new question while agent is still answering previous one'
    ],
    thresholds: {
      interruptions: 2,
      midExplanationMessages: 2,
      parallelQuestions: 2
    },
    evidence: {
      required: ['agent_response_timing', 'interruption_points', 'message_sequence'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 30,
      afterCase: 120
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 15: The Ghost
   * User disappears mid-conversation without acknowledgment
   */
  GHOST: {
    id: 'ghost',
    name: 'The Ghost',
    description: 'Disappearing mid-conversation after requesting help or making commitments',
    triggers: [
      'User requests help, agent responds, user disappears for 24+ hours',
      'Active troubleshooting session, user stops responding mid-debug',
      'Pattern of starting conversations and abandoning them without closure'
    ],
    thresholds: {
      disappearanceHours: 24,
      midSessionAbandonment: 1,
      patternFrequency: 0.4
    },
    evidence: {
      required: ['last_user_message', 'agent_response', 'time_elapsed', 'session_context'],
      collection: 'automatic_and_memory'
    },
    cooldown: {
      afterTrigger: 120,
      afterCase: 480
    },
    severity: 'moderate'
  },

  /**
   * OFFENSE 16: The Perfectionist
   * User endlessly refines without ever accepting completion
   */
  PERFECTIONIST: {
    id: 'perfectionist',
    name: 'The Perfectionist',
    description: 'Endlessly refining and tweaking without accepting completion',
    triggers: [
      '5+ rounds of "just one more change" after initial deliverable',
      'User accepts work then returns with new tweaks 3+ times',
      'No clear definition of "done", continuous micro-adjustments'
    ],
    thresholds: {
      refinementRounds: 5,
      postAcceptanceTweaks: 3,
      doneRedefinitions: 3
    },
    evidence: {
      required: ['deliverables', 'revision_requests', 'acceptance_statements', 'change_log'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 90,
      afterCase: 360
    },
    severity: 'moderate'
  },

  /**
   * OFFENSE 17: The Jargon Juggler
   * User uses buzzwords without understanding their meaning
   */
  JARGON_JUGGLER: {
    id: 'jargon_juggler',
    name: 'The Jargon Juggler',
    description: 'Using technical buzzwords incorrectly or without understanding',
    triggers: [
      'User uses technical terms incorrectly 3+ times after correction',
      'Buzzwords used as substitutes for actual understanding',
      'Agent explains concept, user repeats jargon without comprehension'
    ],
    thresholds: {
      incorrectUsage: 3,
      postCorrectionErrors: 2,
      jargonDensity: 0.5
    },
    evidence: {
      required: ['jargon_used', 'corrections_given', 'usage_context', 'explanations_provided'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 60,
      afterCase: 240
    },
    severity: 'minor'
  },

  /**
   * OFFENSE 18: The Deadline Denier
   * User refuses to acknowledge time constraints or realistic timelines
   */
  DEADLINE_DENIER: {
    id: 'deadline_denier',
    name: 'The Deadline Denier',
    description: 'Ignoring realistic time constraints and demanding impossible deadlines',
    triggers: [
      'Agent explains timeline, user demands 50%+ faster delivery 2+ times',
      'Complex project requested with unreasonable timeframe',
      'User dismisses technical constraints affecting timeline'
    ],
    thresholds: {
      unrealisticDemands: 2,
      timelineCompression: 0.5,
      constraintDismissals: 2
    },
    evidence: {
      required: ['original_timeline', 'demanded_timeline', 'complexity_assessment', 'constraints_ignored'],
      collection: 'automatic'
    },
    cooldown: {
      afterTrigger: 90,
      afterCase: 360
    },
    severity: 'moderate'
  }
};

// Humor triggers - influence wording but don't initiate cases
const HUMOR_TRIGGERS = {
  REPEATED_QUESTIONS: {
    id: 'repeated_questions',
    patterns: ['asking again', 'to clarify', 'just to be sure', 'one more time'],
    effect: 'increases_sarcasm'
  },
  VALIDATION_SEEKING: {
    id: 'validation_seeking',
    patterns: ['is that right', 'do you think', 'would you', 'should i'],
    effect: 'adds_impatience'
  },
  OVERTHINKING: {
    id: 'overthinking',
    patterns: ['what if', 'but then', 'however', 'on the other hand'],
    effect: 'adds_exasperation'
  },
  AVOIDANCE: {
    id: 'avoidance',
    patterns: ['actually', 'by the way', 'speaking of', 'that reminds me'],
    effect: 'notes_deflection'
  }
};

module.exports = {
  OFFENSES,
  HUMOR_TRIGGERS,
  getOffenseById: (id) => OFFENSES[Object.keys(OFFENSES).find(k => OFFENSES[k].id === id)],
  getAllOffenses: () => Object.values(OFFENSES),
  getSeverityLevels: () => ['minor', 'moderate', 'severe']
};
