#!/usr/bin/env node

/**
 * ClawTrial Background Monitor
 * Runs continuously and initializes courtroom when agent becomes available
 */

const fs = require('fs');
const { getConfigDir } = require('./environment');
const path = require('path');
const { logger } = require('./debug');
const { StatusManager } = require('./daemon');

const CONFIG_PATH = path.join(getConfigDir(), 'courtroom_config.json');
const CHECK_INTERVAL = 5000; // Check every 5 seconds
const MAX_RETRIES = 100; // Give up after ~8 minutes

let retryCount = 0;
let courtroom = null;
let statusManager = new StatusManager();

logger.info('MONITOR', 'Background monitor started', { pid: process.pid });

// Save PID
statusManager.update({
  running: true,
  initialized: false,
  pid: process.pid,
  startedAt: new Date().toISOString()
});

/**
 * Check if config exists and is enabled
 */
function checkConfig() {
  try {
    if (!fs.existsSync(CONFIG_PATH)) {
      return { valid: false, reason: 'no_config' };
    }
    
    const config = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
    
    if (!config.consent?.granted) {
      return { valid: false, reason: 'no_consent' };
    }
    
    if (config.enabled === false) {
      return { valid: false, reason: 'disabled' };
    }
    
    return { valid: true, config };
  } catch (err) {
    logger.error('MONITOR', 'Config check failed', { error: err.message });
    return { valid: false, reason: 'error' };
  }
}

/**
 * Detect agent runtime
 */
function detectAgent() {
  // Check various global locations
  if (typeof global.clawdbotAgent !== 'undefined' && global.clawdbotAgent) {
    return { type: 'clawdbot', agent: global.clawdbotAgent };
  }
  
  if (typeof global.agent !== 'undefined' && global.agent) {
    return { type: 'generic', agent: global.agent };
  }
  
  if (typeof process.clawdbotAgent !== 'undefined' && process.clawdbotAgent) {
    return { type: 'clawdbot', agent: process.clawdbotAgent };
  }
  
  // Check for ClawDBot specific indicators
  if (process.env.CLAUDBOT_ENV === 'true') {
    // Try to find agent in parent process or global scope
    if (globalThis.clawdbotAgent) {
      return { type: 'clawdbot', agent: globalThis.clawdbotAgent };
    }
  }
  
  return null;
}

/**
 * Try to initialize courtroom
 */
async function tryInitialize() {
  const configCheck = checkConfig();
  
  if (!configCheck.valid) {
    logger.warn('MONITOR', 'Config not valid', { reason: configCheck.reason });
    return false;
  }
  
  const agentInfo = detectAgent();
  
  if (!agentInfo) {
    logger.debug('MONITOR', 'Agent not detected yet');
    return false;
  }
  
  logger.info('MONITOR', `Agent detected: ${agentInfo.type}`);
  
  try {
    // Import and initialize courtroom
    const { createCourtroom } = require('./index');
    courtroom = createCourtroom(agentInfo.agent);
    
    const result = await courtroom.initialize();
    
    if (result.status === 'initialized') {
      logger.info('MONITOR', 'Courtroom initialized successfully');
      
      statusManager.update({
        running: true,
        initialized: true,
        agentType: agentInfo.type,
        publicKey: result.publicKey
      });
      
      // Attach to global for access
      global.courtroom = courtroom;
      
      console.log('\n🏛️  AI Courtroom is now active and monitoring!\n');
      
      return true;
    } else {
      logger.warn('MONITOR', 'Courtroom not initialized', { status: result.status });
      return false;
    }
  } catch (err) {
    logger.error('MONITOR', 'Initialization failed', { error: err.message });
    return false;
  }
}

/**
 * Main monitoring loop
 */
async function monitor() {
  logger.info('MONITOR', 'Starting monitoring loop');
  
  // Try immediately first
  if (await tryInitialize()) {
    return; // Success! Keep process alive
  }
  
  // Set up interval to keep trying
  const interval = setInterval(async () => {
    retryCount++;
    
    if (retryCount > MAX_RETRIES) {
      logger.warn('MONITOR', 'Max retries reached, giving up');
      clearInterval(interval);
      statusManager.update({ running: false, error: 'max_retries' });
      process.exit(1);
    }
    
    if (await tryInitialize()) {
      clearInterval(interval);
      // Keep process alive - courtroom is running
    }
  }, CHECK_INTERVAL);
  
  // Keep process alive
  setInterval(() => {
    // Heartbeat to keep process alive
    if (courtroom && courtroom.enabled) {
      statusManager.update({ lastHeartbeat: new Date().toISOString() });
    }
  }, 30000);
}

// Handle shutdown
process.on('SIGTERM', () => {
  logger.info('MONITOR', 'Received SIGTERM, shutting down');
  statusManager.update({ running: false });
  process.exit(0);
});

process.on('SIGINT', () => {
  logger.info('MONITOR', 'Received SIGINT, shutting down');
  statusManager.update({ running: false });
  process.exit(0);
});

// Start monitoring
monitor().catch(err => {
  logger.error('MONITOR', 'Monitor crashed', { error: err.message });
  statusManager.update({ running: false, error: err.message });
  process.exit(1);
});
