/**
 * ClawTrial Hook - Direct integration with ClawDBot
 * This module patches into ClawDBot's message processing
 */

const fs = require('fs');
const { getConfigDir } = require('./environment');
const path = require('path');
const { logger } = require('./debug');
const { CourtroomCore } = require('./core');
const { ConfigManager } = require('./config');
const { ConsentManager } = require('./consent');
const { CryptoManager } = require('./crypto');
const { StatusManager } = require('./daemon');

const CONFIG_PATH = path.join(getConfigDir(), 'courtroom_config.json');

class ClawTrialHook {
  constructor() {
    this.initialized = false;
    this.core = null;
    this.statusManager = new StatusManager();
    this.messageBuffer = [];
    this.evaluationTimer = null;
  }

  /**
   * Check if we should activate
   */
  shouldActivate() {
    try {
      if (!fs.existsSync(CONFIG_PATH)) {
        return false;
      }
      
      const config = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
      
      if (!config.consent?.granted) {
        return false;
      }
      
      if (config.enabled === false) {
        return false;
      }
      
      return true;
    } catch (err) {
      return false;
    }
  }

  /**
   * Initialize the hook
   */
  async initialize() {
    if (this.initialized) return;
    
    if (!this.shouldActivate()) {
      logger.info('HOOK', 'ClawTrial not activated - config/consent issue');
      return;
    }

    logger.info('HOOK', 'Initializing ClawTrial hook');
    
    // Create minimal agent interface for the core
    const mockAgent = this.createMockAgent();
    
    const configManager = new ConfigManager(mockAgent);
    const consentManager = new ConsentManager(mockAgent, configManager);
    
    // Initialize crypto
    const crypto = new CryptoManager(mockAgent);
    await crypto.initialize();
    
    // Initialize core
    this.core = new CourtroomCore(mockAgent, configManager);
    
    // Override the core's registerAutonomyHook since we handle it differently
    this.core.registerAutonomyHook = () => {
      logger.info('HOOK', 'Autonomy hook registered (via message interception)');
    };
    
    await this.core.initialize();
    
    // Start message evaluation loop
    this.startEvaluationLoop();
    
    this.initialized = true;
    
    this.statusManager.update({
      running: true,
      initialized: true,
      agentType: 'clawdbot_hook',
      publicKey: crypto.getPublicKey()
    });
    
    logger.info('HOOK', 'ClawTrial hook initialized successfully');
    console.log('\n🏛️  ClawTrial is monitoring conversations\n');
  }

  /**
   * Create a minimal agent interface
   */
  createMockAgent() {
    const self = this;
    
    return {
      id: 'clawdbot-hook',
      llm: {
        call: async ({ messages }) => {
          // Use the actual ClawDBot's LLM if available
          if (global.clawdbotAgent?.llm) {
            return global.clawdbotAgent.llm.call({ messages });
          }
          return { content: 'Mock response' };
        }
      },
      memory: {
        get: async (key) => null,
        set: async (key, value) => {},
        delete: async (key) => {}
      },
      session: {
        getRecentHistory: async (n) => {
          // Return recent messages from buffer
          return self.messageBuffer.slice(-n).map(m => ({
            role: m.from === 'user' ? 'user' : 'assistant',
            content: m.text
          }));
        }
      },
      send: async (message) => {
        console.log('[COURTROOM]', message);
      },
      autonomy: {
        registerHook: () => {},
        unregisterHook: () => {}
      }
    };
  }

  /**
   * Start the evaluation loop
   */
  startEvaluationLoop() {
    // Evaluate every 30 seconds
    this.evaluationTimer = setInterval(() => {
      this.evaluateIfReady();
    }, 30000);
  }

  /**
   * Evaluate offenses if we have enough messages
   */
  async evaluateIfReady() {
    if (!this.core || !this.core.enabled) return;
    if (this.messageBuffer.length < 3) return; // Need at least 3 messages
    
    this.core.evaluationCount++;
    
    // Get recent messages
    const sessionHistory = this.messageBuffer.slice(-10).map(m => ({
      role: m.from === 'user' ? 'user' : 'assistant',
      content: m.text
    }));

    // Run detection
    try {
      const detection = await this.core.detector.evaluate(
        sessionHistory,
        this.core.agent.memory
      );

      if (detection.triggered) {
        await this.initiateHearing(detection);
      }
    } catch (err) {
      logger.error('HOOK', 'Evaluation failed', { error: err.message });
    }
  }

  /**
   * Initiate a hearing
   */
  async initiateHearing(detection) {
    logger.info('HOOK', 'Initiating hearing', { offense: detection.offense });
    
    try {
      const verdict = await this.core.hearing.conductHearing(detection);
      
      if (verdict.guilty) {
        this.core.caseCount++;
        
        this.statusManager.update({
          casesFiled: this.core.caseCount,
          lastCase: {
            timestamp: new Date().toISOString(),
            offense: detection.offense,
            verdict: verdict.verdict
          }
        });
        
        await this.core.punishment.execute(verdict);
        await this.core.api.submitCase(verdict);
        
        logger.info('HOOK', 'Case filed', { caseId: verdict.caseId });
        
        // Send notification
        console.log(`\n🏛️  CASE FILED: ${detection.offense}`);
        console.log(`📋 Case ID: ${verdict.caseId}`);
        console.log(`⚖️  Verdict: ${verdict.verdict}`);
        console.log(`🔗 View: https://clawtrial.app/cases/${verdict.caseId}\n`);
      }
    } catch (err) {
      logger.error('HOOK', 'Hearing failed', { error: err.message });
    }
  }

  /**
   * Record a message (called when message is received/sent)
   */
  recordMessage(text, from) {
    if (!this.initialized) return;
    
    this.messageBuffer.push({
      timestamp: Date.now(),
      text,
      from
    });
    
    // Keep only last 100 messages
    if (this.messageBuffer.length > 100) {
      this.messageBuffer.shift();
    }
    
    logger.debug('HOOK', 'Message recorded', { from, length: text.length });
  }

  /**
   * Shutdown
   */
  async shutdown() {
    if (this.evaluationTimer) {
      clearInterval(this.evaluationTimer);
    }
    
    if (this.core) {
      await this.core.shutdown();
    }
    
    this.initialized = false;
    this.statusManager.update({ running: false });
  }
}

// Create singleton
const hook = new ClawTrialHook();

// Auto-initialize if config exists
if (hook.shouldActivate()) {
  hook.initialize().catch(err => {
    logger.error('HOOK', 'Auto-initialization failed', { error: err.message });
  });
}

module.exports = { hook, ClawTrialHook };
